/** @file Viewport.h
 *  @brief Declare the Viewport class. 
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef VIEWPORT_H_
#define VIEWPORT_H_

#include <gtkmm/table.h>
#include <gtkmm/scrollbar.h>
#include <gtkmm/adjustment.h>

#include <libtrack/document/Document.h>

#include "View.h"

/** An interactive view of the track, with scrollbars and an OpenGL box.
 * This class just handles scrolling the view. For the view itself, see View.
 */
class Viewport : public Gtk::Table
{
public:
    /** Create without sharing OpenGL contexts.
     * Meshes and textures previously loaded into an exisiting Viewport will not
     * be avaliable in the new one.
     */
    Viewport();
    
    /** Create sharing OpenGL contexts with an exisitng Viewport.
     * Meshes and textures already loaded will be usable, and any textures or
     * meshes loaded into either the exisiting Viewport or this new one will be
     * avaliable in both.
     */
    Viewport(Viewport & viewport);
    
    virtual ~Viewport();
    typedef View::ViewAngle ViewAngle;
    
    void set_document(Document::Document & document);
    
    /** Update the Viewport to reflect changes in the document.
     */
    void update();
    
    /// Set which perspective the view displays.
    void set_angle(const ViewAngle angle);
    
    /// magnify the view by a factor of 1.25.
    void zoom_in();
    /// magnify the view by a factor of 0.8, reversing a zoom_in().
    void zoom_out();
    
    /// Set the scale so that the entire track fits in the view.
    void scale_to_fit();
    
    /// Realize the view so it's display lists can be shared.
    void realize_view();
    
    /** Signal emmited when the user requests a change to the track by
     * interacting with the viewport.
     * The singal should be wired to a function
     * @code void on_view_command(boost::shared_ptr<Document::DocumentDelta> delta)
     * @endcode
     * where the delta paramter is the DocumentDelta describing the
     * user requested change.
     * @return signal that is emmited when the user performs an action
     * to the scene by interacting with the view.
     */
    sigc::signal<void, boost::shared_ptr<Document::DocumentDelta> > signal_command();
    
    /// Signal emmited when there is a command to show a preview of.
    sigc::signal<void, boost::shared_ptr<Document::DocumentDelta> > signal_preview_command();
    
    /// Signal emmited when the previewed command should be canceled.
    sigc::signal<void> signal_preview_cancel();
protected:
    /// Set scrollbars and events identical in all constructors.
    void create();
    
    /** Set the adjustments for the scrollbars to allow display of the whole
     * track. This function should be called whenever the size of the window or
     * the scale of the view changes.
     */
    void resize_adjustments();
    
    /** Event handler for horizontal scrollbar being moved.
     * Changes the view to show the new position.
     */
    bool on_h_scrollbar_change_value(Gtk::ScrollType scroll, double new_value);
    /** Event handler for vertical scrollbar being moved.
     * Changes the view to show the new position.
     */
    bool on_v_scrollbar_change_value(Gtk::ScrollType scroll, double new_value);
    
    virtual bool on_scroll_event(GdkEventScroll* event);
    
    // scrollbars
    Gtk::Adjustment m_h_scroll_adjustment;
    Gtk::HScrollbar m_h_scroll_bar;
    Gtk::Adjustment m_v_scroll_adjustment;
    Gtk::VScrollbar m_v_scroll_bar;
    /// 3-d track view
    View m_view;
    
    /// The document to display
    Document::Document * document;
};

#endif /*VIEWPORT_H_*/
