/** @file Viewport.cpp
 *  @brief Implement the Viewport class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "Viewport.h"

#include <Debug.h>

Viewport::Viewport()
    :   Gtk::Table(2, 2)
    ,   m_h_scroll_adjustment(49.5, 0.0, 101.0, 1.5, 1.0, 1.0)
    ,   m_h_scroll_bar(m_h_scroll_adjustment)
    ,   m_v_scroll_adjustment(49.5, 0.0, 101.0, 1.5, 1.0, 1.0)
    ,   m_v_scroll_bar(m_v_scroll_adjustment)
    ,   document(0)
{
    create();
}

Viewport::Viewport(Viewport & viewport)
    :   Gtk::Table(2, 2)
    ,   m_h_scroll_adjustment(49.5, 0.0, 101.0, 1.5, 1.0, 1.0)
    ,   m_h_scroll_bar(m_h_scroll_adjustment)
    ,   m_v_scroll_adjustment(49.5, 0.0, 101.0, 1.5, 1.0, 1.0)
    ,   m_v_scroll_bar(m_v_scroll_adjustment)
    ,   m_view(viewport.m_view)
    ,   document(viewport.document)
{
    create();
}

void Viewport::create()
{
    add_events(Gdk::SCROLL_MASK);
    attach(m_h_scroll_bar, 0, 1, 1, 2,
           Gtk::FILL | Gtk::EXPAND, Gtk::SHRINK);
    m_h_scroll_bar.show();
    attach(m_v_scroll_bar, 1, 2, 0, 1,
           Gtk::SHRINK, Gtk::FILL | Gtk::EXPAND);
    m_v_scroll_bar.show();
    attach(m_view, 0, 1, 0, 1,
           Gtk::FILL | Gtk::EXPAND, Gtk::FILL | Gtk::EXPAND);
    m_view.show();
    m_h_scroll_bar.signal_change_value().connect(
                sigc::mem_fun(*this, &Viewport::on_h_scrollbar_change_value));
    m_v_scroll_bar.signal_change_value().connect(
                sigc::mem_fun(*this, &Viewport::on_v_scrollbar_change_value));
    m_v_scroll_bar.set_inverted();
}

Viewport::~Viewport()
{
}

void Viewport::set_document(Document::Document & document_in)
{
    document = &document_in;
    m_view.set_document(document_in);
    resize_adjustments();
}

void Viewport::update()
{
    resize_adjustments();
    // Redraw the view to reflect the changed track.
    m_view.queue_draw();
}

void Viewport::set_angle(const ViewAngle angle)
{
    m_view.set_angle(angle);
    resize_adjustments();
}

void Viewport::zoom_in()
{
    m_view.set_scale(m_view.get_scale() * 1.25);
    resize_adjustments();
}

void Viewport::zoom_out()
{
    m_view.set_scale(m_view.get_scale() * 0.8);
    resize_adjustments();
}

void Viewport::scale_to_fit()
{
    // find the scale to fit in each direction.
    float scale_x = double(m_view.get_width()) /
                    (   m_h_scroll_adjustment.get_upper() -
                        m_h_scroll_adjustment.get_lower());
    float scale_y = double(m_view.get_height()) /
                    (   m_v_scroll_adjustment.get_upper() -
                        m_v_scroll_adjustment.get_lower());
    // now use the largest scale so they both fit.
    if (scale_x < scale_y)
    {
        m_view.set_scale(scale_x);
    }
    else
    {
        m_view.set_scale(scale_y);
    }
    
    resize_adjustments();
    // move scrollbars to the middle.
    double lower = m_h_scroll_adjustment.get_lower();
    double upper = m_h_scroll_adjustment.get_upper();
    double size = m_h_scroll_adjustment.get_page_size();
    double pos = lower + (upper - lower - size) / 2.0;
    m_h_scroll_adjustment.set_value(pos);
    on_h_scrollbar_change_value(Gtk::SCROLL_NONE, m_h_scroll_adjustment.get_value());
    
    lower = m_v_scroll_adjustment.get_lower();
    upper = m_v_scroll_adjustment.get_upper();
    size = m_v_scroll_adjustment.get_page_size();
    pos = lower + (upper - lower - size) / 2.0;
    m_v_scroll_adjustment.set_value(pos);
    on_v_scrollbar_change_value(Gtk::SCROLL_NONE, m_v_scroll_adjustment.get_value());
}

void Viewport::resize_adjustments()
{
    // don't adjust before a document is assigned.
    if (!document)
    {
        DEBUG_MESSAGE("    No track to use to set scrollbar adjustments");
        return;
    }
    const Track::AxisAlignedBoundingBox box(document->get_track().get_path().get_bounding_box());
    const float scale = m_view.get_scale();
    // Find the half the distance across the screen.
    double h_offset = double(m_view.get_width()) / scale / 2.0;
    double v_offset = double(m_view.get_height()) / scale / 2.0;
    // set the page size of the adjustments to the amount shown on the screen
    m_h_scroll_adjustment.set_page_size(h_offset * 2.0);
    m_v_scroll_adjustment.set_page_size(v_offset * 2.0);
    // adjust step size to reflect the scale.
    m_h_scroll_adjustment.set_step_increment(20.0 / scale);
    m_v_scroll_adjustment.set_step_increment(20.0 / scale);
    if (box.valid())
    {
        switch (m_view.get_angle())
        {
            case View::VIEW_TOP:
                m_h_scroll_adjustment.set_lower(box.get_min_x());
                m_h_scroll_adjustment.set_upper(box.get_max_x());
                m_v_scroll_adjustment.set_lower(box.get_min_y());
                m_v_scroll_adjustment.set_upper(box.get_max_y());
                // scroll bar's value is the begining of the page,
                // so it is half the page size lower than the centre.
                m_h_scroll_adjustment.set_value(m_view.get_centre().x() - h_offset);
                m_v_scroll_adjustment.set_value(m_view.get_centre().y() - v_offset);
                break;
            case View::VIEW_SIDE:
                m_h_scroll_adjustment.set_lower(box.get_min_y());
                m_h_scroll_adjustment.set_upper(box.get_max_y());
                m_v_scroll_adjustment.set_lower(box.get_min_z());
                m_v_scroll_adjustment.set_upper(box.get_max_z());
                m_h_scroll_adjustment.set_value(m_view.get_centre().y() - h_offset);
                m_v_scroll_adjustment.set_value(m_view.get_centre().z() - v_offset);
                break;
            case View::VIEW_FRONT:
                m_h_scroll_adjustment.set_lower(box.get_min_x());
                m_h_scroll_adjustment.set_upper(box.get_max_x());
                m_v_scroll_adjustment.set_lower(box.get_min_z());
                m_v_scroll_adjustment.set_upper(box.get_max_z());
                m_h_scroll_adjustment.set_value(m_view.get_centre().x() - h_offset);
                m_v_scroll_adjustment.set_value(m_view.get_centre().z() - v_offset);
                break;
        }
    } else {
        m_h_scroll_adjustment.set_lower(-30.0);
        m_h_scroll_adjustment.set_upper(30.0);
        m_v_scroll_adjustment.set_lower(-30.0);
        m_v_scroll_adjustment.set_upper(30.0);
        DEBUG_MESSAGE("    No valid bounding box for track!");
    }
    // now Update the scrollbar for the display and to emit notify signals.
    m_h_scroll_bar.set_adjustment(m_h_scroll_adjustment);
    m_v_scroll_bar.set_adjustment(m_v_scroll_adjustment);
    on_h_scrollbar_change_value(Gtk::SCROLL_NONE, m_h_scroll_adjustment.get_value());
    on_v_scrollbar_change_value(Gtk::SCROLL_NONE, m_v_scroll_adjustment.get_value());
}

bool Viewport::on_h_scrollbar_change_value(Gtk::ScrollType scroll, double new_value)
{
    DEBUG_MESSAGE("Horizontal scrollbar's value changed.");
    btVector3 centre = m_view.get_centre();
    double h_offset = m_h_scroll_adjustment.get_page_size() / 2.0;
    switch (m_view.get_angle())
    {
        case View::VIEW_TOP:
            // the value of the scrollbar is the position of the left edge,
            // so the position of the centre is half the screen width more.
            centre.setX(new_value + h_offset);
            break;
        case View::VIEW_SIDE:
            centre.setY(new_value + h_offset);
            break;
        case View::VIEW_FRONT:
            centre.setX(new_value + h_offset);
            break;
    }
    m_view.set_centre(centre);
    return true;
}

bool Viewport::on_v_scrollbar_change_value(Gtk::ScrollType scroll, double new_value)
{
    btVector3 centre = m_view.get_centre();
    double v_offset = m_v_scroll_adjustment.get_page_size() / 2.0;
    switch (m_view.get_angle())
    {
        case View::VIEW_TOP:
            centre.setY(new_value + v_offset);
            break;
        case View::VIEW_SIDE:
            centre.setZ(new_value + v_offset);
            break;
        case View::VIEW_FRONT:
            centre.setZ(new_value + v_offset);
            break;
    }
    m_view.set_centre(centre);
    return true;
}

bool Viewport::on_scroll_event(GdkEventScroll * event)
{
    switch (event->direction)
    {
        case GDK_SCROLL_UP:
            m_v_scroll_bar.set_value(m_v_scroll_bar.get_value()
                    + m_v_scroll_adjustment.get_step_increment());
            on_v_scrollbar_change_value(Gtk::SCROLL_NONE,
                                        m_v_scroll_bar.get_value());
            break;
        case GDK_SCROLL_DOWN:
            m_v_scroll_bar.set_value(m_v_scroll_bar.get_value()
                    - m_v_scroll_adjustment.get_step_increment());
            on_v_scrollbar_change_value(Gtk::SCROLL_NONE,
                                        m_v_scroll_bar.get_value());
            break;
        case GDK_SCROLL_LEFT:
            m_h_scroll_bar.set_value(m_h_scroll_bar.get_value()
                    - m_h_scroll_adjustment.get_step_increment());
            on_h_scrollbar_change_value(Gtk::SCROLL_NONE,
                                        m_h_scroll_bar.get_value());
            break;
        case GDK_SCROLL_RIGHT:
            m_h_scroll_bar.set_value(m_h_scroll_bar.get_value()
                    + m_h_scroll_adjustment.get_step_increment());
            on_h_scrollbar_change_value(Gtk::SCROLL_NONE,
                                        m_h_scroll_bar.get_value());
            break;
    }
    return true;
}

void Viewport::realize_view()
{
    m_view.realize();
}

sigc::signal<void, boost::shared_ptr<Document::DocumentDelta> > Viewport::signal_command()
{
    return m_view.signal_command();
}

sigc::signal<void, boost::shared_ptr<Document::DocumentDelta> > Viewport::signal_preview_command()
{
    return m_view.signal_preview_command();
}

sigc::signal<void> Viewport::signal_preview_cancel()
{
    return m_view.signal_preview_cancel();
}
