/** @file View.h
 *  @brief Declare the View class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef VIEW_H_
#define VIEW_H_

#include <boost/shared_ptr.hpp>

#include <gtkmm/uimanager.h>
#include <gtkmm/menu.h>

#include <gtkglmm.h>

#include <libtrack/document/Document.h>
#include <libtrack/edit_base/Selectable.h>
#include <libtrack/edit_base/Dragable.h>

/** A widget that shows a view of the track.
 */
class View : public Gtk::GL::DrawingArea
{
public:
    /** Create using a new OpenGL context.
     *  Meshes and textures loaded in this View will not be useable by an
     *  existing View.
     */
    View();
    
    /** Create sharing OpenGL the context of an existing View.
     *  Meshes and textures loaded into either this View or the specified View
     * will be useable by both.
     * 
     * @param view The View to share contexts with. The document used by view
     * will be set to the same in the new View.
     */
    View(View & view);
    
    virtual ~View();
    
    /// The set of perspectives to draw from
    enum ViewAngle {
        /// Draw the map in the xy plane.
        VIEW_TOP,
        
        /// Draw the map in the yz plane.
        VIEW_SIDE,
        
        /// Draw the map in the xz plane.
        VIEW_FRONT};
    
    /** Set the document that the area is to represent. The document must valid
     * as long as the widget can be drawn. It may be switched to another one at
     * any time, however.
     * 
     * This should be called before any draw events occur.
     * @param document The document the view should draw and try to change if
     * manipulated.
     */
    void set_document(Document::Document & document);
    
    /** Set the perspective the view has on the stage.
     * @param angle The ViewAngle to use for the perspective.
     */
    void set_angle(const ViewAngle angle);
    
    /// Get the perspective of the view. @see set_angle()
    ViewAngle get_angle() const;
    
    /** Set the scale to display the track at. A scale of 1 means 1 pixel is 
     * one spatial unit.
     */
    void set_scale(const float scale);
    
    /// get the scale. @see set_scale()
    float get_scale() const;
    
    /** Set the centre of the view.
     * @param centre The position to have in the middle of the screen.
     */
    void set_centre(btVector3 centre);
    
    /// Get the centre of the view. @see set_centre()
    btVector3 get_centre() const;
    
    // Make realize public so that the View can have an OpenGL context created
    // for another View to share with.
    void realize()
    {
        Gtk::GL::DrawingArea::realize();
    }
    
    /** Signal emmited when the user requests a change to the track by
     * interacting with the view.
     * The singal should be wired to a function
     * @code void on_view_command(boost::shared_ptr<Document::DocumentDelta> delta)
     * @endcode
     * where the delta paramter is the DocumentDelta describing the
     * user requested change.
     * @return signal that is emmited when the user performs an action
     * to the scene by interacting with the view.
     */
    sigc::signal<void, boost::shared_ptr<Document::DocumentDelta> > signal_command();
    
    /** Signal emmited when during user interaction with the view.
     * The singal should be wired to a function
     * @code void on_view_preview_command(boost::shared_ptr<Document::DocumentDelta> delta)
     * @endcode
     * where the delta paramter is the DocumentDelta describing the
     * the change to preview.
     * @return signal that is emmited when we want to preview the effects of
     * a command.
     */
    sigc::signal<void, boost::shared_ptr<Document::DocumentDelta> > signal_preview_command();
    
    /** Get signal emmited when the previewed command should be canceled.
     * This allows the document to be reset to how it was if the user did not
     * want the command that was previewed.
     * The signal should be connected to a function of the form:
     * @code void on_view_preview_cancel()
     * @endcode
     * @return signal that is emmited when the previewed command needs to be
     * rejected.
     */
    sigc::signal<void> signal_preview_cancel();
protected:
    // signal handlers:
    virtual void on_realize();
    virtual bool on_configure_event(GdkEventConfigure* event);
    virtual bool on_expose_event(GdkEventExpose* event);
    virtual bool on_scroll_event(GdkEventScroll* event);
    // Handle other mouse events
    virtual bool on_button_press_event (GdkEventButton* event);
    virtual bool on_button_release_event (GdkEventButton* event);
    virtual bool on_motion_notify_event (GdkEventMotion* event);
    // actions
    void on_delete_vertex();
    void on_delete_edge();
    void on_delete_attachment();
    void on_flip_edge();
    void on_insert_vertex();
    void on_set_start_edge();
    void on_add_booster();
    
    /** User interface preperation common to any constructor.
     */
    void setup_ui();
    
    /** Set the OpenGL modelview matrix so we can draw the scene with the
     * correct orientation, scale, and centre.
     */
    void recentre_view();
    
    /// Draw the path of the track.
    void draw_path(const Track::Path & path);
    
    /// The document to show and edit
    Document::Document * document;
    
    /// The scale of the drawing. Scale is in pixels per distance unit.
    float scale;
    
    /// The length of the handles used to edit rotations.
    btScalar handle_length;
    
    /// The coordinates at the bottom left corner of the display.
    btVector3 centre;
    
    /// Which perspective of the scene to use.
    ViewAngle view_angle;
    
    /// true if recentre_view() needs to be called during the next draw.
    bool needs_recentre;
    
    /// Find something under the mouse pointer.
    const Track::EditAssist::Selectable * get_selectable_under_mouse();
    
    /// Convert window position to scene position
    btVector3 mouse_to_scene(btScalar x, btScalar y) const;
    
    /** New position of object being dragged, or where the line under
     * the mouse hits the back plane if none.
     */
    btVector3 new_position;
    
    /** move a position vector so its distance into the screen is the
     * view's centre + a given value.
     * @param position vector to move
     * @param depth distance from screen centre: positive is into the
     * screen, negative is out of it.
     */
    void set_relative_depth(btVector3 & position, btScalar depth);
    
    const Track::EditAssist::Dragable * drag_object;
    
    const Track::EditAssist::Selectable * popup_object;
    
    /** Signal emmited when a command is created through the user's
     * interaction.
     */
    sigc::signal<void, boost::shared_ptr<Document::DocumentDelta> > m_signal_command;
    
    /** Signal emmited when a command should be previewed.
     */
    sigc::signal<void, boost::shared_ptr<Document::DocumentDelta> > m_signal_preview_command;
    
    /** Signal emmited when a previewed command should be canceled.
     */
    sigc::signal<void> m_signal_preview_cancel;
    
    Glib::RefPtr<Gtk::ActionGroup> m_action_group; 
    Glib::RefPtr<Gtk::UIManager> m_ui_manager;
    Gtk::UIManager::ui_merge_id m_menu_id;
    /// popup menu that contains the segments list.
    Gtk::Menu * m_popup_menu;
    /// popup menu shown when an empty space is right clicked.
    Gtk::Menu * m_empty_menu;
    /// popup menu shown when an Track::TrackAttachment is right clicked.
    Gtk::Menu * m_attachment_menu;
    
    void pick_segment(std::size_t index);
};

#endif /*VIEW_H_*/
