/** @file View.cpp
 *  @brief Implement the View class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "View.h"
#include <iostream>
#include <Debug.h>

#include <GL/gl.h>
#include <GL/glu.h>

#include <gtkmm/main.h>
#include <gtkmm/radioaction.h>
#include <gtkmm/stock.h>

#include <libtrack/document/ChangeVertexSegmentDelta.h>
#include <libtrack/document/ChangeEdgeSegmentDelta.h>
#include <libtrack/document/InsertVertexDelta.h>
#include <libtrack/document/InsertEdgeDelta.h>
#include <libtrack/document/FlipEdgeDelta.h>
#include <libtrack/document/SetStartEdgeDelta.h>
#include <libtrack/document/InsertTrackAttachmentDelta.h>
#include <libtrack/edit_base/TrackAttachmentHandle.h>
#include <libtrack/TrackBooster.h>
#include <libtrack/StartingPosition.h>

const btScalar back_depth = 5000.0;
const btScalar front_depth = -5000.0;

View::View()
    : Gtk::GL::DrawingArea(Gdk::GL::Config::create(Gdk::GL::MODE_RGB |
                                                   Gdk::GL::MODE_DEPTH |
                                                   Gdk::GL::MODE_DOUBLE))
    ,   document(0)
    ,   scale(1.0)
    ,   handle_length(64.0 / scale)
    ,   centre(0.0, 0.0, 0.0)
    ,   view_angle(VIEW_TOP)
    ,   new_position(centre)
    ,   drag_object(0)
    ,   m_ui_manager(Gtk::UIManager::create())
    ,   m_menu_id(0)
    ,   m_popup_menu(0)
{
    setup_ui();
}

View::View(View & view)
    :   Gtk::GL::DrawingArea(Gdk::GL::Config::create(Gdk::GL::MODE_RGB |
                                                     Gdk::GL::MODE_DEPTH |
                                                     Gdk::GL::MODE_DOUBLE),
                             view.get_gl_context(),
                             false, Gdk::GL::RGBA_TYPE)
    ,   document(view.document)
    ,   scale(view.scale)
    ,   handle_length(64.0 / scale)
    ,   centre(view.centre)
    ,   view_angle(VIEW_TOP)
    ,   new_position(centre)
    ,   drag_object(0)
    ,   m_ui_manager(Gtk::UIManager::create())
    ,   m_menu_id(0)
    ,   m_popup_menu(0)
{
    setup_ui();
}

View::~View()
{
}

void View::setup_ui()
{
    add_events(  Gdk::SCROLL_MASK
           | Gdk::BUTTON_PRESS_MASK
           | Gdk::BUTTON_RELEASE_MASK
           | Gdk::POINTER_MOTION_MASK
          );
    
}

void View::set_document(Document::Document & document_in)
{
    // clear up from last document if there was one.
    if (document)
    {
        m_ui_manager->remove_ui(m_menu_id);
        m_ui_manager->remove_action_group(m_action_group);
        m_menu_id = 0;
        m_popup_menu = 0;
        m_empty_menu = 0;
    }
    
    // new document
    document = &document_in;
    
    // actions which always exist
    m_action_group = Gtk::ActionGroup::create();
    m_action_group->add(Gtk::Action::create("popup-menu", "Vertex popup"));
    m_action_group->add(Gtk::Action::create("segments-menu", Gtk::Stock::CONVERT, "Segments"));
    m_action_group->add(Gtk::Action::create("delete-vertex", Gtk::Stock::DELETE, "Delete Vertex"),
        sigc::mem_fun(*this, &View::on_delete_vertex));
    m_action_group->add(Gtk::Action::create("delete-edge", Gtk::Stock::DELETE, "Delete Edge"),
        sigc::mem_fun(*this, &View::on_delete_edge));
    m_action_group->add(Gtk::Action::create("flip-edge", Gtk::Stock::ORIENTATION_REVERSE_PORTRAIT, "Reverse Edge direction"),
        sigc::mem_fun(*this, &View::on_flip_edge));
    m_action_group->add(Gtk::Action::create("add-booster", Gtk::Stock::ADD, "Add booster"),
        sigc::mem_fun(*this, &View::on_add_booster));
    m_action_group->add(Gtk::Action::create("set-start-edge", Gtk::Stock::GOTO_FIRST, "Make starting positions"),
        sigc::mem_fun(*this, &View::on_set_start_edge));
    // use the document's theme's segments for a menu item and an action.
    Glib::ustring ui_buffer =
        "<ui>"
        "  <popup name='popup-menu' action='popup-menu'>"
        "     <menuitem name='delete-vertex' action='delete-vertex'/>"
        "     <menuitem name='delete-edge' action='delete-edge'/>"
        "     <menuitem name='flip-edge' action='flip-edge'/>"
        "     <menuitem name='set-start-edge' action='set-start-edge'/>"
        "     <menuitem name='add-booster' action='add-booster'/>"
        "     <menu name='segments-menu' action='segments-menu'>";
    
    Gtk::RadioAction::Group radio_group;
    const Track::Theme & theme(document->get_track().get_theme());
    std::size_t number_of_segments = theme.get_number_of_segments();
    for (std::size_t segment_index = 0;
         segment_index < number_of_segments; segment_index++)
    {
        const Track::Segment & segment(theme.get_segment(segment_index));
        ui_buffer += "<menuitem name=\"";
        ui_buffer += Glib::ustring(segment.get_name());
        ui_buffer += "\" action=\"segment-";
        ui_buffer += Glib::ustring(segment.get_name());
        ui_buffer += "\"/>";
        // make the action that uses this.
        Glib::ustring action_name = "segment-";
        action_name += Glib::ustring(segment.get_name());
        m_action_group->add(Gtk::RadioAction::create(radio_group, action_name, segment.get_name()),
                            sigc::bind(sigc::mem_fun(*this, &View::pick_segment), segment_index));
    }
    ui_buffer +=
        "     </menu>"
        "  </popup>"
        "  <popup name='empty-menu' action='empty-menu'>"
        "     <menuitem name='insert-vertex' action='insert-vertex'/>"
        "  </popup>"
        "  <popup name='attachment-menu' action='attachment-menu'>"
        "     <menuitem name='delete-attachment' action='delete-attachment'/>"
        "  </popup>"
        "</ui>";
    m_action_group->add(Gtk::Action::create("insert-vertex", Gtk::Stock::ADD, "Insert Vertex"),
        sigc::mem_fun(*this, &View::on_insert_vertex));
    m_action_group->add(Gtk::Action::create("delete-attachment", Gtk::Stock::DELETE, "Delete Attachment"),
        sigc::mem_fun(*this, &View::on_delete_attachment));
    
    m_ui_manager->insert_action_group(m_action_group);
    m_menu_id = m_ui_manager->add_ui_from_string(ui_buffer);
    m_popup_menu = (Gtk::Menu *)m_ui_manager->get_widget("/popup-menu");
    assert(m_popup_menu);
    m_empty_menu = (Gtk::Menu *)m_ui_manager->get_widget("/empty-menu");
    assert(m_empty_menu);
    m_attachment_menu = (Gtk::Menu *)m_ui_manager->get_widget("/attachment-menu");
    assert(m_attachment_menu);
    set_scale(1.0);
}

void View::set_angle(const ViewAngle angle_in)
{
    view_angle = angle_in;
    needs_recentre = true;
    queue_draw();
    
}

View::ViewAngle View::get_angle() const
{
    return view_angle;
}

void View::set_scale(const float scale_in)
{
    scale = scale_in;
    needs_recentre = true;
    handle_length = 64.0 / scale;
    /// @todo Multiple-view friendly handle lengths.
    queue_draw();
}

float View::get_scale() const
{
    return scale;
}

void View::set_centre(const btVector3 centre_in)
{
    // ignore request if dragging, since it will be confusing to have the
    // scene move while placing something on it.
    if (!drag_object)
    {
        centre = centre_in;
        needs_recentre = true;
        queue_draw();
        DEBUG_MESSAGE("Queued draw.");
    }
}

btVector3 View::get_centre() const
{
    return centre;
}

void View::on_realize()
{
    Gtk::GL::DrawingArea::on_realize();
    Glib::RefPtr<Gdk::GL::Drawable> gldrawable = get_gl_drawable();
    if (!gldrawable->gl_begin(get_gl_context()))
    {
        return;
    }
    
    gldrawable->gl_end();
}

bool View::on_configure_event(GdkEventConfigure* event)
{
    Glib::RefPtr<Gdk::GL::Drawable> gldrawable = get_gl_drawable();
    if (!gldrawable->gl_begin(get_gl_context()))
    {
        return false;
    }
    DEBUG_MESSAGE("Reconfiguring view.");
    glViewport(0, 0, get_width(), get_height());
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0.5, get_width() + 0.5,
            0.5, get_height() + 0.5,
            front_depth, back_depth);
    glMatrixMode(GL_MODELVIEW);
    recentre_view();
    gldrawable->gl_end();
    return true;    
}

bool View::on_scroll_event(GdkEventScroll* event)
{
    if (drag_object)
    {
        // scrolling interferes with dragging objects
        /// @todo fix scrolling while dragging.
        return true;
    }
    return false;
}

void View::recentre_view()
{
    glLoadIdentity();
    // translate half screen width, so we can deal with the centre rather
    // than the corner.
    glTranslatef(get_width() / 2, get_height() / 2, 0);
    glScalef(scale, scale, scale);
    // rotations
    switch (view_angle)
    {
        case VIEW_TOP:
            // xy plane is already in view.
            break;
        case VIEW_SIDE:
            // rotate 90 degrees along the y axis to get xy plane in view.
            glRotatef(90.0, -1.0, 0.0, 0.0);
            // Then rotate about the user to get z up the screen.
            glRotatef(-90.0, 0.0, 0.0, 1.0);
            break;
        case VIEW_FRONT:
            // rotate 90 degrees along the x axis to get yz plane in view.
            glRotatef(90.0, -1.0, 0.0, 0.0);
            break;
    }
    glTranslatef(-centre.x(), -centre.y(), -centre.z());
    // The depth into the screen will be snapped while dragging an object,
    // so leave it alone while dragging.
    // Otherwise, use a fixed depth.
    if (!drag_object)
    {
        set_relative_depth(new_position, back_depth);
    }
    needs_recentre = false;
}

bool View::on_expose_event(GdkEventExpose* event)
{
    assert(document);
    Glib::RefPtr<Gdk::GL::Drawable> gldrawable = get_gl_drawable();
    if (!gldrawable->gl_begin(get_gl_context()))
    {
        DEBUG_MESSAGE("    No gl_drawable.");
        return false;
    }
    
    if (needs_recentre)
    {
        recentre_view();
        DEBUG_MESSAGE("    Recentered.");
    }
    
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    
    // draw grid lines
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glPushMatrix();
    // rotate so the xy plane is on the screen.
    switch (view_angle)
    {
        case VIEW_TOP:
            // nothing needed
            break;
        case VIEW_SIDE:
            glRotatef(90.0, 0.0, 1.0, 0.0);
            break;
        case VIEW_FRONT:
            glRotatef(90.0, 1.0, 0.0, 0.0);
            break;
    }
    // draw grid lines in xy plane.
    glBegin(GL_LINES);
        for (int i = -256; i <= 256; i += 8)
        {
            // vary intensity to make minor grid lines less distinct.
            unsigned int l = (i + 256);
            unsigned char s;
            if (l % 16) s = 29;
            else if (l % 32) s = 40;
            else if (l % 64) s = 53;
            else if (l % 128) s = 71;
            else if (l % 256) s = 95;
            else s = 127;
            glColor4ub(255, 255, 255, s);
            // draw lines
            glVertex2d(i, -256);
            glVertex2d(i,  256);
            glVertex2d(-256, i);
            glVertex2d( 256, i);
        }
    glEnd();
    glPopMatrix();
    glDisable(GL_BLEND);
    
    // get the track to draw.
    const Track::Track & track = document->get_track();
    
    draw_path(track.get_path());
    
    // Swap buffers.
    if (gldrawable->is_double_buffered())
    {
        gldrawable->swap_buffers();
    }
    else
    {
        glFlush();
    }
    
    gldrawable->gl_end();
    return true;
}

inline void glVertex(const btVector3 & position)
{
    glVertex3f(position.x(), position.y(), position.z());
}

void View::draw_path(const Track::Path & path)
{
    /** @todo Find bounds of visible area and set up an occlusion tester.
     */
    const_cast<Track::Path &>(document->get_track().get_path()).set_handle_lengths(handle_length);
    glPointSize(5.0);
    
    glColor3ub(191, 191, 191);
    path.draw();
    
    // draw lines between the vertices.
    typedef boost::graph_traits<Track::Path::Graph>::edge_iterator EdgeIterator;
    std::pair<EdgeIterator, EdgeIterator> edge_range;
    for (edge_range = boost::edges(path.graph);
         edge_range.first != edge_range.second;
         edge_range.first++)
    {
        Track::PathVertex vertex_1 = path.graph[source(*(edge_range.first), path.graph)];
        Track::PathVertex vertex_2 = path.graph[target(*(edge_range.first), path.graph)];
        Track::PathEdge edge = path.graph[*(edge_range.first)];
        glColor3ub(255, 255, 255);
        glBegin(GL_LINE_STRIP);
            for (float t = 0.0; t <= 1.0; t += 0.015625)
            {
                glVertex(edge.get_transform(t).getOrigin());
            }
        glEnd();
        
        edge.draw_control_points();
    }
    
    // draw markers at the vertices.
    glClear(GL_DEPTH_BUFFER_BIT);
        /** @todo use marker size to draw a square around the point regardless
         * of hardware maximum point size. glPointSize is not guranteed to
         * support sizes other than 1, which is too small.
         */
        //const float marker_size = 3.0;
        // float offset = scale * marker_size;
        
    typedef boost::graph_traits<Track::Path::Graph>::vertex_iterator VertexIterator;
    std::pair<VertexIterator, VertexIterator> vertex_range;
    for (vertex_range = boost::vertices(path.graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        const Track::PathVertex & vertex = path.graph[*(vertex_range.first)];
        vertex.draw_control_points();
        
        // Draw the move handle at the vertex centre.
        glColor3ub(255, 255, 0);
        float dim = 8.0 / scale;
        static Track::Texture move_handle("data/generic/move_handle.png");
        move_handle.bind();
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
        glEnable(GL_TEXTURE_2D);
        glEnable(GL_BLEND);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
        glPushMatrix();
            const btVector3 & pos = vertex.get_position();
            glTranslatef(0.5 / scale + pos.x(),
                         0.5 / scale + pos.y(),
                         0.5 / scale + pos.z());
            glBegin(GL_QUADS);
                glTexCoord2f(0.0, 0.0); glVertex2f(-dim, -dim);
                glTexCoord2f(1.0, 0.0); glVertex2f(dim, -dim);
                glTexCoord2f(1.0, 1.0); glVertex2f(dim, dim);
                glTexCoord2f(0.0, 1.0); glVertex2f(-dim, dim);
            glEnd();
            glBegin(GL_QUADS);
                glTexCoord2f(0.0, 0.0); glVertex3f(0, -dim, -dim);
                glTexCoord2f(1.0, 0.0); glVertex3f(0, dim, -dim);
                glTexCoord2f(1.0, 1.0); glVertex3f(0, dim, dim);
                glTexCoord2f(0.0, 1.0); glVertex3f(0, -dim, dim);
            glEnd();
            glBegin(GL_QUADS);
                glTexCoord2f(0.0, 0.0); glVertex3f(-dim, 0, -dim);
                glTexCoord2f(1.0, 0.0); glVertex3f(dim, 0, -dim);
                glTexCoord2f(1.0, 1.0); glVertex3f(dim, 0, dim);
                glTexCoord2f(0.0, 1.0); glVertex3f(-dim, 0, dim);
            glEnd();
        glPopMatrix();
        glDisable(GL_TEXTURE_2D);
        glDisable(GL_BLEND);
    }
    glColor3ub(255, 255, 255);
    
    #ifndef NDEBUG
        // mark new position location
        glBegin(GL_POINTS);
            glVertex(new_position);
        glEnd();
    #endif
}

bool View::on_button_press_event(GdkEventButton * event)
{
    if (drag_object)
    {
        // Ignore presses from all buttons until the user lets go of the
        // object they are already dragging.
        return false;
    }
    // Is there something under the mouse pointer we can drag?
    if (event->button == 1)
    {
        drag_object = dynamic_cast<const Track::EditAssist::Dragable *>(get_selectable_under_mouse());
        return true;
    }
    else if (event->button == 3)
    {
        const Track::EditAssist::Selectable * found_object = get_selectable_under_mouse();
        if (m_popup_menu && found_object)
        {
            // set the menu item reflecting the current segment.
            // we don't want to attribute the following change to any object
            popup_object = 0;
            // Does the select object have a segment?
            bool has_segment = false;
            std::size_t segment_index;
            /** @todo I don't like this big chain of dynamic_casts.
             * It should use polymorphism, but we don't want to mix editor UI
             * stuff with the game components so they should be separated
             * better first.
             */
            const Track::PathVertex * vertex = dynamic_cast<const Track::PathVertex *>(found_object);
            if (vertex)
            {
                segment_index = vertex->get_segment_index();
                has_segment = true;
                m_action_group->get_action("delete-vertex")->set_visible();
                m_action_group->get_action("delete-edge")->set_visible(false);
                m_action_group->get_action("flip-edge")->set_visible(false);
                m_action_group->get_action("set-start-edge")->set_visible(false);
                m_action_group->get_action("add-booster")->set_visible(false);
                // allow all segments
                for (unsigned int i = 0; i < document->get_track().get_theme().get_number_of_segments(); i++)
                {
                    const Track::Segment & segment = document->get_track().get_theme().get_segment(i);
                    Glib::ustring widget_path("/popup-menu/segments-menu/");
                    widget_path += Glib::ustring(segment.get_name());
                    Gtk::RadioMenuItem * menu_item = (Gtk::RadioMenuItem *)m_ui_manager->get_widget(widget_path);
                    assert(menu_item);
                    menu_item->set_sensitive();
                }
            }
            else
            {
                const Track::PathEdge * edge = dynamic_cast<const Track::PathEdge *>(found_object);
                if (edge)
                {
                    segment_index = edge->segment_index;
                    has_segment = true;
                    m_action_group->get_action("delete-edge")->set_visible();
                    m_action_group->get_action("flip-edge")->set_visible();
                    m_action_group->get_action("delete-vertex")->set_visible(false);
                    // enable set-start-edge iff edge is not already the start edge.
                    m_action_group->get_action("set-start-edge")->set_visible(true);
                    m_action_group->get_action("set-start-edge")->set_sensitive(document->get_track().get_path().get_starting_edge() != edge->get_name());
                    m_action_group->get_action("add-booster")->set_visible();
                    // filter segments by edge suitablility.
                    for (unsigned int i = 0; i < document->get_track().get_theme().get_number_of_segments(); i++)
                    {
                        const Track::Segment & segment = document->get_track().get_theme().get_segment(i);
                        Glib::ustring widget_path("/popup-menu/segments-menu/");
                        widget_path += Glib::ustring(segment.get_name());
                        Gtk::RadioMenuItem * menu_item = (Gtk::RadioMenuItem *)m_ui_manager->get_widget(widget_path);
                        assert(menu_item);
                        menu_item->set_sensitive(segment.edges_allowed());
                    }
                }
                else
                {
                    const Track::EditAssist::TrackAttachmentHandle * attachment =
                        dynamic_cast<const Track::EditAssist::TrackAttachmentHandle *>(found_object);
                    if (attachment)
                    {
                        popup_object = found_object;
                        m_attachment_menu->popup(event->button, event->time);
                        return true;
                    }
                }
            }
            if (has_segment)
            {
                // Find the segment's widget.
                const Track::Segment & segment = document->get_track().get_theme().get_segment(segment_index);
                Glib::ustring widget_path("/popup-menu/segments-menu/");
                widget_path += Glib::ustring(segment.get_name());
                Gtk::RadioMenuItem * menu_item = (Gtk::RadioMenuItem *)m_ui_manager->get_widget(widget_path);
                assert(menu_item);
                menu_item->set_active();
                // show the menu
                popup_object = found_object;
                m_popup_menu->popup(event->button, event->time);
                return true;
            }
        }
        // No object, show empty space menu.
        m_empty_menu->popup(event->button, event->time);
        return true;
    }
    return false;
}

bool View::on_button_release_event(GdkEventButton * event)
{
    if (event->button == 1 && drag_object)
    {
        // release the object
        m_signal_preview_cancel.emit();
        boost::shared_ptr<Document::DocumentDelta> delta(
            drag_object->make_delta(new_position)
        );
        // anonunce the delta so it can be applied.
        m_signal_command.emit(delta);
        // Go back to coordinates from back plane.
        set_relative_depth(new_position, back_depth);
        // reset drag_object to 0 to avoid crash if drag object is later
        // removed with a mouse button held down.
        drag_object = 0;
    }
    return false;
}

void View::set_relative_depth(btVector3 & position, btScalar depth)
{
    switch (view_angle)
    {
        case VIEW_TOP:
            new_position.setZ(depth + centre.getZ());
            break;
        case VIEW_FRONT:
            new_position.setY(depth + centre.getY());
            break;
        case VIEW_SIDE:
            new_position.setX(depth + centre.getX());
            break;
    }
}

bool View::on_motion_notify_event(GdkEventMotion * event)
{
    new_position = mouse_to_scene(event->x, event->y);
    if (drag_object)
    {
        btVector3 normal
        (
            view_angle == VIEW_SIDE ? 1 : 0,
            view_angle == VIEW_FRONT ? 1 : 0,
            view_angle == VIEW_TOP ? 1 : 0
        );
        drag_object->snap(new_position, normal);
        if (!Gtk::Main::events_pending())
        {
            m_signal_preview_cancel.emit();
            boost::shared_ptr<Document::DocumentDelta> delta
            (
                drag_object->make_delta(new_position)
            );
            // anonunce the delta so it can be previewed.
            m_signal_preview_command.emit(delta);
        }
    }
    return false;
}

btVector3 View::mouse_to_scene(btScalar x, btScalar y) const
{
    btScalar screen_x = (x - get_width() / 2) / scale;
    btScalar screen_y = (btScalar(get_height() / 2) - y) / scale;
    // coordinates are from the corner of the window, but we keep the centre.
    switch (view_angle)
    {
        case VIEW_TOP:
            return btVector3(screen_x + centre.x(), screen_y + centre.y(), new_position.z());
        case VIEW_FRONT:
            return btVector3(screen_x + centre.x(), new_position.y(), screen_y + centre.z());
        case VIEW_SIDE:
            return btVector3(new_position.x(), screen_x + centre.y(), screen_y + centre.z());
        default:
            DEBUG_MESSAGE("Warning: unhandled view");
            return new_position;
    }
}

const Track::EditAssist::Selectable * View::get_selectable_under_mouse()
{
    btVector3 start = new_position;
    set_relative_depth(start, front_depth);
    btVector3 stop = new_position;
    set_relative_depth(stop, back_depth);
    // find the radius of a few pixels close enough to the line
    btScalar radius = 8.0 / scale;
    const Track::Path & path = document->get_track().get_path();
    
    // edge control points
    typedef boost::graph_traits<Track::Path::Graph>::edge_iterator EdgeIterator;
    std::pair<EdgeIterator, EdgeIterator> edge_range;
    for (edge_range = boost::edges(path.graph);
         edge_range.first != edge_range.second;
         edge_range.first++)
    {
        const Track::PathEdge & edge = path.graph[*(edge_range.first)];
        const Track::EditAssist::ControlPoint * control_point
            (edge.get_control_point_here(start, stop, radius));
        if (control_point)
        {
            // Use this control point
            DEBUG_MESSAGE("Found edge control point on edge " << edge.get_name());
            return control_point;
        }
    }
    
    // vertex centres
    typedef boost::graph_traits<Track::Path::Graph>::vertex_iterator VertexIterator;
    std::pair<VertexIterator, VertexIterator> vertex_range;
    for (vertex_range = boost::vertices(path.graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        const Track::PathVertex & vertex = path.graph[*(vertex_range.first)];
        if (vertex.is_here(start, stop, radius))
        {
            // Use this vertex
            new_position = vertex.get_position();
            DEBUG_MESSAGE("Found vertex " << vertex.get_name());
            return &vertex;
        }
    }
    
    //vertex control points
    for (vertex_range = boost::vertices(path.graph);
         vertex_range.first != vertex_range.second;
         vertex_range.first++)
    {
        const Track::PathVertex & vertex = path.graph[*(vertex_range.first)];
        const Track::EditAssist::ControlPoint * control_point
            (vertex.get_control_point_here(start, stop, radius));
        if (control_point)
        {
            new_position = control_point->get_position();
            DEBUG_MESSAGE("Found control point @ " << control_point << " on vertex " << vertex.get_name());
            return control_point;
        }
    }
    
    // not a vertex or control point, try an edge.
    for (edge_range = boost::edges(path.graph);
         edge_range.first != edge_range.second;
         edge_range.first++)
    {
        const Track::PathEdge & edge = path.graph[*(edge_range.first)];
        if (edge.is_here(start, stop, radius))
        {
            // Use this edge
            DEBUG_MESSAGE("Found edge " << edge.get_name());
            return &edge;
        }
    }
    // nothing suitable found.
    DEBUG_MESSAGE("Could not find suitable object under mouse pointer.");
    return 0;
}

sigc::signal<void, boost::shared_ptr<Document::DocumentDelta> > View::signal_command()
{
    return m_signal_command;
}

sigc::signal<void, boost::shared_ptr<Document::DocumentDelta> > View::signal_preview_command()
{
    return m_signal_preview_command;
}

sigc::signal<void> View::signal_preview_cancel()
{
    return m_signal_preview_cancel;
}

void View::pick_segment(std::size_t index)
{
    /* This is event is recieved when changing the selected menu to reflect a
     * a different vertex. At this point, popup_object is not pointing at
     * either vertex.
     */
    if (popup_object != 0)
    {
        /* This function is called when something is no longer active, as well
         * as when it is activated. Check which state we are in by finding the
         * menu item.
         */
        const Track::Segment & segment = document->get_track().get_theme().get_segment(index);
        Glib::ustring widget_path("/popup-menu/segments-menu/" + Glib::ustring(segment.get_name()));
        Gtk::RadioMenuItem * menu_item = (Gtk::RadioMenuItem *)m_ui_manager->get_widget(widget_path);
        assert(menu_item);
        if (menu_item->get_active())
        {
            const Track::PathVertex * vertex = dynamic_cast<const Track::PathVertex *>(popup_object);
            if (vertex)
            {
                DEBUG_MESSAGE("Switch vertex to segment " << index);
                boost::shared_ptr<Document::DocumentDelta> delta(
                    new Document::ChangeVertexSegmentDelta(vertex->get_name(), index)
                );
                m_signal_command.emit(delta);
            }
            else
            {
                // Not a vertex, so should be an edge.
                const Track::PathEdge * edge = dynamic_cast<const Track::PathEdge *>(popup_object);
                assert(edge); // Isn't an edge either. What else has segments?
                DEBUG_MESSAGE("Switch edge to segment " << index);
                boost::shared_ptr<Document::DocumentDelta> delta(
                    new Document::ChangeEdgeSegmentDelta(edge->get_name(), index)
                );
                m_signal_command.emit(delta);
            }
        }
    }
}

void View::on_delete_vertex()
{
    const Track::PathVertex * vertex = dynamic_cast<const Track::PathVertex *>(popup_object);
    assert(vertex);
    boost::shared_ptr<Document::DocumentDelta> delta
    (
        new Document::RemoveVertexDelta(vertex->get_name())
    );
    m_signal_command.emit(delta);
}

void View::on_delete_edge()
{
    const Track::PathEdge * edge = dynamic_cast<const Track::PathEdge *>(popup_object);
    assert(edge);
    boost::shared_ptr<Document::DocumentDelta> delta
    (
        new Document::RemoveEdgeDelta(edge->get_name())
    );
    m_signal_command.emit(delta);
}

void View::on_delete_attachment()
{
    const Track::EditAssist::TrackAttachmentHandle * attachment_handle =
        dynamic_cast<const Track::EditAssist::TrackAttachmentHandle *>(popup_object);
    assert(attachment_handle);
    boost::shared_ptr<Document::DocumentDelta> delta
    (
        attachment_handle->make_remove_delta()
    );
    m_signal_command.emit(delta);
}

void View::on_flip_edge()
{
    const Track::PathEdge * edge = dynamic_cast<const Track::PathEdge *>(popup_object);
    assert(edge);
    boost::shared_ptr<Document::DocumentDelta> delta
    (
        new Document::FlipEdgeDelta(edge->get_name())
    );
    m_signal_command.emit(delta);
}

void View::on_insert_vertex()
{
    Track::PathVertex vertex(&(document->get_track()));
     switch (view_angle)
    {
        case VIEW_TOP:
            new_position.setZ(0);
            break;
        case VIEW_FRONT:
            new_position.setY(0);
            break;
        case VIEW_SIDE:
            new_position.setX(0);
            break;
        default:
            DEBUG_MESSAGE("Warning: unhandled view");
    }
    vertex.set_position(new_position);
    const Track::Theme & theme = document->get_track().get_theme();
    vertex.set_segment(theme.get_default_segment_index(),
                       &theme.get_segment(theme.get_default_segment_index()));
    vertex.set_angle(btQuaternion::getIdentity());
    vertex.set_handle_lengths(handle_length);
    boost::shared_ptr<Document::DocumentDelta> delta
    (
        new Document::InsertVertexDelta(vertex)
    );
    m_signal_command.emit(delta);
}

void View::on_set_start_edge()
{
    const Track::PathEdge * edge = dynamic_cast<const Track::PathEdge *>(popup_object);
    assert(edge);
    boost::shared_ptr<Document::DocumentDelta> delta
    (
        new Document::SetStartPositionsDelta(edge->get_name())
    );
    m_signal_command.emit(delta);
}

void View::on_add_booster()
{
    const Track::PathEdge * edge = dynamic_cast<const Track::PathEdge *>(popup_object);
    assert(edge);
    
    boost::shared_ptr<Track::TrackAttachment> booster(new Track::TrackBooster);
    booster->set_lateral_position(0.0);
    booster->set_t_position(0.5);
    booster->edge_name = edge->get_name();
    
    boost::shared_ptr<Document::DocumentDelta> delta
    (
        new Document::InsertTrackAttachmentDelta(booster)
    );
    m_signal_command.emit(delta);
}
