/** @file RacerEditorApp.cpp
 *  @brief Implement the RacerEditorApp class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "RacerEditorApp.h"

#include <gtkmm/main.h>
#include <gtkglmm.h>
#include "EditorWindow.h"
#include <libtrack/DataCheck.h>
#include <boost/program_options.hpp>
#include <iostream>
#include <fstream>

RacerEditorApp::RacerEditorApp(int argc, char **argv)
{
    try
    {
        Track::DataCheck();
    }
    catch (Track::DataCheck::FileMissingError e)
    {
        std::cerr << "Cannot find data file: " << e.filename << std::endl;
        std::cerr << "Please run from the directory all the data files accessible." << std::endl;
        return;
    }
    // initialise gtkmm
    Gtk::Main kit(argc, argv);
    // initalise gtkglmm
    Gtk::GL::init(argc, argv);
    
    // program options allowed only on the command line.
    boost::program_options::options_description generic("Generic options");
    generic.add_options()
        ("help,h", "Show this help message")
    ;
    // program options allowed on both the command line and configuration file.
    boost::program_options::options_description config("Configuration");
    config.add_options()
        ("show-guide", boost::program_options::value<bool>(&m_show_guide)->default_value(true), "Show the usage guide above the track view.")
    ;
    // hidden program options allowed only in the command line.
    boost::program_options::options_description hidden("Hidden");
    hidden.add_options()
        ("input-file", boost::program_options::value<std::string>(), "Load the specified file. Relative paths are taken to mean from racer's data directory. For example data/tracks/0")
    ;
    // 'file' on its own means the same as --input-file='file'.
    boost::program_options::positional_options_description p;
    p.add("input-file", -1);
    
    // combine for command line arguments.
    boost::program_options::options_description cmdline_options;
    cmdline_options.add(generic).add(config).add(hidden);
    // combine for visible command line arguments.
    boost::program_options::options_description visible_options;
    visible_options.add(generic).add(config);
    
    // load the variables from the command line and configuration file.
    boost::program_options::variables_map vm;
    boost::program_options::store(boost::program_options::command_line_parser(argc, argv).
          options(cmdline_options).positional(p).run(), vm);
    std::string config_filename = Glib::get_user_config_dir() + "/racer-editor.config";
    DEBUG_MESSAGE("Loading configuration from " << config_filename);
    std::ifstream config_file(config_filename.c_str());
    boost::program_options::store(boost::program_options::parse_config_file(config_file, config), vm);
    boost::program_options::notify(vm);
    
    // Show help?
    if (vm.count("help"))
    {
        std::cout << "Usage: " << argv[0] << " [options] [trackfile]" << std::endl;
        std::cout << visible_options << std::endl;
        return;
    }
    
    // Start editor.
    std::string filename;
    if (vm.count("input-file"))
    {
        filename = vm["input-file"].as<std::string>();
    }
    EditorWindow window(m_show_guide, filename);
    Gtk::Main::run(window);
}

RacerEditorApp::~RacerEditorApp()
{
}
