/** @file NewForm.h
 *  @brief Declare the NewForm class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef NEWFORM_H_
#define NEWFORM_H_

#include <gtkmm/box.h>
#include <gtkmm/button.h>
#include <gtkmm/iconview.h>
#include <gtkmm/liststore.h>
#include <gtkmm/label.h>
#include <gtkmm/scrolledwindow.h>
#include <giomm/file.h>

/** A form for picking the theme of a new track.
 */
class NewForm : public Gtk::VBox
{
public:
    /** Create a new form.
     * @param window the window the forms sits on. When NewForm needs to make a
     * message box appear, it will be centred over this window.
     */
    NewForm(Gtk::Window & window);
    virtual ~NewForm();
    /** Signal emmited when the user picks a theme.
     * The singal should be wired to a function
     * @code void on_pick_theme(std::string filename)
     * @endcode
     * where the parameter is the filename of the theme.
     * @return signal that is emmited when the user double clicks a theme or
     * clicks new with a theme selected.
     */
    sigc::signal<void, std::string> signal_theme_picked();
    
    /** Find and list avaliable themes.
     * This must be called after creation if the NewForm is going to do
     * anything other than say "Loading".
     * This is a separate to the constuctor, so that finding the themes on
     * disk doesn't delay the time until the main window appears.
     */
    void find_themes();
    
    /** Give the theme list keyboard focus for the window it is inside.
     */
    void grab_focus();
protected:
    /// parent window
    Gtk::Window & window;
    /// message asking user to select a theme.
    Gtk::Label caption;
    /// shows the list of themes
    Gtk::IconView theme_icon_view;
    /// a button to press when you have picked the theme.
    Gtk::Button create_button;
    /// contain the icons
    Gtk::ScrolledWindow m_scrolled_window;
    
    
    /// description of data to show per theme icon. 
    class ThemeModelColumns : public Gtk::TreeModel::ColumnRecord
    {
    public:
        ///name that appears on the icon, it is the filename.
        Gtk::TreeModelColumn<Glib::ustring>                name;
        
        /**Not used yet.
         * @todo icons for themes.
         */
        Gtk::TreeModelColumn< Glib::RefPtr<Gdk::Pixbuf> >  thumbnail;
        
        /// Theme's file URI
        Gtk::TreeModelColumn<std::string>                  file;
        ThemeModelColumns()
        {
            add(name);
            add(thumbnail);
            add(file);
        }
    } theme_model_columns;
    
    /// list of themes.
    Glib::RefPtr<Gtk::ListStore> list_model_ptr;
    
    /// directory monitors and their paths
    std::vector<Glib::RefPtr<Gio::FileMonitor> > dir_mon;
    
    /// signal for when theme selected.
    sigc::signal<void, std::string> m_signal_theme_picked;
    
    /// True if 'loading' is displayed.
    bool m_loading;
    
    /** Find the themes.
     * Looks for themes in the data directory.
     * Also causes the data directory to be monitored.
     * Must not be called multiple times.
     */
    void set_themes();
    
    /** Add themes in a directory.
     * The directory will be monitored fo future changes. It is recursive, so
     * a theme directory under the given directory will also be added and
     * monitored.
     * @param filename the directory to add.
     */
    void add_themes_from(std::string filename);
    
    /** Change the list when a directory changes.
     * @param path the path from the working directory to the directory being
     * monitored that causes this event.
     */
    void on_dir_changed(const Glib::RefPtr<Gio::File>& file,
                        const Glib::RefPtr<Gio::File>& other_file,
                        Gio::FileMonitorEvent event_type,
                        std::string path);
    
    /** Show warning if there are no themes, otherwise show the themes.
     */
    void test_themes_avaliable();
    
    /** Callback for when the button to confirm the theme choice was pressed.
     */
    void on_new_button_clicked();
    
    /**Callback for when an theme was activated from the theme_icon_view.
     */
    void on_item_activated(const Gtk::TreeModel::Path & path);
    
    /** Add a file to the list, or monitor a directory, if it could contain a theme.
     * A valid file or directory must contain the word 'theme' (it is case
     * sensitive) in the name, must not be hidden, must not contain a '.', and
     * must not end in '~'. If the file is invlaid, it is ignored.
     * @param file_info Information about the file/directory to check.
     * @param path the path to the file to use in the full theme file name.
     * @return true if the file or directory was used, false if not.
     */
    bool add_file(Glib::RefPtr<Gio::FileInfo> file_info, std::string path);
};

#endif /*NEWFORM_H_*/
