/** @file NewForm.cpp
 *  @brief Implement the NewForm class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "NewForm.h"

#include <gtkmm/stock.h>
#include <gtkmm/messagedialog.h>

#include <Debug.h>

NewForm::NewForm(Gtk::Window & window)
    :   Gtk::VBox(false, 2)
    ,   window(window)
    ,   caption("Loading")
    ,   create_button(Gtk::Stock::NEW)
    ,   list_model_ptr(Gtk::ListStore::create(theme_model_columns))
    ,   m_loading(true)
{
    // caption
    pack_start(caption, false, true);
    caption.show();
    
    // theme icon view
    theme_icon_view.set_model(list_model_ptr);
    theme_icon_view.set_text_column(0);
    theme_icon_view.set_pixbuf_column(1);
    theme_icon_view.set_tooltip_column(2);
    theme_icon_view.signal_item_activated().connect(sigc::mem_fun(*this, &NewForm::on_item_activated));
    
    // pack and show icon view container
    m_scrolled_window.add(theme_icon_view);
    m_scrolled_window.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
    pack_start(m_scrolled_window);
    theme_icon_view.show();
    // Show area, only to reduce flicker in normal case when themes are found.
    m_scrolled_window.show();
    
    // create button
    create_button.get_image()->show();
    pack_end(create_button, false, true);
    create_button.signal_clicked().connect(sigc::mem_fun(*this, &NewForm::on_new_button_clicked));
    // Show create button, only to reduce flicker in normal case.
    create_button.show();
}

NewForm::~NewForm()
{
    
}

void NewForm::find_themes()
{
    if (m_loading)
    {
        m_loading = false;
        // Add themes to the theme_icon_view.
        set_themes();
        // check we found some.
        test_themes_avaliable();
    }
}

void NewForm::grab_focus()
{
    theme_icon_view.grab_focus();
}

void NewForm::set_themes()
{
    add_themes_from("data");
}

void NewForm::add_themes_from(std::string path)
{
    DEBUG_MESSAGE("Monitoring directory " << path);
    try
    {
        // get file list in this directory.
        Glib::RefPtr<Gio::File> file = Gio::File::create_for_path(path);
        // monitor the directory.
        dir_mon.push_back(file->monitor_directory());
        dir_mon.back()->signal_changed().connect(sigc::bind(sigc::mem_fun(*this, &NewForm::on_dir_changed), path)); 
        // get the files in the directory.
        Glib::RefPtr<Gio::FileEnumerator> dir = file->enumerate_children("standard::*");
        Glib::RefPtr<Gio::FileInfo> file_info;
        // iterate over files.
        while (file_info = dir->next_file())
        {
            add_file(file_info, path);
        }
        dir->close();
    }
    catch (Gio::Error err)
    {
        DEBUG_MESSAGE("Cannot access " << path << ": " << err.what());
        // nevermind, path probably doesn't exist anymore.
    }
}

void NewForm::on_dir_changed(const Glib::RefPtr<Gio::File>& file,
                             const Glib::RefPtr<Gio::File>& other_file,
                             Gio::FileMonitorEvent event_type,
                             std::string path)
{
    DEBUG_MESSAGE("A theme directory has changed.");
    Glib::RefPtr<Gio::FileInfo> file_info(file->query_info("standard:*"));
    switch (event_type)
    {
        case Gio::FILE_MONITOR_EVENT_CREATED:
            // new file, add to list or directories to check if valid.
            if (add_file(file_info, path))
            {
                // redraw the list
                theme_icon_view.queue_draw();
                test_themes_avaliable();
                return;
            }
        case Gio::FILE_MONITOR_EVENT_DELETED:
            /// iterate over objects in list to find the one that was deleted.
            {
                Gtk::ListStore::Children children = list_model_ptr->children();
                std::string name = path + '/' + file_info->get_name();
                for (Gtk::ListStore::Children::iterator it = children.begin();
                     it != children.end(); ++it)
                {
                    Gtk::TreeModel::Row row = *it;
                    if (it->get_value(theme_model_columns.file) == name)
                    {
                        // Found it.
                        // remove from list
                        DEBUG_MESSAGE("Removing deleted file from the list.");
                        list_model_ptr->erase(it);
                        theme_icon_view.queue_draw();
                        test_themes_avaliable();
                        return;
                    }
                }
            }
            // Cannot find it. It is probably a directory.
            /* It is easiest to clear the list and start again than to work
             * out what was in the directory.
             */
            list_model_ptr->clear();
            dir_mon.clear();
            set_themes();
            theme_icon_view.queue_draw();
            test_themes_avaliable();
            return;
        default:
            // different event, ignore.
            break;
    }
}

void NewForm::test_themes_avaliable()
{
    Gtk::ListStore::Children children = list_model_ptr->children();
    if (children.begin() == children.end())
    {
        // No themes. Try to explain... 
        caption.set_text(
            "You have no themes available!\n\n"
            "You can copy theme files into a folder under racer's data directory. "
            "Make sure your themes are in a non-hidden file, named without any "
            "'.' characters, not ending in '~', and containing the word "
            "'theme' (case sensative). You can place themes in any number of "
            "folders under racer/data meeting the same restrictions.\n\n"
            "You can also create a new theme in Blender and export it with "
            "the script provided. For details, please read "
            "http://sourceforge.net/apps/mediawiki/racer/index.php?title=Making_themes\n\n"
            "You cannot create or edit a track without a theme, so until you do "
            "one of the above this editor is useless."
            );
        caption.set_selectable();
        caption.set_line_wrap();
        // hide the icons and button
        m_scrolled_window.hide();
        create_button.hide();       
    } else {
        // themes are avaliable
        caption.set_text("Please select a theme for your new track:");
        caption.set_selectable(false);
        caption.set_line_wrap(false);
        // show the icons and button
        m_scrolled_window.show();
        create_button.show();
        
        // pick the first theme.
        theme_icon_view.select_path(Gtk::TreeModel::Path(children.begin()));
    }
}

void NewForm::on_new_button_clicked()
{
    // If a valid theme is selected, signal it.
    Gtk::IconView::ArrayHandle_TreePaths selection = theme_icon_view.get_selected_items();
    if (selection.size() == 1)
    {
        Gtk::TreePath path = *selection.begin();
        Gtk::TreeModel::Row row = *list_model_ptr->get_iter(path);
        m_signal_theme_picked.emit(row->get_value(theme_model_columns.file));
    } else
    {
        DEBUG_MESSAGE("Invalid selection.");
        Gtk::MessageDialog dialog(window,
                                  "Invalid theme selection",
                                  false,
                                  Gtk::MESSAGE_ERROR,
                                  Gtk::BUTTONS_OK,
                                  true);
        dialog.set_secondary_text(
          "You must select the theme for the new track before pressing new.");
        dialog.run();
    }
}

sigc::signal<void, std::string> NewForm::signal_theme_picked()
{
    return m_signal_theme_picked;
}

void NewForm::on_item_activated(const Gtk::TreeModel::Path & path)
{
    Gtk::TreeModel::Row row = *list_model_ptr->get_iter(path);
    m_signal_theme_picked.emit(row->get_value(theme_model_columns.file));
}

bool NewForm::add_file(Glib::RefPtr<Gio::FileInfo> file_info, std::string path)
{
    std::string file_name = file_info->get_name();
    if (  file_info->is_hidden()
        || file_name.find("theme") == std::string::npos
        || file_name.find(".") != std::string::npos
        || file_name[file_name.size() - 1] == '~')
    {
        /* remove hidden files, files which don't contain the word
         * 'theme', files ending in ~, and files containing dots.
         */
        return false;
    }
    switch (file_info->get_file_type())
    {
        case Gio::FILE_TYPE_NOT_KNOWN:
        case Gio::FILE_TYPE_SPECIAL:
        case Gio::FILE_TYPE_MOUNTABLE:
        case Gio::FILE_TYPE_SHORTCUT:
            // not a theme.
            return false;
            break;
        case Gio::FILE_TYPE_DIRECTORY:
            // a directory which could contain a theme. Monitor it.
            add_themes_from(path + '/' + file_name);
            return true;
            break;
        case Gio::FILE_TYPE_REGULAR:
        case Gio::FILE_TYPE_SYMBOLIC_LINK:
            // these can be a theme. Add them to the list.
            std::string base_name = Glib::path_get_basename(file_name);
            // add to list displayed.
            Gtk::ListStore::iterator it = list_model_ptr->append();
            (*it)[theme_model_columns.name] = file_name;
            (*it)[theme_model_columns.file] = path + '/' + base_name;
            break;
    }
    return true;
}
