/** @file LightingWindow.h
 *  @brief Declare the LightingWindow class. 
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef LIGHTING_WINDOW_H_
#define LIGHTING_WINDOW_H_

#include <vector>

#include <boost/shared_ptr.hpp>

#include <gtkmm/window.h>
#include <gtkmm/box.h>
#include <gtkmm/frame.h>
#include <gtkmm/table.h>
#include <gtkmm/label.h>
#include <gtkmm/colorbutton.h>
#include <gtkmm/spinbutton.h>
#include <gtkmm/checkbutton.h>

#include <libtrack/Lighting.h>

/// Widget to set the properties of a Track::LightSettings object.
class LightSettingWidget
    :   public Gtk::HBox
{
    public:
        LightSettingWidget();
        Track::LightSettings get_light_settings() const;
        void set_light_settings(Track::LightSettings settings);
        sigc::signal<void> signal_changed();
    protected:
        Gtk::Label m_x_label;
        Gtk::Adjustment m_x_adjustment;
        Gtk::SpinButton m_x_spin;
        Gtk::Label m_y_label;
        Gtk::Adjustment m_y_adjustment;
        Gtk::SpinButton m_y_spin;
        Gtk::Label m_z_label;
        Gtk::Adjustment m_z_adjustment;
        Gtk::SpinButton m_z_spin;
        Gtk::Label m_w_label;
        Gtk::Adjustment m_w_adjustment;
        Gtk::SpinButton m_w_spin;
        Gtk::Label m_colour_label;
        Gtk::ColorButton m_colour_button;
        
        Track::LightSettings m_light_settings;
        
        sigc::signal<void> m_signal_changed;
        
        void on_x_change();
        void on_y_change();
        void on_z_change();
        void on_w_change();
        void on_colour_change();
};

/// Window to set properties of a Track::Lighting object.
class LightingWindow
    : public Gtk::Window
{
    public:
        LightingWindow();
        const Track::Lighting & get_lighting() const;
        void set_lighting(const Track::Lighting & lighting);
        sigc::signal<void> signal_changed();
    protected:
        Gtk::VBox m_box;
        
        /// Frame surrounding fog settings
        Gtk::Frame m_fog_frame;
        /// Label in the title of m_fog_frame
        Gtk::Label m_fog_frame_label;
        /// Table in m_fog_frame that contains fog setting widgets.
        Gtk::Table m_fog_table;
        Gtk::CheckButton m_fog_enabled_button;
        Gtk::Label m_fog_colour_label;
        Gtk::ColorButton m_fog_colour_button;
        Gtk::Label m_fog_density_label;
        Gtk::Adjustment m_fog_density_adjustment;
        Gtk::SpinButton m_fog_density_spin;
        
        /// Frame surrounding light settings
        Gtk::Frame m_light_frame;
        /// Label in the title of m_light_frame
        Gtk::Label m_light_frame_label;
        /// Table in m_light_frame that contains LightSettingWidgets.
        Gtk::Table m_light_table;
        Gtk::Label m_ambient_light_label;
        Gtk::ColorButton m_ambient_light_button;
        Gtk::Label m_num_lights_label;
        Gtk::Adjustment m_num_lights_adjustment;
        Gtk::SpinButton m_num_lights_spin;
        std::vector<boost::shared_ptr<LightSettingWidget> > m_light_setting_widgets;
        
        sigc::signal<void> m_signal_changed;
        
        Track::Lighting m_lighting;
        
        void on_fog_colour_change();
        void on_fog_density_change();
        void on_fog_enabled_change();
        void on_ambient_light_change();
        void on_num_lights_change();
        void on_light_setting_changed(int index);
        
        /// Make m_light_table and its LightSettingWidgets reflect m_lighting.
        void update_light_widgets();
};

#endif // LIGHTING_WINDOW_H_
