/** @file LightingWindow.cpp
 *  @brief Implement the LightingWindow class. 
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "LightingWindow.h"
#include <Debug.h>

LightSettingWidget::LightSettingWidget()
    :   Gtk::HBox(false, 6)
    ,   m_x_label("x")
    ,   m_x_adjustment(0.5, -128, 128, 0.05, 0.2)
    ,   m_x_spin(m_x_adjustment, 0, 3)
    ,   m_y_label("y")
    ,   m_y_adjustment(0.5, -128, 128, 0.05, 0.2)
    ,   m_y_spin(m_y_adjustment, 0, 3)
    ,   m_z_label("z")
    ,   m_z_adjustment(0.5, -128, 128, 0.05, 0.2)
    ,   m_z_spin(m_z_adjustment, 0, 3)
    ,   m_w_label("w")
    ,   m_w_adjustment(0.5, -128, 128, 0.05, 0.2)
    ,   m_w_spin(m_w_adjustment, 0, 3)
    ,   m_colour_label("colour")
{
    pack_start(m_x_label);
    pack_start(m_x_spin);
    m_x_spin.signal_value_changed().connect(
            sigc::mem_fun(*this, &LightSettingWidget::on_x_change));
    pack_start(m_y_label);
    pack_start(m_y_spin);
    m_y_spin.signal_value_changed().connect(
            sigc::mem_fun(*this, &LightSettingWidget::on_y_change));
    pack_start(m_z_label);
    pack_start(m_z_spin);
    m_z_spin.signal_value_changed().connect(
            sigc::mem_fun(*this, &LightSettingWidget::on_z_change));
    pack_start(m_w_label);
    pack_start(m_w_spin);
    m_w_spin.signal_value_changed().connect(
            sigc::mem_fun(*this, &LightSettingWidget::on_w_change));
    pack_start(m_colour_label);
    pack_start(m_colour_button);
    m_colour_button.signal_color_set().connect(
            sigc::mem_fun(*this, &LightSettingWidget::on_colour_change));
}

Track::LightSettings LightSettingWidget::get_light_settings() const
{
    return m_light_settings;
}

void LightSettingWidget::set_light_settings(Track::LightSettings settings)
{
    m_light_settings = settings;
    m_x_spin.set_value(m_light_settings.position[0]);
    m_y_spin.set_value(m_light_settings.position[1]);
    m_z_spin.set_value(m_light_settings.position[2]);
    m_w_spin.set_value(m_light_settings.position[3]);
    Gdk::Color colour;
    colour.set_rgb(m_light_settings.colour[0]*65535,
                   m_light_settings.colour[1]*65535,
                   m_light_settings.colour[2]*65535);
    m_colour_button.set_color(colour);
    m_colour_button.set_alpha(m_light_settings.colour[3]*65535);
}

sigc::signal<void> LightSettingWidget::signal_changed()
{
    return m_signal_changed;
}

void LightSettingWidget::on_x_change()
{
    m_x_spin.update();
    m_light_settings.position[0] = m_x_spin.get_value();
    m_signal_changed.emit();
}

void LightSettingWidget::on_y_change()
{
    m_y_spin.update();
    m_light_settings.position[1] = m_y_spin.get_value();
    m_signal_changed.emit();
}

void LightSettingWidget::on_z_change()
{
    m_z_spin.update();
    m_light_settings.position[2] = m_z_spin.get_value();
    m_signal_changed.emit();
}

void LightSettingWidget::on_w_change()
{
    m_w_spin.update();
    m_light_settings.position[3] = m_w_spin.get_value();
    m_signal_changed.emit();
}

void LightSettingWidget::on_colour_change()
{
    Gdk::Color c = m_colour_button.get_color();
    m_light_settings.colour[0] = float(c.get_red()) / 65535.0;
    m_light_settings.colour[1] = float(c.get_green()) / 65535.0;
    m_light_settings.colour[2] = float(c.get_blue()) / 65535.0;
    m_light_settings.colour[3] = float(m_colour_button.get_alpha()) / 65535.0;
    m_signal_changed.emit();
}

LightingWindow::LightingWindow()
    :   m_box(false, 12)
    ,   m_fog_frame()
    ,   m_fog_frame_label("<b>Fog</b>")
    ,   m_fog_table(3, 2)
    ,   m_fog_enabled_button("Enable fog")
    ,   m_fog_colour_label("Fog colour")
    ,   m_fog_density_label("Fog density")
    ,   m_fog_density_adjustment(0.004, 0.00001, 1.0, 0.0005, 0.001)
    ,   m_fog_density_spin(m_fog_density_adjustment, 0.0, 5)
    ,   m_light_frame()
    ,   m_light_frame_label("<b>Lights</b>")
    ,   m_light_table(2, 2)
    ,   m_ambient_light_label("ambient light colour")
    ,   m_num_lights_label("number of lights")
    ,   m_num_lights_adjustment(2, 0, 8, 1, 2)
    ,   m_num_lights_spin(m_num_lights_adjustment)
{
    set_type_hint(Gdk::WINDOW_TYPE_HINT_DIALOG); 
    set_title("Lighting");
    
    m_fog_table.attach(m_fog_enabled_button, 0, 2, 0, 1);
    m_fog_table.attach(m_fog_colour_label, 0, 1, 1, 2);
    m_fog_table.attach(m_fog_colour_button, 1, 2, 1, 2);
    m_fog_table.attach(m_fog_density_label, 0, 1, 2, 3);
    m_fog_table.attach(m_fog_density_spin, 1, 2, 2, 3);
    m_fog_table.set_spacings(6);
    m_fog_table.set_border_width(4);
    m_fog_frame.add(m_fog_table);
    m_fog_frame.set_label_widget(m_fog_frame_label);
    m_fog_frame_label.set_use_markup();
    m_fog_frame.set_shadow_type(Gtk::SHADOW_NONE);
    
    m_light_table.attach(m_ambient_light_label, 0, 1, 0, 1);
    m_light_table.attach(m_ambient_light_button, 1, 2, 0, 1);
    m_light_table.attach(m_num_lights_label, 0, 1, 1, 2);
    m_light_table.attach(m_num_lights_spin, 1, 2, 1, 2);
    m_light_table.set_spacings(6);
    m_light_table.set_border_width(4);
    m_light_frame.add(m_light_table);
    m_light_frame.set_label_widget(m_light_frame_label);
    m_light_frame_label.set_use_markup();
    m_light_frame.set_shadow_type(Gtk::SHADOW_NONE);
    
    m_box.pack_start(m_fog_frame);
    m_box.pack_start(m_light_frame);
    add(m_box);
    m_box.show_all();
    
    m_fog_enabled_button.signal_clicked().connect(
            sigc::mem_fun(*this, &LightingWindow::on_fog_enabled_change));
    m_fog_colour_button.signal_color_set().connect(
            sigc::mem_fun(*this, &LightingWindow::on_fog_colour_change));
    m_fog_density_spin.signal_value_changed().connect(
            sigc::mem_fun(*this, &LightingWindow::on_fog_density_change));
    m_ambient_light_button.signal_color_set().connect(
            sigc::mem_fun(*this, &LightingWindow::on_ambient_light_change));
    m_num_lights_spin.signal_value_changed().connect(
            sigc::mem_fun(*this, &LightingWindow::on_num_lights_change));
}

const Track::Lighting & LightingWindow::get_lighting() const
{
    return m_lighting;
}

void LightingWindow::set_lighting(const Track::Lighting & settings)
{
    m_lighting = settings;
    /// @todo set widgets
    m_fog_density_adjustment.set_value(m_lighting.get_fog().density);
    m_fog_enabled_button.set_active(m_lighting.get_fog().enabled);
    Gdk::Color c;
    c.set_rgb(m_lighting.get_ambient_light()[0] * 65535, 
              m_lighting.get_ambient_light()[1] * 65535,
              m_lighting.get_ambient_light()[2] * 65535);
    m_ambient_light_button.set_color(c);
    m_ambient_light_button.set_alpha(m_lighting.get_ambient_light()[3]);
    c.set_rgb(m_lighting.get_fog().colour[0] * 65535, 
              m_lighting.get_fog().colour[1] * 65535,
              m_lighting.get_fog().colour[2] * 65535);
    m_fog_colour_button.set_color(c);
    m_fog_colour_button.set_alpha(m_lighting.get_fog().colour[3]);
    m_num_lights_spin.set_value(m_lighting.get_lights().size());
    update_light_widgets();
}

sigc::signal<void> LightingWindow::signal_changed()
{
    return m_signal_changed;
}

void LightingWindow::on_fog_colour_change()
{
    Track::FogSettings fog = m_lighting.get_fog();
    Gdk::Color c = m_fog_colour_button.get_color();
    fog.colour[0] = float(c.get_red())/ 65535.0;
    fog.colour[1] = float(c.get_green())/ 65535.0;
    fog.colour[2] = float(c.get_blue())/ 65535.0;
    fog.colour[3] = float(m_ambient_light_button.get_alpha()) / 65535.0;
    m_lighting.set_fog(fog);
    m_signal_changed.emit();
}

void LightingWindow::on_fog_density_change()
{
    Track::FogSettings fog = m_lighting.get_fog();
    m_fog_density_spin.update();
    fog.density = m_fog_density_spin.get_value();
    m_lighting.set_fog(fog);
    m_signal_changed.emit();
}

void LightingWindow::on_fog_enabled_change()
{
    Track::FogSettings fog = m_lighting.get_fog();
    fog.enabled = m_fog_enabled_button.get_active();
    m_fog_density_spin.set_sensitive(fog.enabled);
    m_fog_colour_button.set_sensitive(fog.enabled);
    m_lighting.set_fog(fog);
    m_signal_changed.emit();
}

void LightingWindow::on_ambient_light_change()
{
    float a[4];
    Gdk::Color c(m_ambient_light_button.get_color());
    a[0] = float (c.get_red()) / 65535.0;
    a[1] = float (c.get_green()) / 65535.0;
    a[2] = float (c.get_blue()) / 65535.0;
    a[3] = float (m_ambient_light_button.get_alpha()) / 65535.0;
    m_lighting.set_ambient(a);
    m_signal_changed.emit();
}

void LightingWindow::on_num_lights_change()
{
    int num_lights = m_num_lights_spin.get_value_as_int();
    std::vector<Track::LightSettings> lights = m_lighting.get_lights();
    lights.resize(num_lights);
    m_lighting.set_lights(lights);
    update_light_widgets();
    m_signal_changed.emit();
}

void LightingWindow::on_light_setting_changed(int index)
{
    std::vector<Track::LightSettings> settings = m_lighting.get_lights();
    settings[index] = m_light_setting_widgets[index]->get_light_settings();
    m_lighting.set_lights(settings);
    m_signal_changed.emit();
}

void LightingWindow::update_light_widgets()
{
    int num_lights = m_lighting.get_lights().size();
    m_light_setting_widgets.resize(num_lights);
    m_light_table.resize(num_lights+2, 2);
    for (int i = 0; i < num_lights; i++)
    {
        m_light_setting_widgets[i] = boost::shared_ptr<LightSettingWidget>(new LightSettingWidget);
        m_light_setting_widgets[i]->set_light_settings(m_lighting.get_lights()[i]);
        m_light_setting_widgets[i]->show_all();
        m_light_setting_widgets[i]->signal_changed().connect(
            sigc::bind(sigc::mem_fun(*this,
                                     &LightingWindow::on_light_setting_changed),
                       i));
        m_light_table.attach(*m_light_setting_widgets[i], 0, 2, i+2, i+3);
    }
}
