/** @file EditorWindow.h
 *  @brief Declare the EditorWindow class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef EDITORWINDOW_H_
#define EDITORWINDOW_H_

#include <boost/shared_ptr.hpp>

#include <gtkmm/window.h>
#include <gtkmm/box.h>
#include <glibmm/refptr.h>
#include <gtkmm/actiongroup.h>
#include <gtkmm/uimanager.h>
#include <gtkmm/statusbar.h>
#include <gtkmm/paned.h>
#include <gtkmm/frame.h>
#include <gtkmm/toggleaction.h>

#include "NewForm.h"
#include "Viewport.h"
#include "EditorGuide.h"
#include "LightingWindow.h"
#include <libtrack/document/Document.h>

/** A big window for editing tracks in.
 * It creates a top level window, managed by the users Window manager.
 * There is menus, a toolbar, a status bar, and a place to edit the track in
 * the middle.
 */
class EditorWindow : public Gtk::Window
{
public:
    /** Create and show the editor window.
     * @param show_guide True iff the guide should be displayed.
     * @param filename The file name of the track to open, or the empty string
     * if none.
     */
    EditorWindow(bool show_guide, std::string filename = std::string());
    virtual ~EditorWindow();
    /** Copy constructor.
     * The new window will work on the same document as source, but have a
     * different view of it.
     * @param source The window to share with.
     */
    EditorWindow (EditorWindow & source);
protected:
    /// Assemble widgets and menu and toolbar items.
    void assemble_ui();
    
    /// main box for menu bar, toolbar, work area, and status bar.
    Gtk::VBox m_box;
    /// status bar
    Gtk::Statusbar m_status_bar;
    // action stuff
    Glib::RefPtr<Gtk::ActionGroup> m_ref_action_group; 
    Glib::RefPtr<Gtk::UIManager> m_ref_ui_manager;
    Glib::RefPtr<Gtk::ToggleAction> m_ref_action_guide;
    
    /// Form asking for a theme that appears when a new track is created.
    NewForm m_new_form;
    
    /// The bar that displays usage tips.
    EditorGuide m_guide;
    
    /// Extra window for displaying the light settings.
    LightingWindow m_lighting_window;
    
    Viewport m_viewport_top;
    
    // do commands when activated (through toolbar, menu, or keyboard shortcut).
    // file menu commands
    void on_action_file_new();
    void on_action_file_open();
    void on_action_file_save();
    void on_action_file_save_as();
    void on_action_file_test();
    void on_action_file_quit();
    // edit menu commands
    void on_action_edit_undo();
    void on_action_edit_redo();
    void on_action_edit_cut();
    void on_action_edit_copy();
    void on_action_edit_paste();
    void on_action_edit_select_all();
    void on_action_edit_delete();
    void on_action_edit_lighting();
    // view commands
    void on_action_view_zoom_in();
    void on_action_view_zoom_out();
    void on_action_view_zoom_to_fit();
    void on_action_view_top();
    void on_action_view_front();
    void on_action_view_side();
    void on_action_view_new_window();
    
    // help commands
    void on_action_help_guide();
    void on_action_help_wiki();
    void on_action_help_about();
    
    /// theme picked
    void on_theme_picked(std::string filename);
    
    /// Change from m_lighting_window
    void on_lighting_changed();
    
    /// Update after a change in the document.
    void on_command_run();
    
    /// Change the document using a command generated by a view
    void on_view_command(boost::shared_ptr<Document::DocumentDelta> delta);
    
    /// Preview a change to the document using a command generated by a view
    void on_view_preview_command(boost::shared_ptr<Document::DocumentDelta> delta);
    
    /// Reset the document when a previewed command is not wanted.
    void on_view_preview_cancel();
    
    /// Confirm when closing the window with unsaved changes.
    virtual bool on_delete_event(GdkEventAny* event);

    
    /** Check if it is OK to clear the document. If the document has unsaved
     * changes, this asks the user for the result. If the user agrees but wishes
     * to save first, the document is saved and true is called.
     * @note The document is not cleared.
     * @return true if the document is cleared, false if you shuoldn't do it.
     */
    bool check_clear_document();
    
    /// load a theme from a URI
    void load_theme(std::string filename);
    
    /// show / activate controls for after a theme is picked or file is loaded. 
    void show_editor();
    
    /// Open a track file.
    void open_file(std::string filename);
    
    boost::shared_ptr<Track::Theme>  m_theme;
    boost::shared_ptr<Document::Document> m_document;
    
    /// false if the filename has not been picked, true if it has.
    bool filename_set;
    
    /// filename for the open document.
    std::string filename;
    
    /// filename for the theme
    std::string theme_filename;
    
    /// Filename for saving customised keyboard accelerators.
    std::string accel_filename;
    
    bool m_guide_preference;
};

#endif /*EDITORWINDOW_H_*/
