/** @file EditorWindow.cpp
 *  @brief Implement the EditorWindow class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "EditorWindow.h"

#include <gtkmm/stock.h>
#include <gtkmm/filechooserdialog.h>
#include <gtkmm/aboutdialog.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/main.h>
#include <gtkmm/stock.h>
#include <gtkmm/accelmap.h>

#include <giomm/appinfo.h>

#include <Debug.h>
#include <config.h>
#include <fstream>
#include <sstream>

#include <libtrack/FormErrors.h>
#include <libtrack/document/ChangeLightingDelta.h>

EditorWindow::EditorWindow(bool show_guide, std::string filename)
    :   m_ref_action_group(Gtk::ActionGroup::create())
    ,   m_ref_ui_manager(Gtk::UIManager::create())
    ,   m_new_form(*this)
    ,   m_viewport_top()
    ,   m_guide_preference(show_guide)
{
    set_title("Racer Editor");
    assemble_ui();
    m_viewport_top.set_angle(View::VIEW_TOP);
    m_new_form.show();
    on_action_file_new();
    show();
    // Find the themes after the window is shown to make starting the program
    // more responsive.
    
    while (Gtk::Main::instance()->events_pending())
    {
        Gtk::Main::instance()->iteration();
    }
    m_new_form.find_themes();
    if (filename != "")
    {
        open_file(filename);
    } else {
        // give the theme list focus.
        m_new_form.grab_focus();
    }
}

void EditorWindow::assemble_ui()
{
    add(m_box);
    
    
    // file menu stuff
    m_ref_action_group->add( Gtk::Action::create("MenuFile", "_File") );
    m_ref_action_group->add( Gtk::Action::create("New", Gtk::Stock::NEW),
      sigc::mem_fun(*this, &EditorWindow::on_action_file_new) );
    m_ref_action_group->add( Gtk::Action::create("Open", Gtk::Stock::OPEN),
      sigc::mem_fun(*this, &EditorWindow::on_action_file_open) );
    m_ref_action_group->add( Gtk::Action::create("Save", Gtk::Stock::SAVE),
      sigc::mem_fun(*this, &EditorWindow::on_action_file_save) );
    m_ref_action_group->add( Gtk::Action::create("SaveAs", Gtk::Stock::SAVE_AS),
      sigc::mem_fun(*this, &EditorWindow::on_action_file_save_as) );
    
    m_ref_action_group->add( Gtk::Action::create("Test", Gtk::Stock::MEDIA_PLAY, "Test", "Try a race"),
      sigc::mem_fun(*this, &EditorWindow::on_action_file_test) );
      
    m_ref_action_group->add( Gtk::Action::create("Quit", Gtk::Stock::QUIT),
      sigc::mem_fun(*this, &EditorWindow::on_action_file_quit) );
    
    // edit menu stuff
    m_ref_action_group->add( Gtk::Action::create("MenuEdit", "_Edit") );
    
    m_ref_action_group->add( Gtk::Action::create("Undo", Gtk::Stock::UNDO),
      Gtk::AccelKey('z', Gdk::CONTROL_MASK, "<racer-editor-track>/Edit/Undo"),
      sigc::mem_fun(*this, &EditorWindow::on_action_edit_undo) );
    m_ref_action_group->add( Gtk::Action::create("Redo", Gtk::Stock::REDO),
      Gtk::AccelKey('z', Gdk::CONTROL_MASK | Gdk::SHIFT_MASK, "<racer-editor-track>/Edit/Redo"),
      sigc::mem_fun(*this, &EditorWindow::on_action_edit_redo) );
    
    m_ref_action_group->add( Gtk::Action::create("Cut", Gtk::Stock::CUT),
      sigc::mem_fun(*this, &EditorWindow::on_action_edit_cut) );
    m_ref_action_group->add( Gtk::Action::create("Copy", Gtk::Stock::COPY),
      sigc::mem_fun(*this, &EditorWindow::on_action_edit_copy) );
    m_ref_action_group->add( Gtk::Action::create("Paste", Gtk::Stock::PASTE),
      sigc::mem_fun(*this, &EditorWindow::on_action_edit_paste) );
      
    m_ref_action_group->add( Gtk::Action::create("Delete", Gtk::Stock::DELETE),
      sigc::mem_fun(*this, &EditorWindow::on_action_edit_delete) );
    m_ref_action_group->add( Gtk::Action::create("SelectAll", Gtk::Stock::SELECT_ALL),
      sigc::mem_fun(*this, &EditorWindow::on_action_edit_select_all) );
    m_ref_action_group->add( Gtk::Action::create("Lighting", Gtk::Stock::COLOR_PICKER, "Lighting..."),
      Gtk::AccelKey("F2", "<racer-editor-track>/Edit/Lighting"),
      sigc::mem_fun(*this, &EditorWindow::on_action_edit_lighting) );
    
    // view menu stuff
    m_ref_action_group->add(Gtk::Action::create("MenuView", "_View"));
    m_ref_action_group->add( Gtk::Action::create("ZoomIn", Gtk::Stock::ZOOM_IN),
      Gtk::AccelKey('+', Gdk::CONTROL_MASK, "<racer-editor-track>/View/ZoomIn"),
      sigc::mem_fun(*this, &EditorWindow::on_action_view_zoom_in) );
    m_ref_action_group->add( Gtk::Action::create("ZoomOut", Gtk::Stock::ZOOM_OUT),
      Gtk::AccelKey('-', Gdk::CONTROL_MASK, "<racer-editor-track>/View/ZoomOut"),
      sigc::mem_fun(*this, &EditorWindow::on_action_view_zoom_out) );
    m_ref_action_group->add( Gtk::Action::create("ZoomToFit", Gtk::Stock::ZOOM_FIT),
      Gtk::AccelKey('0', Gdk::CONTROL_MASK, "<racer-editor-track>/View/ZoomToFit"),
      sigc::mem_fun(*this, &EditorWindow::on_action_view_zoom_to_fit) );
    /// @todo make icons for top/front/side
    m_ref_action_group->add( Gtk::Action::create("ShowTop", Gtk::Stock::GO_DOWN, "Show top view"),
      Gtk::AccelKey('1', Gdk::CONTROL_MASK, "<racer-editor-track>/View/ShowTop"),
      sigc::mem_fun(*this, &EditorWindow::on_action_view_top) );
    m_ref_action_group->add( Gtk::Action::create("ShowFront", Gtk::Stock::GO_FORWARD, "Show front view"),
      Gtk::AccelKey('2', Gdk::CONTROL_MASK, "<racer-editor-track>/View/ShowFront"),
      sigc::mem_fun(*this, &EditorWindow::on_action_view_front) );
    m_ref_action_group->add( Gtk::Action::create("ShowSide", Gtk::Stock::GO_BACK, "Show side view"),
      Gtk::AccelKey('3', Gdk::CONTROL_MASK, "<racer-editor-track>/View/ShowSide"),
      sigc::mem_fun(*this, &EditorWindow::on_action_view_side) );
    m_ref_action_group->add( Gtk::Action::create("NewWindow", Gtk::Stock::LEAVE_FULLSCREEN, "New window"),
      sigc::mem_fun(*this, &EditorWindow::on_action_view_new_window) );
    
    // help menu stuff
    m_ref_action_group->add(Gtk::Action::create("MenuHelp", "_Help"));
    m_ref_action_guide = Gtk::ToggleAction::create("HelpGuide", Gtk::Stock::HELP, "Show Guide", "Show tips under the toolbar.", true);
    m_ref_action_group->add(m_ref_action_guide,
      Gtk::AccelKey("F1", "<racer-editor-track>/Help/ShowGuide"),
      sigc::mem_fun(*this, &EditorWindow::on_action_help_guide) );
    m_ref_action_group->add( Gtk::Action::create("HelpWiki", Gtk::Stock::HELP, "Open website"),
      sigc::mem_fun(*this, &EditorWindow::on_action_help_wiki) );
    m_ref_action_group->add( Gtk::Action::create("HelpAbout", Gtk::Stock::ABOUT),
      sigc::mem_fun(*this, &EditorWindow::on_action_help_about) );
    
    // load any customised keyboard accelerators.
    accel_filename = Glib::get_user_config_dir() + "/racer-editor_accel_map";
    Gtk::AccelMap::load(accel_filename);
    DEBUG_MESSAGE("Loaded accelerators from " << accel_filename);
      
    // use all these actions
    m_ref_ui_manager->insert_action_group(m_ref_action_group);
    add_accel_group(m_ref_ui_manager->get_accel_group());
    
    Glib::ustring ui_info =
        "<ui>"
        "  <menubar name='MenuBar'>"
        "    <menu action='MenuFile'>"
        "      <menuitem action='New'/>"
        "      <menuitem action='Open'/>"
        "      <menuitem action='Save'/>"
        "      <menuitem action='SaveAs'/>"
        "      <separator/>"
        "      <menuitem action='Test'/>"
        "      <separator/>"
        "      <menuitem action='Quit'/>"
        "    </menu>"
        "    <menu action='MenuEdit'>"
        "      <menuitem action='Undo'/>"
        "      <menuitem action='Redo'/>"
        "      <separator/>"
        "      <menuitem action='Lighting'/>"
    //    "      <separator/>"
    //    "      <menuitem action='Cut'/>"
    //    "      <menuitem action='Copy'/>"
    //    "      <menuitem action='Paste'/>"
    //    "      <separator/>"
    //    "      <menuitem action='Delete'/>"
    //    "      <menuitem action='SelectAll'/>"
        "    </menu>"
        "    <menu action='MenuView'>"
        "      <menuitem action='ZoomIn'/>"
        "      <menuitem action='ZoomOut'/>"
        "      <menuitem action='ZoomToFit'/>"
        "      <separator/>"
        "      <menuitem action='ShowTop'/>"
        "      <menuitem action='ShowFront'/>"
        "      <menuitem action='ShowSide'/>"
        "      <separator/>"
        "      <menuitem action='NewWindow'/>"
        "    </menu>"
        "    <menu action='MenuHelp'>"
        "      <menuitem action='HelpGuide'/>"
        "      <menuitem action='HelpWiki'/>"
        "      <menuitem action='HelpAbout'/>"
        "    </menu>"
        "  </menubar>"
        "  <toolbar  name='ToolBar'>"
        "    <toolitem action='Quit'/>"
        "    <separator/>"
        "    <toolitem action='New'/>"
        "    <toolitem action='Open'/>"
        "    <toolitem action='Save'/>"
        "    <separator/>"
        "    <toolitem action='Test'/>"
        "    <separator/>"
        "    <toolitem action='Undo'/>"
        "    <toolitem action='Redo'/>"
        "    <separator/>"
    //    "    <toolitem action='Cut'/>"
    //    "    <toolitem action='Copy'/>"
    //    "    <toolitem action='Paste'/>"
    //    "    <separator/>"
    //    "    <toolitem action='Delete'/>"
    //    "    <separator/>"
        "    <toolitem action='ZoomIn'/>"
        "    <toolitem action='ZoomOut'/>"
        "    <toolitem action='ZoomToFit'/>"
        "    <separator/>"
        "    <toolitem action='ShowTop'/>"
        "    <toolitem action='ShowFront'/>"
        "    <toolitem action='ShowSide'/>"
        "    <separator/>"
        "    <toolitem action='NewWindow'/>"        
        "  </toolbar>"
        "</ui>";

    m_ref_ui_manager->add_ui_from_string(ui_info);
    Gtk::Widget* menu_bar = m_ref_ui_manager->get_widget("/MenuBar");
    m_box.pack_start(*menu_bar, Gtk::PACK_SHRINK);
    Gtk::Widget* tool_bar = m_ref_ui_manager->get_widget("/ToolBar");
    m_box.pack_start(*tool_bar, Gtk::PACK_SHRINK);
    tool_bar->show();
    
    m_box.pack_end(m_status_bar, Gtk::PACK_SHRINK);
    m_status_bar.show();
    
    // The usage guide bar.
    m_box.pack_start(m_guide, Gtk::PACK_SHRINK);
    m_guide.set_visible(m_guide_preference);
    // Set the menu item's active state to reflect the visiblity preference.
    Glib::RefPtr<Gtk::Action> guide= m_ref_action_group->get_action("HelpGuide");
    Glib::RefPtr<Gtk::ToggleAction> guide_t= Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(guide);
    assert(guide_t);
    guide_t->set_active(m_guide_preference);
    
    // the scrollable top down view.
    m_box.pack_start(m_viewport_top);
    m_viewport_top.signal_command().connect(
        sigc::mem_fun(*this, &EditorWindow::on_view_command));
    m_viewport_top.signal_preview_command().connect(
        sigc::mem_fun(*this, &EditorWindow::on_view_preview_command));
    m_viewport_top.signal_preview_cancel().connect(
        sigc::mem_fun(*this, &EditorWindow::on_view_preview_cancel));
    
    m_box.pack_start(m_new_form);
    m_new_form.signal_theme_picked().connect(
        sigc::mem_fun(*this, &EditorWindow::on_theme_picked));
    
    m_lighting_window.signal_changed().connect(
        sigc::mem_fun(*this, &EditorWindow::on_lighting_changed));
    
    m_box.show();
    
    set_default_size(640, 480);
}

EditorWindow::EditorWindow (EditorWindow & source)
    :   m_ref_action_group(Gtk::ActionGroup::create())
    ,   m_ref_ui_manager(Gtk::UIManager::create())
    ,   m_new_form(*this)
    ,   m_viewport_top(source.m_viewport_top)
    ,   m_theme(source.m_theme)
    ,   m_document(source.m_document)
    ,   filename_set(source.filename_set)
    ,   filename(source.filename)
    ,   theme_filename(source.theme_filename)
{
    set_title("Racer Editor (copy)");
    assemble_ui();
    show();
    if (m_document)
    {
        m_document->signal_command_run().connect(
            sigc::mem_fun(*this, &EditorWindow::on_command_run));
        // show tools.
        show_editor();
        // make sure undo / redo actions' availability is set.
        on_command_run();
    }
    else
    {
        m_new_form.show();
    }
}

EditorWindow::~EditorWindow()
{
    // save settings.
    std::string config_filename = Glib::get_user_config_dir() + "/racer-editor.config";
    DEBUG_MESSAGE("Saving configuration to " << config_filename);
    std::ofstream config_file(config_filename.c_str());
    
    config_file << "#Set to true to show the usage guide." << std::endl;
    config_file << "show-guide=" << (m_guide_preference ? "true" : "false") << std::endl;
    
}

void EditorWindow::on_action_file_new()
{
    // check if it is allowed first
    if (!check_clear_document()) return;
    // allowed. Begin by forgeting the current file if applicable.
    filename_set = false;
    m_document = boost::shared_ptr<Document::Document>();
    m_theme = boost::shared_ptr<Track::Theme>();
    
    m_new_form.show();
    m_viewport_top.hide();
    
    m_guide.hide();
    
    // disable actions that make no sense with no file open.
    m_ref_action_group->get_action("New")->set_sensitive(false);
    m_ref_action_group->get_action("Save")->set_sensitive(false);
    m_ref_action_group->get_action("SaveAs")->set_sensitive(false);
    m_ref_action_group->get_action("Undo")->set_sensitive(false);
    m_ref_action_group->get_action("Redo")->set_sensitive(false);
    m_ref_action_group->get_action("Test")->set_sensitive(false);
    m_ref_action_group->get_action("Cut")->set_sensitive(false);
    m_ref_action_group->get_action("Copy")->set_sensitive(false);
    m_ref_action_group->get_action("Paste")->set_sensitive(false);
    m_ref_action_group->get_action("Delete")->set_sensitive(false);
    m_ref_action_group->get_action("SelectAll")->set_sensitive(false);
    m_ref_action_group->get_action("Lighting")->set_sensitive(false);
    m_ref_action_group->get_action("ZoomIn")->set_sensitive(false);
    m_ref_action_group->get_action("ZoomOut")->set_sensitive(false);
    m_ref_action_group->get_action("ZoomToFit")->set_sensitive(false);
    m_ref_action_group->get_action("ShowTop")->set_sensitive(false);
    m_ref_action_group->get_action("ShowFront")->set_sensitive(false);
    m_ref_action_group->get_action("ShowSide")->set_sensitive(false);
    m_ref_action_group->get_action("NewWindow")->set_sensitive(false);
    m_ref_action_group->get_action("HelpGuide")->set_sensitive(false);
}

void EditorWindow::on_action_file_open()
{
    // check for unsaved changes...
    if (!check_clear_document()) return;
    // open the file.
    Gtk::FileChooserDialog open_dialog(*this, "Choose file to open");
    open_dialog.add_shortcut_folder("data/tracks");
    open_dialog.set_current_folder("data/tracks");
    open_dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_REJECT);
    open_dialog.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_ACCEPT);
    int result = open_dialog.run();
    if (result == Gtk::RESPONSE_ACCEPT)
    {
        open_dialog.hide();
        // open the file
        open_file(open_dialog.get_filename());
    }
}

void EditorWindow::open_file(std::string filename_in)
{
    filename_set = true;
    filename = filename_in;
    std::ifstream file;
    file.exceptions(std::ifstream::eofbit |
                    std::ifstream::failbit |
                    std::ifstream::badbit);
    try
    {
        m_status_bar.push("Loading track");
        while (Gtk::Main::events_pending()) Gtk::Main::iteration();
        file.open(filename.c_str());
        std::getline(file, theme_filename);
        m_status_bar.push("Loading theme");
        while (Gtk::Main::events_pending()) Gtk::Main::iteration();
        load_theme(theme_filename);
        m_status_bar.pop();
        while (Gtk::Main::events_pending()) Gtk::Main::iteration();
        m_document = boost::shared_ptr<Document::Document>(new Document::Document(file, *m_theme));
        // monitor document for changes
        m_document->signal_command_run().connect(
            sigc::mem_fun(*this, &EditorWindow::on_command_run));
        // show tools.
        show_editor();
        // update display.
        on_command_run();
        m_status_bar.pop();
    }
    catch (std::ifstream::failure error) 
    {
        Gtk::MessageDialog dialog(*this,
                                  "Error, cannot load file. "
                                  "It may be invalid or unreadable.",
                                  false,
                                  Gtk::MESSAGE_ERROR);
        dialog.run();
        on_action_file_new();
        m_status_bar.pop(); m_status_bar.pop(); // clear loading messages.
    }
    catch (ThemeChangedError)
    {
        Gtk::MessageDialog dialog(*this,
                                  "Error, cannot load file. "
                                  "The track's theme had data removed which the track depended on.",
                                  false,
                                  Gtk::MESSAGE_ERROR);
        dialog.run();
        on_action_file_new();
        m_status_bar.pop(); m_status_bar.pop(); // clear loading messages.
    }
    catch (CorruptedError)
    {
        Gtk::MessageDialog dialog(*this,
                                  "Error, cannot load file. "
                                  "The file has been corrupted.",
                                  false,
                                  Gtk::MESSAGE_ERROR);
        dialog.run();
        m_status_bar.pop(); m_status_bar.pop(); // clear loading messages.
    }
    catch (NewVersionError)
    {
        Gtk::MessageDialog dialog(*this,
                                  "Error, cannot load file. "
                                  "The file has been written in a version newer than this editor understands. Please upgrade.",
                                  false,
                                  Gtk::MESSAGE_ERROR);
        dialog.run();
        on_action_file_new();
        m_status_bar.pop(); m_status_bar.pop(); // clear loading messages.
    }
    catch (DepreciatedVersionError)
    {
        Gtk::MessageDialog dialog(*this,
                                  "Error, cannot load file. "
                                  "The file uses data in a format no longer supported. You may be able to open the file using an older version of the editor.",
                                  false,
                                  Gtk::MESSAGE_ERROR);
        dialog.run();
        on_action_file_new();
        m_status_bar.pop(); m_status_bar.pop(); // clear loading messages.
    }
}

void EditorWindow::on_action_file_save()
{
    if (filename_set)
    {
        std::ofstream file;
        file.exceptions(std::ofstream::eofbit |
                        std::ofstream::failbit |
                        std::ofstream::badbit);
        try
        {
            m_status_bar.push("Saving track");
            while (Gtk::Main::events_pending()) Gtk::Main::iteration();
            file.open(filename.c_str());
            file << theme_filename << "\n";
            m_document->save(file);
            m_status_bar.pop();
        }
        catch (std::ofstream::failure error)
        {
            Gtk::MessageDialog dialog(*this, "Error, cannot save file.",
                                      false, Gtk::MESSAGE_ERROR);
            dialog.run();
        }
    }
    else
    {
        // Ask for a filename since it wasn't set. If the user doesn't press
        // cancel, this function will be called again but filename_set will be
        // true, so the file will be written.
        on_action_file_save_as();
    }
}

void EditorWindow::on_action_file_save_as()
{
    Gtk::FileChooserDialog dialog (*this, "Choose where to save",
                                   Gtk::FILE_CHOOSER_ACTION_SAVE);
    dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_REJECT);
    dialog.add_button(Gtk::Stock::SAVE,   Gtk::RESPONSE_ACCEPT);
    dialog.add_shortcut_folder("data/tracks");
    dialog.set_current_folder("data/tracks");
    dialog.set_do_overwrite_confirmation();
    if (dialog.run() == Gtk::RESPONSE_ACCEPT)
    {
        // user accepts, save
        filename_set = true;
        filename = dialog.get_filename();
        on_action_file_save();
    }
}

void EditorWindow::on_action_file_test()
{
    if (!m_document->is_saved())
    {
        // not saved, ask for confirmation to save it first.
        Gtk::MessageDialog dialog(*this,
                                  Glib::ustring("Would you like to save?"),
                                  false,
                                  Gtk::MESSAGE_QUESTION,
                                  Gtk::BUTTONS_OK_CANCEL);
        dialog.set_secondary_text("You must save your track before you can play on it.");
        int result = dialog.run();
        switch (result)
        {
            case Gtk::RESPONSE_OK:
                on_action_file_save();
                break;
            default:
                return;
                break;
        }
    }
    std::string command("./racer " + filename);
    DEBUG_MESSAGE("Starting racer with command " << command);
    try
    {
        Glib::spawn_command_line_async(command);
    }
    catch (Glib::Error e)
    {
        /** @todo This doesn't happen for all errors, but should happen for
         * more.
         */
        Gtk::MessageDialog dialog(*this,
                                  "Cannot launch racer, please start it manually.",
                                  false,
                                  Gtk::MESSAGE_ERROR);
        dialog.set_secondary_text("The command " + command + " failed, with error " + e.what() + ".");
    }
}

void EditorWindow::on_action_file_quit()
{
    GdkEventAny event;
    if (!on_delete_event(&event)) hide();
}

void EditorWindow::on_action_edit_undo()
{
    m_document->undo_command();
}

void EditorWindow::on_action_edit_redo()
{
    m_document->redo_command();
}

void EditorWindow::on_action_edit_cut()
{
    PRINT_STUB_MESSAGE;
}

void EditorWindow::on_action_edit_copy()
{
    PRINT_STUB_MESSAGE;
}

void EditorWindow::on_action_edit_paste()
{
    PRINT_STUB_MESSAGE;
}

void EditorWindow::on_action_edit_select_all()
{
    PRINT_STUB_MESSAGE;
}

void EditorWindow::on_action_edit_delete()
{
    PRINT_STUB_MESSAGE;
}

void EditorWindow::on_action_edit_lighting()
{
    /// @todo Provide timestamp of event to present.
    m_lighting_window.present();
    m_lighting_window.raise();
}

void EditorWindow::on_action_view_zoom_in()
{
    m_viewport_top.zoom_in();
}

void EditorWindow::on_action_view_zoom_out()
{
    m_viewport_top.zoom_out();
}

void EditorWindow::on_action_view_zoom_to_fit()
{
    m_viewport_top.scale_to_fit();
}

void EditorWindow::on_action_view_top()
{
    m_viewport_top.set_angle(View::VIEW_TOP);
}

void EditorWindow::on_action_view_front()
{
    m_viewport_top.set_angle(View::VIEW_FRONT);
}

void EditorWindow::on_action_view_side()
{
    m_viewport_top.set_angle(View::VIEW_SIDE);
}

void EditorWindow::on_action_view_new_window()
{
    EditorWindow new_window(*this);
    new_window.show();
}

void EditorWindow::on_action_help_guide()
{
    m_guide.set_visible(m_ref_action_guide->get_active());
    m_guide_preference = m_guide.is_visible();
}

void EditorWindow::on_action_help_wiki()
{
    std::string url("http://sourceforge.net/apps/mediawiki/racer/index.php?title=Editor");
    std::string command("xdg-open " + url);
    try
    {
        Glib::spawn_command_line_async(command);
    }
    catch (Glib::Error e)
    {
        /** @todo This doesn't happen for all errors, but should happen for
         * more.
         */
        Gtk::MessageDialog dialog(*this,
                                  "Cannot open web browser, please open " + url + " manually.",
                                  false,
                                  Gtk::MESSAGE_ERROR);
        dialog.set_secondary_text("The command " + command + " failed, with error " + e.what() + ".");
    }
}

void EditorWindow::on_action_help_about()
{
    Gtk::AboutDialog about;
    about.set_comments("A track editor for the game Racer.\n"
                        "Built on " BUILD_TIME ".");
    about.set_copyright("Copyright © 2009 James Legg.");
    about.set_version(VERSION);
    about.set_license(
    "Racer Editor is free software: you can redistribute it and/or modify "
    "it under the terms of the GNU General Public License as published by "
    "the Free Software Foundation, either version 3 of the License, or "
    "(at your option) any later version.\n\n"
    
    "Racer Editor is distributed in the hope that it will be useful, "
    "but without any warranty; without even the implied warranty of "
    "merchantability or fitness for a particular purpose.  See the "
    "GNU General Public License for more details. \n\n"

    "You should have received a copy of the GNU General Public License "
    "along with Racer Editor.  If not, see <http://www.gnu.org/licenses/>."
    );
    about.set_wrap_license(true);
    about.set_website("http://racer.sourceforge.net/");
    about.run();
}

void EditorWindow::on_theme_picked(std::string filename)
{
    m_status_bar.push("Loading theme.");
    while (Gtk::Main::events_pending()) Gtk::Main::iteration();
    // load the selected theme
    Glib::ustring host_name;
    load_theme(filename.c_str());
    
    // create a new document
    m_document = boost::shared_ptr<Document::Document>(new Document::Document(*m_theme));
    // monitor the document for changes.
    m_document->signal_command_run().connect(
                sigc::mem_fun(*this, &EditorWindow::on_command_run));
    
    // show tools that now make sense
    show_editor();
    
    // show the new document
    on_command_run();
    m_status_bar.pop();
}

void EditorWindow::on_lighting_changed()
{
    if (m_lighting_window.get_lighting() != m_document->get_track().get_lighting())
    {
        boost::shared_ptr<Document::DocumentDelta> delta(
                    new Document::ChangeLightingDelta(
                        m_lighting_window.get_lighting()));
        m_document->do_command(delta);
    }
}

void EditorWindow::on_command_run()
{
    // check undo/redo state
    m_ref_action_group->get_action("Undo")->set_sensitive(
                    m_document->get_undo_avaliable());
    m_ref_action_group->get_action("Redo")->set_sensitive(
                    m_document->get_redo_avaliable());
    // update displayed data
    m_viewport_top.update();
    m_lighting_window.set_lighting(m_document->get_track().get_lighting());
}

void EditorWindow::on_view_command(boost::shared_ptr<Document::DocumentDelta> delta)
{
    m_document->do_command(delta);
}

void EditorWindow::on_view_preview_command(boost::shared_ptr<Document::DocumentDelta> delta)
{
    m_document->preview_command(delta);
}

void EditorWindow::on_view_preview_cancel()
{
    m_document->cancel_preview();
}

bool EditorWindow::on_delete_event(GdkEventAny* event)
{
    // save keyboard shortcuts.
    DEBUG_MESSAGE("Saving accelerators to " << accel_filename);
    Gtk::AccelMap::save(accel_filename);
    if (check_clear_document())
    {
        return Gtk::Window::on_delete_event(event);
    }
    return true;
}

bool EditorWindow::check_clear_document()
{
    // it doesn't matter if the document is already clear.
    if (!m_document)
    {
        return true;
    }
    // we already have a document open, is it good to replace it?
    if (m_document->is_saved())
    {
        // saved, so allow it.
        return true;
    }
    // not saved, ask for confirmation
    std::stringstream ss;
    ss << m_document->get_changes_since_save();
    Glib::ustring changes_since_last_save_string(ss.str());
    Gtk::MessageDialog dialog(*this,
                              filename_set ? Glib::ustring("Save the changes to track \"") + Glib::ustring(filename) + Glib::ustring("\" before closing?") :"Save changes to unsaved track before closing?",
                              false,
                              Gtk::MESSAGE_QUESTION,
                              Gtk::BUTTONS_NONE,
                              true);
    dialog.set_secondary_text
    (
        Glib::ustring("If you don't save, your ")
      + changes_since_last_save_string
      + Glib::ustring(" changes since the last save will be permanetly lost.")
    );
    Gtk::Button * close_button = dialog.add_button("Close _without Saving", Gtk::RESPONSE_NO);
    Gtk::Image close_icon(Gtk::Stock::CLOSE, Gtk::ICON_SIZE_BUTTON);
    close_button->set_image(close_icon);
    close_icon.show();
    dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL)->get_image()->show();
    dialog.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_ACCEPT)->get_image()->show();
    dialog.set_default_response(Gtk::RESPONSE_ACCEPT);
    switch (dialog.run())
    {
        case Gtk::RESPONSE_CANCEL:
        case Gtk::RESPONSE_DELETE_EVENT: // I'll take that as a cancel.
            return false;
            break;
        case Gtk::RESPONSE_ACCEPT:
            // save before continuing
            on_action_file_save();
            break;
        default:
            // don't save and lose the changes.
            break;
    }
    // We've got confirmation now.
    return true;
}

void EditorWindow::load_theme(std::string filename)
{
    DEBUG_MESSAGE("Loading theme " << filename);
    theme_filename = filename;
    std::ifstream theme_file(theme_filename.c_str());
    m_theme = boost::shared_ptr<Track::Theme>(new Track::Theme(theme_file));
}

void EditorWindow::show_editor()
{
    m_ref_action_group->get_action("New")->set_sensitive(true);
    m_ref_action_group->get_action("Save")->set_sensitive(true);
    m_ref_action_group->get_action("SaveAs")->set_sensitive(true);
    m_ref_action_group->get_action("Test")->set_sensitive(true);
    m_ref_action_group->get_action("SelectAll")->set_sensitive(true);
    m_ref_action_group->get_action("Lighting")->set_sensitive(true);
    m_ref_action_group->get_action("ZoomIn")->set_sensitive(true);
    m_ref_action_group->get_action("ZoomOut")->set_sensitive(true);
    m_ref_action_group->get_action("ZoomToFit")->set_sensitive(true);
    m_ref_action_group->get_action("ShowTop")->set_sensitive(true);
    m_ref_action_group->get_action("ShowFront")->set_sensitive(true);
    m_ref_action_group->get_action("ShowSide")->set_sensitive(true);
    m_ref_action_group->get_action("NewWindow")->set_sensitive(true);
    m_ref_action_group->get_action("HelpGuide")->set_sensitive(true);
    
    m_new_form.hide();
    m_viewport_top.set_document(*m_document);
    m_viewport_top.show();
    m_guide.set_visible(m_guide_preference);
    
    // Move keyboard focus to the view controls on the toolbar.
    m_ref_ui_manager->get_widget("/ToolBar/ShowTop")->grab_focus();
}
