/** @file MenuItem.h
 *  @brief Declare the UI::MenuItem class.
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef MENUITEM_H_
#define MENUITEM_H_

#include <string>

namespace UI
{

/// an Item that appears on a menu.
class MenuItem
{
public:
    /** The appearence of the menu item. The state changes so that the user can
     * tell what is going on based on the appearnce of the menus.
     */
    enum State
    {
        /// Normal appearence when not highlighted
        S_NORMAL = 0,
        /** Highlighted to show that pressing the RT_MENU_SELECT button
         * activates this item.
         */
        S_PRELIGHT = 1,
        /** Actually selected.
         * Only shown when there is a submenu or some other excuse to leave the
         * menu displayed
         */
        S_ACTIVE = 2,
        /// Insensitive state, for when the item cannot be picked.
        S_DISABLED = 3
    };
    /** Construct with a label.
     * @param label The text to display on the menu item.
     */
    MenuItem(std::wstring label);
    virtual ~MenuItem();
    
    /** Render the menu item with OpenGL calls.
     * The item will be drawn from 64 pixels high centred around 0 vertically,
     * and from the 0 to the width set by set_width plus a constant horizontally.
     */
    virtual void draw();
    
    /// called when the item is picked by the user.
    virtual void activate();
    
    /// Change the state to display the item in. Defaults to S_NORMAL.
    void set_state(State state);
    
    /// Get the state the item will be displayed in.
    State get_state();
    
    /// Get the minimium width to make the menu to fit this item in.
    unsigned int get_min_width();
    
    /** Make the item this wide. Should be at least the value returned by
     * get_min_width().
     */
    void set_width(unsigned int width);
protected:
    std::wstring label;
    State state;
    float tex_coord_y_start;
    float tex_coord_y_end;
    unsigned int width;
};

}

#endif /*MENUITEM_H_*/
