/** @file Menu.cpp
 *  @brief Implement the UI::Menu class.
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "Menu.h"
#include "SubMenuItem.h"
#include <Debug.h>

#include <GL/gl.h>
#include <typeinfo>

namespace UI
{

Menu::Menu()
    :   max_width(0),
        selected_item(0),
        active_sub_menu(0),
        parent(0),
        quit(false)
{
}

Menu::~Menu()
{
}

void Menu::insert_item(MenuItem * item)
{
    items.push_back(item);
    // see if maximum size must be adjusted.
    if (item->get_min_width() > max_width)
    {
        max_width = item->get_min_width();
        DEBUG_MESSAGE("Setting new maximum width of " << max_width << ".");
        for (std::vector<MenuItem *>::iterator it = items.begin();
             it != items.end(); it++)
        {
            (*it)->set_width(max_width);
        }
    } else {
        // just set the size of the new item to the size of the largest of the
        // existing items
        items.back()->set_width(max_width);
    }
    
    // if it is the first item added, make it selected.
    if (items.size() == 1)
    {
        items.back()->set_state(MenuItem::S_PRELIGHT);
    }
}

void Menu::draw()
{
    // are we drawing a sub menu?
    if (active_sub_menu)
    {
        // draw the sub menu, then shift across and draw ourselves.
        active_sub_menu->draw();
    }
    glPushMatrix();
        // draw with whatever is at the top centred around (0,0)
        glTranslatef(-float(get_total_width()),
                     float (items.size() * 32),0.0);
        for (std::vector<MenuItem *>::iterator it = items.begin();
             it != items.end(); it++)
        {
            (*it)->draw();
            glTranslatef(0.0, -64.0, 0.0);
        }
    glPopMatrix(); 
}
    
void Menu::take_input(Engine::InputReport & report)
{
    if (active_sub_menu)
    {
        active_sub_menu->take_input(report);
        return;
    }
    switch (report.get_report_type())
    {
        case Engine::InputReport::RT_MENU_UP:
            items[selected_item]->set_state(MenuItem::S_NORMAL);
            if (selected_item == 0)
            {
                selected_item = items.size() - 1;
            } else {
                selected_item--;
            }
            items[selected_item]->set_state(MenuItem::S_PRELIGHT);
            break;
        case Engine::InputReport::RT_MENU_DOWN:
            items[selected_item]->set_state(MenuItem::S_NORMAL);
            if (selected_item == items.size() - 1)
            {
                selected_item = 0;
            } else {
                selected_item++;
            }
            items[selected_item]->set_state(MenuItem::S_PRELIGHT);
            break;
        case Engine::InputReport::RT_MENU_SELECT:
            items[selected_item]->activate();
            // check if it is a submenu item.
            {
                SubMenuItem * item = dynamic_cast<SubMenuItem *>(items[selected_item]);
                if (item)
                {
                    // hand over control
                    DEBUG_MESSAGE("Switching to submenu.");
                    active_sub_menu = item->get_menu();
                    active_sub_menu->parent = this;
                } else {
                    DEBUG_MESSAGE("Activated a (non SubMenuItem) MenuItem.");
                    // not a submenu item. Close menu?
                }
            }
            break;
        case Engine::InputReport::RT_MENU_BACK:
            // if we are a submenu, give control back to parent
            if (parent)
            {
                parent->active_sub_menu = 0;
                parent->items[parent->selected_item]->set_state(MenuItem::S_PRELIGHT);
                parent = 0;
            } else {
                quit = true;
            }
            break;
        default:
            // don't respond to other events.
            break;
    }
}

bool Menu::want_to_quit()
{
    return quit;
}

unsigned int Menu::get_total_width()
{
    if (active_sub_menu)
    {
        return active_sub_menu->get_total_width() + max_width;
    } else {
        return max_width / 2;
    } 
}

}
