/** @file racer/UI/CarSelectScene.h
 *  @brief Declare the UI::CarSelectScene class.
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef UI_CAR_SELECT_SCENE_H
#define UI_CAR_SELECT_SCENE_H

#include "../Engine/Scene.h"
#include "../Engine/InputHandler.h"
#include <libtrack/Texture.h>
#include <vector>
#include <utility>
#include <map>

#include <LinearMath/btVector3.h>

namespace UI
{

/** Scene for choosing controllers and cars.
 * Each InputDevice avaliable is either assigned a car model or left unused.
 * When at least one InputDevice is not unused, the game can be started with
 * a RT_MENU_SELECT event from a used InputDevice. RT_MENU_BACK from any
 * InputDevice goes back without starting a game.
 */
class CarSelectScene
    :   public Engine::Scene
{
    public:
        CarSelectScene();
        bool get_canceled() const;
        std::vector<std::pair<Engine::InputHandler::iterator, unsigned int> > get_choice() const;
        virtual void take_input(Engine::InputReport &report);
        virtual void update_logic(unsigned int milliseconds_elapsed);
        virtual void draw();
        virtual void do_sound();
    protected:
        /// True if the scene was canceled.
        bool m_canceled;
        /// Texture for arrow that indicates selected car
        Track::Texture * m_car_select_texture;
        /// Texture for border for a car
        Track::Texture * m_car_border_texture;
        /// Texture for the background of the unused devices section.
        Track::Texture * m_unused_background_texture;
        /// The textures for the car previews.
        Track::Texture * m_car_preview_texture[2];
        
        /** Which InputDevices each car has.
         * m_selection[0] is the unused option, m_selection[1] is the first
         * car, m_selection[2] is the second car.
         */
        std::vector<std::set<Engine::InputHandler::iterator> > m_selection;
        /// colours for each device
        std::map<Engine::InputHandler::iterator, btVector3> m_device_colours;
        
        /** Draw the selection arrows for a given car.
         * @param choice 0 for unused cars, otherwise car number + 1.
         */
        void draw_selections(std::size_t choice);
        /** Removes a device from all cars and the unused slot in m_selection.
         */
        void remove_device(Engine::InputHandler::iterator device);
};

} // namespace UI

#endif // UI_CAR_SELECT_SCENE_H
