/** @file racer/UI/CarSelectScene.cpp
 *  @brief Implement the UI::CarSelectScene class.
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "CarSelectScene.h"
#include "BasicFonts.h"
#include "../Graphics/Window.h"
#include "../Engine/ResourceHandler.h"
#include "../MainLoop.h"
#include <Debug.h>

#include <algorithm>
#include <cmath>

/// Draw some text using the big font in white with a black outline.
void bordered_text(const char * message)
{
    glPushMatrix();
        // border
        glColor4f(0.0, 0.0, 0.0, 1.0);
        glPushMatrix();
            glTranslatef(-1, -1, 0);
            UI::BasicFonts::get_instance().big_font.Render(message);
        glPopMatrix();
        glPushMatrix();
            glTranslatef(1, -1, 0);
            UI::BasicFonts::get_instance().big_font.Render(message);
        glPopMatrix();
        glPushMatrix();
            glTranslatef(1, 1, 0);
            UI::BasicFonts::get_instance().big_font.Render(message);
        glPopMatrix();
        glPushMatrix();
            glTranslatef(-1, 1, 0);
            UI::BasicFonts::get_instance().big_font.Render(message);
        glPopMatrix();
        // main text.
        glColor4f(1.0, 1.0, 1.0, 1.0);
        glPushMatrix();
            UI::BasicFonts::get_instance().big_font.Render(message);
        glPopMatrix();
    glPopMatrix();
}

namespace UI
{

const unsigned int number_of_cars = 2;

CarSelectScene::CarSelectScene()
    :   m_canceled(false)
{
    // Load shared textures if not already loaded.
    /// stores shared textures, indexed by a string name
    typedef Engine::ResourceHandler<Track::Texture, std::string, std::string> TextureStore;
    TextureStore & tex_store = TextureStore::get_instance();
    #define tex_load(bind_name, file_name, variable)\
                tex_store.check_load(bind_name, file_name);\
                variable = tex_store.get(bind_name);\
                variable->make_cache()
    tex_load("select.car.ui", "data/ui/car_select.png", m_car_select_texture);
    tex_load("border.car.ui", "data/ui/car_border.png", m_car_border_texture);
    tex_load("unused.bg.car.ui", "data/ui/car_unused_background.png", m_unused_background_texture);
    tex_load("0.pre.car.ui", "data/cars/1/preview.png", m_car_preview_texture[0]);
    tex_load("1.pre.car.ui", "data/cars/2/preview.png", m_car_preview_texture[1]);
    #undef tex_load
    
    // Record all InputDevices, which default to unused.
    m_selection.resize(number_of_cars + 1);
    btScalar a = 0;
    const btScalar diff = 6.283185308 /
        btScalar(Engine::InputHandler::get_instance().get_number_of_devices());
    const btScalar t = 2.094395103;
    const btScalar tt = t * 2.0;
    for (Engine::InputHandler::iterator it = Engine::InputHandler::get_instance().begin();
         it != Engine::InputHandler::get_instance().end();
         it++)
    {
        a += diff;
        m_selection[0].insert(it);
        DEBUG_MESSAGE("Using hue " << a);
        m_device_colours[it] = 
                btVector3(std::sin(a) * 0.5 + 0.5,
                          std::sin(a+t) * 0.5 + 0.5,
                          std::sin(a+tt) * 0.5 + 0.5);
        DEBUG_MESSAGE("Colour " << m_device_colours[it].x() << ", "
                                << m_device_colours[it].y() << ", "
                                << m_device_colours[it].z());
    }
}

bool CarSelectScene::get_canceled() const
{
    return m_canceled;
}

std::vector<std::pair<Engine::InputHandler::iterator, unsigned int> >
CarSelectScene::get_choice() const
{
    // copy device list except for the devices that aren't playing.
    std::vector<std::pair<Engine::InputHandler::iterator, unsigned int> > result;
    for (unsigned int i = 0; i < number_of_cars; i++)
    {
        for (std::set<Engine::InputHandler::iterator>::iterator it = m_selection[i+1].begin();
             it != m_selection[i+1].end();
             it++)
        {
            result.push_back(std::make_pair(*it, i));
        }
    }
    return result;
}

void CarSelectScene::take_input(Engine::InputReport &report)
{
    switch (report.get_report_type())
    {
        case Engine::InputReport::RT_MENU_DOWN:
            // makes device unused.
            remove_device(report.get_input_device());
            m_selection[0].insert(report.get_input_device());
            break;
        case Engine::InputReport::RT_MENU_UP:
        case Engine::InputReport::RT_MENU_LEFT:
            remove_device(report.get_input_device());
            m_selection[1].insert(report.get_input_device());
            break;
        case Engine::InputReport::RT_MENU_RIGHT:
            remove_device(report.get_input_device());
            m_selection[2].insert(report.get_input_device());
            break;
        case Engine::InputReport::RT_MENU_BACK:
            m_canceled = true;
            main_loop->pop_scene();
            break;
        case Engine::InputReport::RT_MENU_SELECT:
            // At least one InputDevice needs a car assigned to continue.
            if (m_selection[0].size() < Engine::InputHandler::get_instance().get_number_of_devices())
            {
                main_loop->pop_scene();
            }
            break;
        default:
            // Other inputs ignored.
            break;
    }
}

void CarSelectScene::update_logic(unsigned int milliseconds_elapsed)
{
}

void CarSelectScene::draw()
{
    glClear(GL_COLOR_BUFFER_BIT);
    
    Graphics::Window::get_instance().set_ortho_projection();
    
    glDisable(GL_DEPTH_TEST);
    glEnable(GL_TEXTURE_2D);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glColor4f(1.0, 1.0, 1.0, 1.0); 
    glLoadIdentity();
    glTranslatef(Graphics::Window::get_instance().get_width() / 2,
                 Graphics::Window::get_instance().get_height() / 2,
                 0.0 );
    for (unsigned int i = 0; i < number_of_cars; i++)
    {
        glPushMatrix();
            glTranslatef((signed(i) - 1) * 320 + 160, 34, 0);
            m_car_border_texture->bind();
#if 0 //defined(HAVE_GLES)
            glBegin(GL_QUADS);
                glTexCoord2i(0, 1); glVertex2i(-128, -128);
                glTexCoord2i(1, 1); glVertex2i(127, -128);
                glTexCoord2i(1, 0); glVertex2i(127, 127);
                glTexCoord2i(0, 0); glVertex2i(-128, 127);
            glEnd();
#else
	GLshort vtx1[] = {	-128, -128,	127, -128,	127, 127,	-128, 127	};
	GLshort tex1[] = {	0, 1,		1, 1,		1, 0,		0, 0 	};

	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glVertexPointer(2, GL_SHORT, 0, vtx1);
	glTexCoordPointer(2, GL_SHORT, 0, tex1);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
#endif
            m_car_preview_texture[i]->bind();
            // previews have premultiplied alpha
            glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
#if 0 //defined(HAVE_GLES)
            glBegin(GL_QUADS);
                glTexCoord2i(0, 1); glVertex2i(-128, -128);
                glTexCoord2i(1, 1); glVertex2i(127, -128);
                glTexCoord2i(1, 0); glVertex2i(127, 127);
                glTexCoord2i(0, 0); glVertex2i(-128, 127);
            glEnd();
#else
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

#endif
            glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
            glTranslatef(0, 187, 0);
            draw_selections(i + 1);
        glPopMatrix();
    }
    
    
    // Text for unused cars:
    const char message[] = "Unused controllers";
    const int width_o = BasicFonts::get_instance().big_font.Advance(message) / 2;
    glPushMatrix();
    glTranslatef(0, -208, 0);
    // background
    m_unused_background_texture->bind();
#if 0 //defined(HAVE_GLES)
    glBegin(GL_QUADS);
        glColor4f(1.0, 1.0, 1.0, 0.0);
        glTexCoord2f(0.0, 0.0); glVertex2i(-1000, 37);
        glTexCoord2f(0.0, 1.0); glVertex2i(-1000, -26);
        glColor4f(1.0, 1.0, 1.0, 1.0);
        glTexCoord2f(125.0, 1.0); glVertex2i(0, -26);
        glTexCoord2f(125.0, 0.0); glVertex2i(0, 37);
        
        glTexCoord2f(0.0, 1.0); glVertex2i(0, -26);
        glTexCoord2f(0.0, 0.0); glVertex2i(0, 37);
        glColor4f(1.0, 1.0, 1.0, 0.0);
        glTexCoord2f(125.0, 0.0); glVertex2i(1000, 37);
        glTexCoord2f(125.0, 1.0); glVertex2i(1000, -26);
    glEnd();
#else
	GLshort vtx2[] = {	-1000, 37,	-1000, -26,	0, -26,		0, 37	};
	GLshort vtx3[] = {	0, -26,		0, 37,		1000, 37,	1000, -26	};
	GLfloat tex2[] = {	0.0, 0.0,	0.0, 1.0,	125.0, 1.0,	125.0, 0.0	};
	GLfloat tex3[] = {	0.0, 1.0,	0.0, 0.0,	125.0, 0.0,	125.0, 1.0	};
	GLfloat col2[] = {	1.0, 1.0, 1.0, 0.0,	1.0, 1.0, 1.0, 0.0,	1.0, 1.0, 1.0, 1.0,	1.0, 1.0, 1.0, 1.0	};
	GLfloat col3[] = {	1.0, 1.0, 1.0, 1.0,	1.0, 1.0, 1.0, 1.0,	1.0, 1.0, 1.0, 0.0,	1.0, 1.0, 1.0, 0.0	};

	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);

	glVertexPointer(2, GL_SHORT, 0, vtx2);
	glTexCoordPointer(2, GL_FLOAT, 0, tex2);
	glColorPointer(4, GL_FLOAT, 0, col2);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);

	glVertexPointer(2, GL_SHORT, 0, vtx3);
	glTexCoordPointer(2, GL_FLOAT, 0, tex3);
	glColorPointer(4, GL_FLOAT, 0, col3);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);

	glDisableClientState(GL_COLOR_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glColor4f(1.0, 1.0, 1.0, 1.0);
    
    // markers for unused devices
    glPushMatrix();
        glTranslatef(0, 95, 0);
        draw_selections(0);
    glPopMatrix();
    // text
    glTranslatef(-width_o, 0, 0);
    bordered_text(message);
    glPopMatrix();
    
    // Heading at the top
    const char title[] = "Select vehicles";
    const int title_width_o = BasicFonts::get_instance().big_font.Advance(title) / 2;
    // background: dark rectangle.
    glDisable(GL_TEXTURE_2D);
    glColor4f(0.0, 0.0, 0.0, 0.5);
    glTranslatef(0, 208, 0);
#if 0 //defined(HAVE_GLES)
    glBegin(GL_QUADS);
        glVertex2i(-title_width_o-4, 23);
        glVertex2i(-title_width_o-4, -6);
        glVertex2i(title_width_o+4, -6);
        glVertex2i(title_width_o+4, 23);
    glEnd();
    glColor4f(0.0, 0.0, 0.0, 1.0);
    glBegin(GL_LINE_LOOP);
        glVertex2i(-title_width_o-4, 23);
        glVertex2i(-title_width_o-4, -6);
        glVertex2i(title_width_o+4, -6);
        glVertex2i(title_width_o+4, 23);
    glEnd();
#else
	GLshort vtx4[] = { -title_width_o-4, 23,	-title_width_o-4, -6,	title_width_o+4, -6,	title_width_o+4, 23	};
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(2, GL_SHORT, 0, vtx4);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);

	glColor4f(0.0, 0.0, 0.0, 1.0);
	glDrawArrays(GL_LINE_LOOP,0,4);
	glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glEnable(GL_TEXTURE_2D);
    // Text
    glTranslatef(-title_width_o, 0, 0);
    bordered_text(title);
    glDisable(GL_BLEND);
}

void CarSelectScene::draw_selections(std::size_t choice)
{
    std::size_t size = m_selection[choice].size();
    if (size == 0) return;
    int offset = 256 / size;
    m_car_select_texture->bind();
    glTranslatef(offset / 2 - 128, 0, 0);
    glPushMatrix();
        for (std::set<Engine::InputHandler::iterator>::iterator it = m_selection[choice].begin();
             it != m_selection[choice].end();
             it++)
        {
#if 0 //defined(HAVE_GLES)
            glBegin(GL_QUADS);
                glTexCoord2f(0.5 + 1.0 / 128.0, 0.0); glVertex2i(-64, 63);
                glTexCoord2f(1.0, 0.0); glVertex2i(63, 63);
                glTexCoord2f(1.0, 1.0); glVertex2i(63, -64);
                glTexCoord2f(0.5 + 1.0 / 128.0, 1.0); glVertex2i(-64, -64);
            glEnd();
#else
	GLshort vtx1[] = {	-64, 63,		63, 63,		63, -64,	-64, -64	};
	GLfloat tex1[] = {	0.5 + 1.0 / 128.0, 0.0,	1.0, 0.0,	1.0, 1.0,	0.5 + 1.0 / 128.0, 1.0	};
	GLfloat tex2[] = {	0.0, 0.0,		0.5, 0.0,	0.5, 1.0,	0.0, 1.0	};

	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	
	glVertexPointer(2, GL_SHORT, 0, vtx1);
	glTexCoordPointer(2, GL_FLOAT, 0, tex1);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
#endif
            const btVector3 & c = m_device_colours[*it];
            glColor4f(c.x(), c.y(), c.z(), 1.0);
            glBlendFunc(GL_SRC_COLOR, GL_ONE);
#if 0 //defined(HAVE_GLES)
            glBegin(GL_QUADS);
                glTexCoord2f(0.0, 0.0); glVertex2i(-64, 63);
                glTexCoord2f(0.5, 0.0); glVertex2i(63, 63);
                glTexCoord2f(0.5, 1.0); glVertex2i(63, -64);
                glTexCoord2f(0.0, 1.0); glVertex2i(-64, -64);
            glEnd();
#else
	glTexCoordPointer(2, GL_FLOAT, 0, tex2);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);

	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);
#endif
            glColor4f(1.0, 1.0, 1.0, 1.0);
            glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
            glTranslatef(offset, 0, 0);
        }
    glPopMatrix();
}

void CarSelectScene::remove_device(Engine::InputHandler::iterator device)
{
    for (std::vector<std::set<Engine::InputHandler::iterator> >::iterator it = m_selection.begin();
         it != m_selection.end();
         it++)
    {
        it->erase(device);
    }
}

void CarSelectScene::do_sound()
{
}

} // namespace UI
