/** @file RacerApp.cpp
 *  @brief Implement the RacerApp class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include <iostream>
#include <fstream>
#include <string>

#include <SDL.h>

#include "RacerApp.h"
#include "UI/TitleScene.h"
#include "Engine/GameScene.h"
#include "Engine/InputHandler.h"
#include "Engine/AITrainer.h"
#include "Graphics/Window.h"
#ifndef HAVE_GLES
#include <config.h>
#else
#include "Graphics/eglport.h"
#endif
#include <Debug.h>

#include <libtrack/DataCheck.h>

#include <glibmm/init.h>


RacerApp::RacerApp(int argc, char ** argv)
    :   title_scene(0)
    ,   main_loop(0)
    ,   replay(0)
{
    Glib::init();
    try
    {
        Track::DataCheck();
    }
    catch (Track::DataCheck::FileMissingError e)
    {
        std::cerr << "Cannot find data file: " << e.filename << std::endl;
        std::cerr << "Please run from the directory all the data files accessible." << std::endl;
        return;
    }
    
    // usage info
    if (argc == 2)
    {
        if (argv[1] == std::string("-h") || argv[1] == std::string("--help"))
        {
            show_usage();
            return;
        }
    }
    Graphics::Window::get_instance().set_size(640, 480);
    parse_arguments(argc, argv);
    
    if (!title_scene)
    {
        title_scene = new UI::TitleScene();
    }
    
    main_loop = new MainLoop();
    main_loop->push_scene(*title_scene);
    // shutdown
#if defined(HAVE_GLES)
	EGL_Destroy();
#endif
    SDL_Quit();
}

RacerApp::~RacerApp()
{
    if(main_loop) delete main_loop;
    // A replay deletes its own scene when destroyed.
    if (replay) delete replay;
    else if (title_scene) delete title_scene;
}

void RacerApp::parse_arguments(int argc, char ** argv)
{
    /// @todo show usage info when given malformed arguments.
    for (int i = 0; i < argc; i++)
    {
        DEBUG_MESSAGE("Argument " << i << " is " << argv[i]);
    }
    if (argc > 1 && argv[1][0] == 'r' && argv[1][1] == '\0')
    {
        // Load a replay.
        DEBUG_MESSAGE("replay");
        
        if (argc == 2)
        {
            // no file given, assume standard input.
            DEBUG_MESSAGE("    Replaying from standard input.");
            replay = new Engine::ReplayReader(std::cin);
            
        }
        else
        {
            DEBUG_MESSAGE("    Replaying from a file.");
            std::ifstream replay_file(argv[2]);
            if (replay_file.fail())
            {
                // cannot load file
                std::cerr << "Unable to access the replay file (" << argv[2] << "). Perhaps it doesn't exist?"
                          << std::endl;
                exit(EXIT_FAILURE);
            }
            replay = new Engine::ReplayReader(replay_file);
        }
        title_scene = replay->get_scene();
    }
    else if (argc >= 2 && argv[1][0] == 't' && argv[1][1] == '\0')
    {
        DEBUG_MESSAGE("Train");
        std::cout << "Training AI on track " << argv[2] << "\n";
        Engine::AITrainer *spork = new Engine::AITrainer(std::string(argv[2]));
        delete spork;
    }
    else
    {
        if (argc == 2)
        {
            // use a game scene
            std::string track = argv[1];
            // create a player for each controller.
            DEBUG_MESSAGE("Using one player per avaliable controller.");
            std::vector<std::pair<Engine::InputHandler::iterator, unsigned int> > input_devices;
            input_devices.reserve(Engine::InputHandler::get_instance()
                                                      .get_number_of_devices());
            for (Engine::InputHandler::iterator it = Engine::InputHandler::get_instance().begin();
                 it != Engine::InputHandler::get_instance().end();
                 it++)
            {
                input_devices.push_back(std::make_pair(it, 0));
            }
            /// @todo allow disused controllers by commandline arguments.
            /// @todo Allow picking the number of computer players by command line arguments.
            /// @todo give it a main_loop or callback.
            Engine::LoadScene<Engine::GameScene, RacerApp> * scene;
            scene = new Engine::LoadScene<Engine::GameScene, RacerApp>(input_devices, track);
            scene->set_done_notifier(this);
            title_scene = scene;
        }
    }
}

void RacerApp::operator ()(Engine::GameScene * new_scene)
{
    main_loop->push_scene(*new_scene);
}

void RacerApp::show_usage()
{
    std::cout << "Usage:" << std::endl
              << "racer         start the game showing the menu." << std::endl
              << "racer track   play track multiplayer using all avaliable controllers." << std::endl
              << "racer r file  Show the replay saved in file." << std::endl
              << "racer r       Show replay passed in by standard input." << std::endl
              << "racer t track Train the AI on the track." << std::endl
              << "racer --help  Show this help and exit." << std::endl
              << "racer -h      Show this help and exit." << std::endl
              << std::endl
              << "A replay is automatically saved to a file called racer_last_replay, which is" << std::endl
              << "stored in your configuration directory (usually ~/.config)." << std::endl
              << "File paths must either be absolute, or relative to racer's path, for example" << std::endl
              << "'racer data/tracks/0' should work anywhere once racer is installed." << std::endl
              << std::endl;
//    std::cout << PACKAGE_NAME " version " PACKAGE_VERSION << std::endl
//              << "Built time: " BUILD_TIME "." << std::endl;
    #ifndef NDEBUG
        std::cout << "This is a debug build." << std::endl;
    #endif
}
