/** @file MainLoop.h
 *  @brief Declare the MainLoop class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef MAINLOOP_H_
#define MAINLOOP_H_

#include "Engine/Scene.h"

/** Runs the main loop.
 * Scene can take over for a while with push_scene().
 * Once they are done, they should call pop_scene().
 * Scenes are recursive, pop_scene() will return to the caller of
 * push_scene(), which is often another scene.
 */ 
class MainLoop
{
    public:
        MainLoop();
        
        virtual ~MainLoop();
        /** Stop the current scene.
         * When there are no more scenes, the main loop ends.
         * Must be called only once per scene.
         */
        void pop_scene();
        
        /** Run a scene.
         * The scene should call pop_scene() when it wants to quit.
         * push_scene() does not return until this happens.
         */
        void push_scene(Engine::Scene & scene);
    private:
        /// Run the current scene until it quits.
        void run(Engine::Scene & scene);
        /// Record input
        void process_events();
        /// Update game state (do logic / physics)
        void do_logic(Engine::Scene & scene);
        /// Update the display
        void redraw(Engine::Scene & scene);
        /// Update the sound
        void update_sound(Engine::Scene & scene);
        
        unsigned int milliseconds;
        /// true iff current scene should end.
        bool done;
        /// true iff all scenes should end.
        bool quit;
};

#endif /*MAINLOOP_H_*/
