/** @file MainLoop.cpp
 *  @brief Implement the MainLoop object.  
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "MainLoop.h"

#include <SDL.h>
#include "Engine/InputHandler.h"
#include "Debug.h"
#include "Graphics/Window.h"

#include <climits>
 
MainLoop::MainLoop()
    :   done(false)
    ,   quit(false)
{
}

MainLoop::~MainLoop()
{
}

void MainLoop::run(Engine::Scene & scene)
{
    while (!(done || quit))
    {
        Engine::InputHandler::get_instance().set_scene(scene);
        redraw(scene);
        update_sound(scene);
        process_events();
        do_logic(scene);
    }
    done = false;
}

void MainLoop::pop_scene()
{
    done = true;
}

void MainLoop::push_scene(Engine::Scene & scene_in)
{
    scene_in.attach_main_loop(*this);
    // don't let the time spent loading the scene cause a large amount of time
    // to pass at the start.
    milliseconds = SDL_GetTicks();
    run(scene_in);
}

void MainLoop::process_events()
{
    // Process events. We handle all quit and resize events.
    SDL_Event event;
    SDL_PumpEvents();
    while (SDL_PeepEvents(&event, 1, SDL_GETEVENT,
               SDL_EVENTMASK(SDL_QUIT) | SDL_EVENTMASK(SDL_VIDEORESIZE)) > 0)
    {
        switch (event.type)
        {
            case SDL_QUIT:
                // exit
                quit = true;
                break;
            case SDL_VIDEORESIZE:
                // tell the window.
                Graphics::Window::get_instance().set_size(event.resize.w, event.resize.h);
                break;
        }
    }
    // Get the input handler to poll to process the other events.
    Engine::InputHandler::get_instance().poll();
    // Any remaining events can be ignored. We can't pump events, as this will
    // cause us to potentially ignore good events. SDLPollEvents implicitly
    // pumps the event queue, so we have to peep at all events.
    // If we don't remove all the ignored events from the queue, they eventually
    // clog up the event queue, and we cannot get to the events we want.
    while (SDL_PeepEvents(&event, 1, SDL_GETEVENT, SDL_ALLEVENTS)); 
    
}

void MainLoop::do_logic(Engine::Scene & scene)
{
    // calculate elapsed time
    unsigned int new_milliseconds = SDL_GetTicks();
    unsigned int elapsed_milliseconds = new_milliseconds - milliseconds;
    if (elapsed_milliseconds > UINT_MAX / 2)
    {
        // assume it has wrapped.
        elapsed_milliseconds = UINT_MAX - elapsed_milliseconds;
    }
    scene.update_logic(elapsed_milliseconds);
    milliseconds = new_milliseconds;
}

void MainLoop::redraw(Engine::Scene & scene)
{
    scene.draw();
    Graphics::Window::get_instance().swap_buffers();
}

void MainLoop::update_sound(Engine::Scene & scene)
{
    scene.do_sound();
}
