/**
*
* eglport.c/.h
* Copyright (C) 2011 Scott Smith
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program. If not, see <http://www.gnu.org/licenses/>.
*
*/
#ifdef HAVE_GLES

#include "eglport.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
/* Pandora VSync */
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/fb.h>


#ifndef FBIO_WAITFORVSYNC
#define FBIO_WAITFORVSYNC _IOW('F', 0x20, __u32)
#endif
int fbdev = -1;
/* Pandora VSync End */

#define EGLNativeWindowType NativeWindowType
#define EGLNativeDisplayType NativeDisplayType

EGLDisplay g_eglDisplay = 0;
EGLConfig g_eglConfig = 0;
EGLContext g_eglContext = 0;
EGLSurface g_eglSurface = 0;

#define g_totalConfigsIn 20
int g_totalConfigsFound = 0;
EGLConfig g_allConfigs[g_totalConfigsIn];
Display *g_x11Display = NULL;


/*======================================================
 * Kill off any opengl specific details
  ====================================================*/
void EGL_Destroy()
{
    if( g_eglSurface || g_eglContext || g_eglDisplay )
    {
        eglMakeCurrent(g_eglDisplay, NULL, NULL, EGL_NO_CONTEXT);
        eglDestroyContext(g_eglDisplay, g_eglContext);
        eglDestroySurface(g_eglDisplay, g_eglSurface);
        eglTerminate(g_eglDisplay);
    }

    g_eglSurface = 0;
    g_eglContext = 0;
    g_eglDisplay = 0;

	if (g_x11Display)
		XCloseDisplay(g_x11Display);

    g_x11Display = NULL;

    printf( "EGL Closed\n");

    /* Pandora VSync */
    close(fbdev);
    fbdev = -1;
    /* Pandora VSync End */
}

/*===========================================================
Setup EGL context and surface
===========================================================*/
int EGL_Init(int fsaa)
{
    FindAppropriateEGLConfigs(fsaa);

    int configIndex = 0;

	printf( "Config %d\n", configIndex );

	if (!ConfigureEGL(g_allConfigs[configIndex]))
	{
		TestEGLError();
		fprintf(stderr, "ERROR: Unable to initialise EGL. See previous error.\n");
		return 1;
	}

    /* Pandora VSync */
    fbdev = open ("/dev/fb0", O_RDONLY /* O_RDWR */ );
    if ( fbdev < 0 ) {
      fprintf ( stderr, "Couldn't open /dev/fb0 for vsync\n" );
    }
    /* Pandora VSync End */

    return 0;
}

/*===========================================================
Swap EGL buffers and update the display
===========================================================*/
void EGL_SwapBuffers( void )
{
    /* Pandora VSync */
    if ( fbdev >= 0 ) {
        int arg = 0;
        ioctl( fbdev, FBIO_WAITFORVSYNC, &arg );
    }
    /* Pandora VSync End */
	eglSwapBuffers(g_eglDisplay, g_eglSurface);
}


/*========================================================
 *  Init base EGL
 * ======================================================*/
int EGL_Open( void )
{
    // use EGL to initialise GLES
    printf( "EGL Open display\n" );
    g_x11Display = XOpenDisplay(NULL);

    if (!g_x11Display)
    {
        fprintf(stderr, "ERROR: unable to get display!\n");
        return 0;
    }

    printf( "EGL Get display\n" );
    g_eglDisplay = eglGetDisplay((EGLNativeDisplayType)g_x11Display);

    if (g_eglDisplay == EGL_NO_DISPLAY)
    {
        TestEGLError();
        fprintf(stderr, "ERROR: Unable to initialise EGL display.\n");
        return 0;
    }

    // Initialise egl
    printf( "EGL Init\n" );
    if (!eglInitialize(g_eglDisplay, NULL, NULL))
    {
        TestEGLError();
        fprintf(stderr, "ERROR: Unable to initialise EGL display.\n");
        return 0;
    }

    return 1;
}

/*===========================================================
Initialise OpenGL settings
===========================================================*/
int ConfigureEGL(EGLConfig config)
{
    // Cleanup in case of a reset
    if( g_eglSurface || g_eglContext || g_eglDisplay )
    {
        eglMakeCurrent(g_eglDisplay, NULL, NULL, EGL_NO_CONTEXT);
        eglDestroyContext(g_eglDisplay, g_eglContext);
        eglDestroySurface(g_eglDisplay, g_eglSurface);
    }

    // Bind GLES and create the context
    printf( "EGL Bind\n" );
    eglBindAPI(EGL_OPENGL_ES_API);
	if (!TestEGLError() )
	{
		return 0;
	}

    printf( "EGL Create Context\n" );
    g_eglContext = eglCreateContext(g_eglDisplay, config, NULL, NULL);
    if (g_eglContext == EGL_NO_CONTEXT)
    {
        TestEGLError();
        fprintf(stderr, "ERROR: Unable to create GLES context!\n");
        return 0;
    }

    // Get the SDL window handle
    SDL_SysWMinfo sysInfo; //Will hold our Window information
    SDL_VERSION(&sysInfo.version); //Set SDL version
    if(SDL_GetWMInfo(&sysInfo) <= 0)
    {
        TestEGLError();
        fprintf( stderr, "ERROR: Unable to get window handle\n");
        return 0;
    }

    printf( "EGL Create window surface\n" );
    g_eglSurface = eglCreateWindowSurface(g_eglDisplay, config, (EGLNativeWindowType)sysInfo.info.x11.window, 0);

    if ( g_eglSurface == EGL_NO_SURFACE)
    {
        TestEGLError();
        fprintf(stderr, "ERROR: Unable to create EGL surface!\n");
        return 0;
    }

    printf( "EGL Make Current\n" );
    if (eglMakeCurrent(g_eglDisplay,  g_eglSurface,  g_eglSurface, g_eglContext) == EGL_FALSE)
    {
        TestEGLError();
        fprintf(stderr, "ERROR: Unable to make GLES context current\n");
        return 0;
    }

    printf( "EGL Done\n" );
    return 1;
}

/*=======================================================
* Detect available video resolutions
=======================================================*/
int FindAppropriateEGLConfigs(int fsaa)
{
    static const EGLint s_configAttribs[] =
    {
          EGL_RED_SIZE,     5,
          EGL_GREEN_SIZE,   6,
          EGL_BLUE_SIZE,    5,
          EGL_DEPTH_SIZE,       16,
          EGL_SURFACE_TYPE,         EGL_WINDOW_BIT,
          EGL_RENDERABLE_TYPE,      EGL_OPENGL_ES_BIT,
	  EGL_SAMPLE_BUFFERS,	1,
	  EGL_SAMPLES,		4,
          EGL_NONE
    };
    static const EGLint s_configAttribsFSAA[] =
    {
          EGL_RED_SIZE,     5,
          EGL_GREEN_SIZE,   6,
          EGL_BLUE_SIZE,    5,
          EGL_DEPTH_SIZE,       16,
          EGL_SURFACE_TYPE,         EGL_WINDOW_BIT,
          EGL_RENDERABLE_TYPE,      EGL_OPENGL_ES_BIT,
          EGL_SAMPLE_BUFFERS, 1,
          EGL_SAMPLES, fsaa,
          EGL_NONE
    };

    GLboolean got_config;
    if ((fsaa == 2) || (fsaa == 4))
    {
        got_config = eglChooseConfig(g_eglDisplay, s_configAttribsFSAA,
                    g_allConfigs, g_totalConfigsIn, &g_totalConfigsFound);
    } else {
        got_config = eglChooseConfig(g_eglDisplay, s_configAttribs,
                    g_allConfigs, g_totalConfigsIn, &g_totalConfigsFound);
    }
    if (got_config != EGL_TRUE || g_totalConfigsFound == 0)
    {
        TestEGLError();
        fprintf(stderr, "ERROR: Unable to query for available configs.\n");
        return 0;
    }
    fprintf(stderr, "Found %d available configs\n", g_totalConfigsFound);
    return 1;
}

int TestEGLError( void )
{
	EGLint iErr = eglGetError();
	while (iErr != EGL_SUCCESS)
	{
		printf("EGL failed (%d).\n", iErr);
		return 0;
	}

	return 1;
}


/*
This is an limited implementation based on this game need.
Only these flags are managed :
GL_ALPHA_TEST flag ; glEnable(GL_ALPHA_TEST); and glDisable(GL_ALPHA_TEST);
GL_BLEND flag
GL_CULL_FACE flag
GL_DEPTH_TEST flag
GL_LINE_SMOOTH flag
GL_MULTISAMPLE flag
GL_POLYGON_OFFSET_FILL flag
GL_TEXTURE_2D flag
GL_LIGHTING flag
GL_LIGHTi where 0 <= i < GL_MAX_LIGHTS
*/
#define GL_ENABLE_BIT 1
typedef struct node
{
    unsigned int data;
    struct node * next;
} node;

node * stackTop = NULL;

#define PUSHFLAG(a)                     \
        glGetBooleanv(a, &b);           \
        if (b == GL_TRUE) n->data++;    \
        n->data = n->data<<1;
#define POPFLAG(a)                      \
        c->data = c->data>>1;           \
        if (c->data & 1 > 0)            \
                glEnable(a);            \
        else                            \
                glDisable(a);

void glPushAttrib(int t)
{
        static node     *n;
        GLboolean       b;
        GLint           ln;
        n = (node*)calloc(1, sizeof(node));
        glGetIntegerv(GL_MAX_LIGHTS, &ln);      // 8 for current SGX

        n->data = 0;
        PUSHFLAG(GL_ALPHA_TEST)
        PUSHFLAG(GL_BLEND)
        PUSHFLAG(GL_CULL_FACE)
        PUSHFLAG(GL_DEPTH_TEST)
        PUSHFLAG(GL_LINE_SMOOTH)
        PUSHFLAG(GL_MULTISAMPLE)
        PUSHFLAG(GL_POLYGON_OFFSET_FILL)
        PUSHFLAG(GL_TEXTURE_2D)
        PUSHFLAG(GL_LIGHTING)

        for(GLint i=0;i<ln;i++)
        {
                PUSHFLAG(GL_LIGHT0+i)
        }
        printf("glPushAttrib : %d\n", n->data);

        n->next = stackTop;
        stackTop = n;
}

void glPopAttrib()
{
        GLint           ln;
        node * c = stackTop;
        if (c == NULL)
                return;
        glGetIntegerv(GL_MAX_LIGHTS, &ln);
        printf("glPopAttrib  : %d\n", c->data);

        for(GLint i=ln-1;i>=0;i--)
        {
                POPFLAG(GL_LIGHT0+i)
        }
        POPFLAG(GL_LIGHTING)
        POPFLAG(GL_TEXTURE_2D)
        POPFLAG(GL_POLYGON_OFFSET_FILL)
        POPFLAG(GL_MULTISAMPLE)
        POPFLAG(GL_LINE_SMOOTH)
        POPFLAG(GL_DEPTH_TEST)
        POPFLAG(GL_CULL_FACE)
        POPFLAG(GL_BLEND)
        POPFLAG(GL_ALPHA_TEST)

        stackTop = c->next;
        free(c);
}



static void __gluMakeIdentityf(GLfloat m[16])
{
    m[0+4*0] = 1; m[0+4*1] = 0; m[0+4*2] = 0; m[0+4*3] = 0;
    m[1+4*0] = 0; m[1+4*1] = 1; m[1+4*2] = 0; m[1+4*3] = 0;
    m[2+4*0] = 0; m[2+4*1] = 0; m[2+4*2] = 1; m[2+4*3] = 0;
    m[3+4*0] = 0; m[3+4*1] = 0; m[3+4*2] = 0; m[3+4*3] = 1;
}

void
gluOrtho2D(GLfloat left, GLfloat right, GLfloat bottom, GLfloat top)
{
    glOrthof(left, right, bottom, top, -1, 1);
}

#define __glPi 3.14159265358979323846

void
gluPerspective(GLfloat fovy, GLfloat aspect, GLfloat zNear, GLfloat zFar)
{
    GLfloat m[4][4];
    float sine, cotangent, deltaZ;
    float radians = fovy / 2 * __glPi / 180;

    deltaZ = zFar - zNear;
    sine = sinf(radians);
    if ((deltaZ == 0) || (sine == 0) || (aspect == 0)) {
	return;
    }
    cotangent = cosf(radians) / sine;

    __gluMakeIdentityf(&m[0][0]);
    m[0][0] = cotangent / aspect;
    m[1][1] = cotangent;
    m[2][2] = -(zFar + zNear) / deltaZ;
    m[2][3] = -1;
    m[3][2] = -2 * zNear * zFar / deltaZ;
    m[3][3] = 0;
    glMultMatrixf(&m[0][0]);
}

static void normalize(float v[3])
{
    float r;

    r = sqrt( v[0]*v[0] + v[1]*v[1] + v[2]*v[2] );
    if (r == 0.0) return;

    v[0] /= r;
    v[1] /= r;
    v[2] /= r;
}

static void cross(float v1[3], float v2[3], float result[3])
{
    result[0] = v1[1]*v2[2] - v1[2]*v2[1];
    result[1] = v1[2]*v2[0] - v1[0]*v2[2];
    result[2] = v1[0]*v2[1] - v1[1]*v2[0];
}

void
gluLookAt(GLfloat eyex, GLfloat eyey, GLfloat eyez, GLfloat centerx,
	  GLfloat centery, GLfloat centerz, GLfloat upx, GLfloat upy,
	  GLfloat upz)
{
    float forward[3], side[3], up[3];
    GLfloat m[4][4];

    forward[0] = centerx - eyex;
    forward[1] = centery - eyey;
    forward[2] = centerz - eyez;

    up[0] = upx;
    up[1] = upy;
    up[2] = upz;

    normalize(forward);

    /* Side = forward x up */
    cross(forward, up, side);
    normalize(side);

    /* Recompute up as: up = side x forward */
    cross(side, forward, up);

    __gluMakeIdentityf(&m[0][0]);
    m[0][0] = side[0];
    m[1][0] = side[1];
    m[2][0] = side[2];

    m[0][1] = up[0];
    m[1][1] = up[1];
    m[2][1] = up[2];

    m[0][2] = -forward[0];
    m[1][2] = -forward[1];
    m[2][2] = -forward[2];

    glMultMatrixf(&m[0][0]);
    glTranslatef(-eyex, -eyey, -eyez);
}
#endif
