/** @file Window.cpp
 *  @brief Implement the Graphics::Window class.
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "Window.h"

#include <iostream>

#include <Debug.h>
#if !defined(HAVE_GLES)
#include <GL/glew.h>
#include <GL/gl.h>
#else
#include <GLES/gl.h>
#include "eglport.h"
#define GLdouble     GLfloat
#define glOrtho      glOrthof
#endif


namespace Graphics
{

Window::Window()
    :   created(false),
        bpp(32),
        flags(SDL_OPENGL | SDL_RESIZABLE)
{
}

Window & Window::get_instance()
{
    static Window * window = new Window;
    return *window;
}

void Window::set_size(unsigned int width_in, unsigned int height_in, bool fullscreen_in)
{
    width = width_in;
    height = height_in;
    fullscreen = fullscreen_in;
    flags = SDL_OPENGL | SDL_RESIZABLE | SDL_SWSURFACE;
    bpp = 32;
    if (fullscreen) flags |= SDL_FULLSCREEN; 
#ifdef PANDORA 
	width = 800;
	height = 480;
#endif
#ifdef HAVE_GLES
	flags = SDL_FULLSCREEN | SDL_SWSURFACE;
	bpp = 16;
#endif
    bool need_glew_init = false;
    if (!created)
    {
        if (SDL_Init(SDL_INIT_VIDEO) == -1)
        {
            std::cerr << "Error initalising SDL.\n";
            throw;
        }
        
#ifndef HAVE_GLES
        SDL_GL_SetAttribute(SDL_GL_RED_SIZE, 8);
        SDL_GL_SetAttribute(SDL_GL_GREEN_SIZE, 8);
        SDL_GL_SetAttribute(SDL_GL_BLUE_SIZE, 8);
        SDL_GL_SetAttribute(SDL_GL_ALPHA_SIZE, 0);
        SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 16);
        SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
#else
	if (!EGL_Open())
		exit(1);
#endif  
        SDL_WM_SetIcon(SDL_LoadBMP("data/icons/r32.bmp"), NULL);
        
        created = true;
        need_glew_init = true;
    }
    if (SDL_SetVideoMode(width, height, bpp, flags ) == 0)
    {
        /* Might happen if no DISPLAY variable, fullscreen resolution not
         * available, or some other error. */
        std::cerr << "Error creating OpenGL window." << std::endl;
        throw; 
    }
#ifdef HAVE_GLES
	EGL_Init();
#else
    DEBUG_MESSAGE("Got a window.");
    if (need_glew_init)
    {
        // initalise GLEW on first run.
        glewInit();
        GLenum err = glewInit();
        if (GLEW_OK != err)
        {
          // glewInit failed
          std::cerr << "Error initalising GLEW." << std::endl;
          throw;
        }
        DEBUG_MESSAGE("Using GLEW " << glewGetString(GLEW_VERSION));
    }
#endif
    SDL_WM_SetCaption("Racer", "Racer");
    glViewport(0, 0, width, height);
}

Window::~Window()
{
}

void Window::swap_buffers()
{
#if !defined(HAVE_GLES)
    SDL_GL_SwapBuffers();
#else
	EGL_SwapBuffers();
#endif
}


void Window::set_ortho_projection()
{
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(0.0, (GLdouble) width, 0.0, (GLdouble) height, -1.0, 1.0);
    glMatrixMode(GL_MODELVIEW);
}

unsigned int Window::get_height() const
{
    return height;
}

unsigned int Window::get_width() const
{
    return width;
}

}
