/** @file racer/Graphics/SkyParticles.cpp
 *  @brief Implement the Graphics::SkyParticles class.
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "SkyParticles.h"
#include <cstdlib>
#include <cmath>
#include <GL/gl.h>

#include "../Engine/ResourceHandler.h"
typedef Engine::ResourceHandler<Track::Texture, std::string, std::string> TextureStore;

namespace Graphics
{

/// The amount of space covered around the camera in each direction.
const btScalar covered_size = 64.0;
const btScalar half_covered_size = covered_size / 2.0;
const btScalar square_half_covered_size = half_covered_size * half_covered_size;

SkyParticles::SkyParticles()
    :   phase (0)
{
    for (int i = 0; i < m_num_particles; i++)
    {
        particles[i]=btVector3(float (std::rand()) / float(RAND_MAX) * covered_size,
                               float (std::rand()) / float(RAND_MAX) * covered_size,
                               float (std::rand()) / float(RAND_MAX) * covered_size);
    }
    
    TextureStore & tex_store = TextureStore::get_instance();
    #define tex_load(bind_name, file_name, variable)\
                tex_store.check_load(bind_name, file_name);\
                variable = tex_store.get(bind_name);\
                variable->make_cache()
    tex_load("fire.particle.generic", "data/generic/particle_fire.png", m_texture);
    #undef tex_load
}

void SkyParticles::update(unsigned int time)
{
    // move particles.
    btScalar move = time * 0.004;
    btScalar side_move = time * 0.002;
    phase += float(time) * 0.005;
    
    for (int i = 0; i < m_num_particles; i++)
    {
        particles[i].setZ(particles[i].getZ() + move);
        if (particles[i].getZ() > covered_size) particles[i].setZ(particles[i].getZ()-covered_size);
        // make them spiral.
        particles[i].setX(particles[i].x() + sin(phase + i) * side_move);
        particles[i].setY(particles[i].y() + cos(phase + i) * side_move);
    }
}

void SkyParticles::draw(const Engine::CarCamera & camera)
{
    // The scale to draw the particles horizontally.
    const float s = 0.1;
    
    // Particles are drawn around the centre point. The position is wrapped so
    // that they form a 3d repeated pattern.
    // They are rotated to face the centre.
    const btVector3 & centre = camera.get_location();
    
    glEnable(GL_BLEND);
    glDepthMask(GL_FALSE);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE);
    glDisable(GL_CULL_FACE);
    m_texture->bind();
    
#if 0 //defined(HAVE_GLES)
    glBegin(GL_QUADS);
#else
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);

	GLfloat tex1[] = {	0.0, 0.0,	0.0, 1.0,	1.0, 1.0,	1.0, 0.0	};
	glTexCoordPointer(2, GL_FLOAT, 0, tex1);
#endif
    btVector3 major_offset = centre * (1.0/covered_size);
    major_offset = btVector3(floor(major_offset.x() - 0.5),
                             floor(major_offset.y() - 0.5),
                             floor(major_offset.z() - 0.5));
    major_offset *= covered_size;
    btScalar x_max = centre.x()+half_covered_size;
    btScalar y_max = centre.y()+half_covered_size;
    btScalar z_max = centre.z()+half_covered_size;
    btScalar x_min = centre.x()-half_covered_size;
    btScalar y_min = centre.y()-half_covered_size;
    btScalar z_min = centre.z()-half_covered_size;
    // offset the tail to make it look like it has moved during the exposure
    // of a regular camera.
    // 4 physics ticks = 40 msec = max exposure time for 25 fps.
    btVector3 a_offset = camera.get_velocity() * -4.0 +
                         // component for particles' constant vertical motion.
                         btVector3(0.0, 0.0, -0.16);
    btVector3 b_offset = a_offset;
    for (int i = 0; i < m_num_particles; i++)
    {
        btVector3 loc = particles[i]+major_offset;
        if (loc.x() > x_max) loc.setX(loc.x()-covered_size);
        if (loc.y() > y_max) loc.setY(loc.y()-covered_size);
        if (loc.z() > z_max) loc.setZ(loc.z()-covered_size);
        if (loc.x() < x_min) loc.setX(loc.x()+covered_size);
        if (loc.y() < y_min) loc.setY(loc.y()+covered_size);
        if (loc.z() < z_min) loc.setZ(loc.z()+covered_size);
        
        b_offset.setX(sin(phase + i)*-0.08 + a_offset.x());
        b_offset.setY(cos(phase + i)*-0.08 + a_offset.y());
        
        btScalar angle = std::atan2(loc.y() - centre.y(), loc.x() - centre.x());
        // scale the width of particles so distance particles have 0 width.
        // This means that distant particles don't pop into existence.
        btScalar distance_measure = (1.0 - loc.distance2(centre) / square_half_covered_size) * 10.0;
        if (distance_measure > 1.0) distance_measure = 1.0;
        btScalar distance_scale = s * distance_measure;
        if (distance_scale < 0.0) continue;
        // find angular components so that particle faces the camera in xy.
        btScalar sin = std::sin(angle) * distance_scale;
        btScalar cos = std::cos(angle) * distance_scale;
        
#if 0 //defined(HAVE_GLES)
        glTexCoord2f(0.0, 0.0);
        glVertex3f(loc.getX() - sin, loc.getY() + cos, loc.getZ());
        glTexCoord2f(0.0, 1.0);
        glVertex3f(loc.getX() - sin + b_offset.x(), loc.getY() + cos + b_offset.y(), loc.getZ() + b_offset.z());
        glTexCoord2f(1.0, 1.0);
        glVertex3f(loc.getX() + sin + b_offset.x(), loc.getY() - cos + b_offset.y(), loc.getZ() + b_offset.z());
        glTexCoord2f(1.0, 0.0);
        glVertex3f(loc.getX() + sin, loc.getY() - cos, loc.getZ());
#else
	GLfloat vtx1[] = {
		loc.getX() - sin, loc.getY() + cos, loc.getZ(),
		loc.getX() - sin + b_offset.x(), loc.getY() + cos + b_offset.y(), loc.getZ() + b_offset.z(),
		loc.getX() + sin + b_offset.x(), loc.getY() - cos + b_offset.y(), loc.getZ() + b_offset.z(),
		loc.getX() + sin, loc.getY() - cos, loc.getZ()
	};
 
	glVertexPointer(3, GL_FLOAT, 0, vtx1);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
#endif
    }
#if 0 //defined(HAVE_GLES)
    glEnd();
#else
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
#endif
    glDepthMask(GL_TRUE);
    glDisable(GL_BLEND);
    glEnable(GL_CULL_FACE);
}

} // namespace Graphics
