/** @file CubeMap.cpp
 *  @brief Implement the Graphics::CubeMap class.
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "CubeMap.h"
#include <Debug.h>

#include <SDL_image.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <cmath>

namespace Graphics
{

CubeMap::CubeMap(std::string filename_base)
    :   Texture()
{
    /** @todo Cubemap loading not quite right, but not used yet anyway.
     */
#ifndef HAVE_GLES
    m_target = GL_TEXTURE_CUBE_MAP;
    glBindTexture(m_target, m_texture_name);
    for (unsigned int face = 0; face < 6; face++)
    {
        m_filename = filename_base + char('0' + face);
        switch (face)
        {
            case 0:
                m_target = GL_TEXTURE_CUBE_MAP_POSITIVE_X;
                break;
            case 1:
                m_target = GL_TEXTURE_CUBE_MAP_POSITIVE_Y;
                break;
            case 2:
                m_target = GL_TEXTURE_CUBE_MAP_NEGATIVE_X;
                break;
            case 3:
                m_target = GL_TEXTURE_CUBE_MAP_NEGATIVE_Y;
                break;
            case 4:
                m_target = GL_TEXTURE_CUBE_MAP_POSITIVE_Z;
                break;
            case 5:
                m_target = GL_TEXTURE_CUBE_MAP_NEGATIVE_Z;
                break;
        }
        load_from_file();
    }
    m_target = GL_TEXTURE_CUBE_MAP;
    glTexParameteri(GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    DEBUG_MESSAGE("Loaded cubemap " << filename_base);
#endif
}

CubeMap::~CubeMap()
{
    DEBUG_MESSAGE("Deleting cubemap.");
}

void CubeMap::draw_sky_box()
{
    // draw the skybox.
#ifndef HAVE_GLES
    bind();
    glEnable(GL_TEXTURE_CUBE_MAP);
    const float l = std::sqrt(1.0 / 3.0);
    
    glBegin(GL_QUADS);
        // [face][vertex][dimension]
        const float cmz[6][4][3] = {{{-l, -l, -l}, {l, -l, -l}, {l, l, -l}, {-l, l, -l}}, // -z
                              {{-l, -l, l}, {l, -l, l}, {l, l, l}, {-l, l, l}}, // +z
                              {{-l, -l, -l}, {-l, l, -l}, {-l, l, l}, {-l, -l, l}}, // -x
                              {{l, -l, -l}, {l, l, -l}, {l, l, l}, {l, -l, l}}, // +x
                              {{-l, -l, -l}, {l, -l, -l}, {l, -l, l}, {-l, -l, l}}, // -y
                              {{-l, l, -l}, {l, l, -l}, {l, l, l}, {-l, l, l}}}; // +y
        
        for (unsigned int f = 0; f < 6; f++)
        {                  
            for (unsigned int v = 0; v < 4; v++)
            {
                glNormal3f(cmz[f][v][0], cmz[f][v][1], cmz[f][v][2]);
                // Specify a texture coordinate implies crash.
                glTexCoord3f(cmz[f][v][0], cmz[f][v][1], cmz[f][v][2]);
                glVertex3f(cmz[f][v][0], cmz[f][v][1], cmz[f][v][2]);
            } 
        }
    glEnd();
    glDisable(GL_TEXTURE_CUBE_MAP);
#endif
}

}
