/** @file Scene.h
 *  @brief Declare the Engine::Scene class.
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef SCENE_H_
#define SCENE_H_

#include "InputReport.h"

class MainLoop;

/// The game engine for racer
namespace Engine
{

/** A scene is something that occupies a MainLoop.
 * It should update it self every frame depending on the time taken, and respond
 * to user input. It will also get to draw things every frame.
 * 
 * Only one Scene may be active in a MainLoop at any one time.
 */
class Scene
{
public:
    Scene();
    virtual ~Scene();
    
    /** Take input from an input device.
     * @param report An input report, which may be freed immediately after the
     * call returns.
     */
    virtual void take_input(InputReport & report) = 0;
    
    /** Process any status changes that occur due to elapsed time.
     * @param milliseconds_elapsed time in milliseconds since last update_logic.
     */
    virtual void update_logic(unsigned int milliseconds_elapsed) = 0;
    
    /** Draw the scene using OpenGL commands.
     * Must go from any state to ready for a buffer swap.
     */
    virtual void draw() = 0;
    
    /** Update the audio.
     */
    virtual void do_sound() = 0;
    
    /** Attach a Main loop object. If the scene wishes to end, it can notify
     * the main loop obect by calling exit.
     */
    virtual void attach_main_loop(MainLoop & main_loop);
protected:
    /// pointer to the main_loop to contact if we should end.
    MainLoop * main_loop;
};

}

#endif /*SCENE_H_*/
