/** @file ResourceHandler.h
 *  @brief Declare  the Engine::ResourceHandler class.
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef RESOURCEHANDLER_H_
#define RESOURCEHANDLER_H_

#include <map>

namespace Engine
{

/** Handle a large quantity of data that may be shared and used in multiple
 * places, but we only want it loaded once. For example textures and meshes.
 * 
 * @tparam Content the class for the objects we wish to store.
 * @tparam Key the class for the identifier used to distinguish the objects. It
 * must be orderable, i.e. it has an well-behaved
 * @code operator<(Key lvalue, Key rvalue). @endcode .
 * The Key should be much smaller than the Content.
 * @tparam ContentConstructionInformation A class that can be passed to the
 * constuctor of the Content class. It should also be much smaller than the
 * Content object it will create.
 */
template<class Content, class Key, class ContentConstructionInformation>
class ResourceHandler
{
    // We only want one resource handler, so block public access to the default
    // constructor, copy constructor, and assigment operator.
    ResourceHandler();
    ResourceHandler(const ResourceHandler & source);
    ResourceHandler & operator=(const ResourceHandler & source);
public:
    /** Get an instance to the resource handler.
     * As the class is templatised, there will be exactly one object per
     * combination of Content, Key and ConstructionInfo types: be careful not to
     * mix Key types when you don't want to. For constructors that take multiple
     * parameters, you'll have to make another class for the paramters. If you
     * want to store objects with a different constructor argument signature in
     * the same ResourceHandler, you'll need to make an abstract class like that
     * and redesign the class.
     */
    static ResourceHandler<Content, Key, ContentConstructionInformation> & get_instance();
    virtual ~ResourceHandler();
    
    /** If a key is not paired with a Content object, create one using the
     * information provided.
     * If the key is already present, nothing happens.
     * @param key The key that should be attached.
     * @param construction_information The argument to pass to the constructor
     * of Content to create the object, if the key is not found.
     */
    void check_load(const Key & key, ContentConstructionInformation construction_information);
    /** Get a pointer to the data stored for the given key, or 0 if the key is
     * unused.
     * @param key The key to look up.
     * @return A pointer to the Content object created by the first call to
     * check_load() with this key. If no such call was made, returns 0.
     */
    Content * get(const Key & key);
private:
    std::map<Key, Content *> data;
};

template<class Content, class Key, class ContentConstructionInformation>
ResourceHandler<Content, Key, ContentConstructionInformation>::ResourceHandler()
{
}

template<class Content, class Key, class ContentConstructionInformation>
ResourceHandler<Content, Key, ContentConstructionInformation>::~ResourceHandler()
{
    // delete all of the data.
    typename std::map<Key, Content *>::iterator iterator;
    for (iterator = data.begin(); iterator != data.end(); iterator++)
    {
        delete iterator->second;
    }
}

template<class Content, class Key, class ConstructionInfo>
ResourceHandler<Content, Key, ConstructionInfo> & ResourceHandler<Content, Key, ConstructionInfo>::get_instance()
{
    static ResourceHandler<Content, Key, ConstructionInfo> resource_handler;
    return resource_handler;
}

template<class Content, class Key, class ContentConstructionInformation>
void ResourceHandler<Content, Key, ContentConstructionInformation>::check_load(const Key & key,
                                                                 ContentConstructionInformation construction_information)
{
    // Does the object already exist?
    typename std::map<Key, Content *>::iterator it = data.find(key);
    if (it == data.end())
    {
        // We didn't find it. Constuct it.
        data[key] = new Content(construction_information);
    }
}

template<class Content, class Key, class ConstructionInfo>
Content * ResourceHandler<Content, Key, ConstructionInfo>::get(const Key & key)
{
    typename std::map<Key, Content *>::iterator it = data.find(key);
    if (it == data.end())
    {
        // not found
        return 0;
    }
    // found, return the pointer.
    return it->second;
}

}

#endif /*RESOURCEHANDLER_H_*/
