/** @file ReplayReader.h
 *  @brief Declare the Engine::ReplayReader class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef REPLAY_READER_H_
#define REPLAY_READER_H_

#include <istream>
#include <vector>
#include <set>

#include "Physics/TickObserver.h"
#include "InputReport.h"
#include "InputDeviceReplay.h"
#include "GameScene.h"
#include "LoadScene.h"

namespace Engine
{

/** Show a replay from a stream.
 * 
 * The replay is stored as some metadata, plus a series of timestamped events
 * recorded from input devices.
 * When created it reads the metadata. It can explain the metadata, and create
 * an appropriate scene. As the scene runs, it will report the events at the
 * correct time so that the same things happen.
 * 
 * For this method to work, we require that all physics, game logic, and AI is
 * deterministic, i.e. when presented with the same events with the same
 * timings, things turn out just as before.
 * 
 * The alternative would be to store a set of states.
 * This has the following disadvantages:
 * - More complex, as there would be a wider variety of events to record.
 * - Every part of the game state would have to be regenerated from state saved
 *   in the replay and calculated in the inital run, or saved and restored.
 * - It is likey to take up far more space.
 * 
 * With just the input, we only need to have a list of InputReport objects saved
 * and timestamped, and determinstic state determined only from an input.
 */
class ReplayReader
    :   public Physics::TickObserver
{
public:
    /** Replay from a stream.
     * This expects some game identifying metadata, and a list of events in a
     * scene.
     * @param data The stream to take input from. 
     */
    ReplayReader(std::istream & data);
    virtual ~ReplayReader();
    
    /// Process any events that happen next tick.
    virtual void posttick();
    
    /// Called by LoadScene when it has finished loading.
    void operator ()(GameScene *);
    
    /// Return the scene that will show the replay.
    Scene * get_scene();
protected:
    struct ReplayEvent
    {
        unsigned long int tick_number;
        InputReport::ReportType type;
        int value;
        long unsigned int car_identifier;
    };
    
    /// Read the events in the file into replay_events.
    void ReadEvents(std::istream & data);
    
    /// The list of events in the replay, sorted in time order.
    std::vector<ReplayEvent> replay_events;
    
    /// Iterator to the next event to process.
    std::vector<ReplayEvent>::iterator current_iterator;
    
    /// Input devices to relay the events through. There is one per player.
    std::vector<boost::shared_ptr<InputDeviceReplay> > devices;
    /// Map identifiers found in the replay data to InputDeviceReplays.
    std::map<long unsigned int, Engine::InputHandler::iterator> input_map;
    std::vector<std::pair<Engine::InputHandler::iterator, unsigned int> > game_scene_devices_list;
    
    /// The scene used to load the track.
    LoadScene<GameScene, ReplayReader> * load_scene;
    /// The scene the replay takes place in.
    GameScene * scene;
};

}

#endif /*REPLAY_READER_H_*/
