/** @file ReplayReader.cpp
 *  @brief Implement the Engine::ReplayReader class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "ReplayReader.h"
#include "InputHandler.h"
#include <Debug.h>

namespace Engine
{

ReplayReader::ReplayReader(std::istream & data)
    :   load_scene(0)
{
    // Read metadata
    std::string stage;
    std::getline(data, stage);
    
    std::size_t number_of_cars;
    data >> number_of_cars;
    DEBUG_MESSAGE("Replay has " << number_of_cars << " car(s).");
    game_scene_devices_list.reserve(number_of_cars);
    for (unsigned int i = 0; i < number_of_cars; i++)
    {
        // The identifier was a pointer to the input device that recorded the
        // replay. It is not valid across sessions, so use a long int.
        long unsigned int car_identifier;
        data >> car_identifier;
        unsigned int car_model;
        data >> car_model;
        
        boost::shared_ptr<Engine::InputDeviceReplay> new_device(new Engine::InputDeviceReplay());
        devices.push_back(new_device);
        Engine::InputHandler::iterator it = new_device->get_handle();
        input_map.insert(std::pair<long unsigned int, Engine::InputHandler::iterator>(car_identifier, it));
        
        game_scene_devices_list.push_back(std::make_pair(it, car_model));
    }
    // Fill the list of events using data on the stream
    ReadEvents(data);
    
    // create scene to load game.
    load_scene = new Engine::LoadScene<Engine::GameScene, ReplayReader> (game_scene_devices_list, stage);
    load_scene->set_done_notifier(this);
}

void ReplayReader::operator()(GameScene * scene_in)
{
    scene = scene_in;
    // monitor ticks so that we can submit events at the right time.
    scene->get_world().add_tick_observer(this);
    // we don't want to save a replay of the replay.
    scene->set_save_replay(false);
    // Process events that happen before the first tick.
    posttick();
}

ReplayReader::~ReplayReader()
{
    delete load_scene;
}

Scene * ReplayReader::get_scene()
{
    return load_scene;
}

void ReplayReader::posttick()
{
    if (current_iterator == replay_events.end())
    {
        // no more events.
        return;
    }
    // We want to recreate each event on the tick one before it was recorded,
    // since the cars have already postticked this physics tick, and normally
    // the input they receive is before the posttick.
    
    // This is <= even though in most cases it should be exactly equal.
    // This is so that the first tick works.
    while (current_iterator->tick_number <= scene->get_world().get_tick_number() + 1)
    {
        // process this event, and start waiting for the next.
        // The car identifier for this event might not have been assigned a
        // car if the replay file was corrupt.
        std::map<long unsigned int, Engine::InputHandler::iterator>::iterator
                input_map_iterator =
                            input_map.find(current_iterator->car_identifier);
        if (input_map_iterator != input_map.end())
        {
            dynamic_cast<Engine::InputDeviceReplay *>(*input_map_iterator->second)->
                    relay(current_iterator->type, current_iterator->value);
        }
        current_iterator++;
    }
}

void ReplayReader::ReadEvents(std::istream & source)
{
    while (source.good())
    {
        ReplayEvent event;
        int type_enum;
        source >> event.tick_number
               >> event.car_identifier
               >> type_enum
               >> event.value;
        event.type = InputReport::ReportType(type_enum);
        if (!(source.fail()||source.bad()))
        {
            replay_events.push_back(event);
        }
    }
    current_iterator = replay_events.begin();
    DEBUG_MESSAGE("Replay has " << replay_events.size() << " events.");
}

}
