/** @file World.h
 *  @brief Declare the Engine::Physics::World class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef WORLD_H_
#define WORLD_H_

#include <set>
#include <vector>
#include <ostream>
#include <btBulletDynamicsCommon.h>
#include "TickObserver.h"
#include "../InputReport.h"
#include <libtrack/Track.h>

namespace Engine
{

/** Objects that only do physics.
 * Stuff that has a physical representation but has other game play related
 * effects or graphics are generally in GameObjects instead.
 */
namespace Physics
{

class World
{
private:
    World & operator=(const World & world);
    World(const World & world);
public:
    World(const Track::Track & track);
    virtual ~World();
    void update(unsigned int milliseconds_elapsed);
    btDiscreteDynamicsWorld & get_dynamics_world();
    void add_tick_observer(TickObserver * observer);
    void remove_tick_observer(TickObserver * observer);
    /// Get the world containing only the floors you can drive on.
    btCollisionWorld & get_floor_world();
    
    /** Record an input report in order to create a replay.
     * Replays are reconstructed from the original input.
     * The current tick is also recorded with the report.
     * @param report to use.
     */
    void add_replay_report(const InputReport & report);
    
    /** Write the events that have occured to a stream for use as a replay.
     * The stream should already have data written to it tht identifies the
     * scene and players. This just records input device presses since the scene
     * has started, along with the tick number of each event.
     * @param stream The stream to write the data to.
     */
    void write_replay_events(std::ostream & stream);
    
    /// Get the number of physics ticks simulated.
    unsigned long int get_tick_number();
    
    /// Get a reference to the track being simulated.
    const Track::Track & get_track();
private:
    const Track::Track & m_track;
    
    btAxisSweep3* broadphase;
    btAxisSweep3* floor_broadphase;
    btDefaultCollisionConfiguration* collisionConfiguration;
    btDefaultCollisionConfiguration* floor_collision_configuration;
    btCollisionDispatcher* dispatcher;
    btCollisionDispatcher* floor_dispatcher;
    btSequentialImpulseConstraintSolver* solver;
    /// Main world for physics simulation.
    btDiscreteDynamicsWorld* dynamicsWorld;
    /// world containing just the surfaces you can drive on.
    btCollisionWorld * floor_world;
    
    std::set<TickObserver *> observers;
    
    int milliseconds_remaining;
    
    /// An event recorded for the replay
    struct ReplayEvent
    {
        ReplayEvent(unsigned long int tick_number, const InputReport & report);
        unsigned long int tick_number;
        InputReport report;
    };
    
    /// replay data
    std::vector<ReplayEvent> replay_events;
    /// The current tick number.
    unsigned long int tick_number;
};

}

}

#endif /*WORLD_H_*/
