/** @file World.cpp
 *  @brief Implement the Engine::Physics::World class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include <fstream>

#include "World.h"
#include <Debug.h>
#include "../InputHandler.h"
#include "../Audio.h"

namespace Engine
{

namespace Physics
{

const int milliseconds_per_tick = 12;
const float seconds_per_tick = btScalar(milliseconds_per_tick) / btScalar(1000.0);
/* Limit ticks so if physics takes longer to calculate than the time we are
 * simulating, we don't grind to a halt. The maximum amount of ticks should
 * be about enough to support 12 frames per second before slowing down the
 * simulation.
 */
const unsigned int max_tick_count = 1000 / 12 / milliseconds_per_tick + 1;

World::World(const Track::Track & track)
    :   m_track(track)
    ,   milliseconds_remaining(milliseconds_per_tick)
    ,   tick_number(1)
{
    // create world stuff
    
    // maximum number of rigid bodies in the scene
    int maxProxies = 64;
    // maximum bounds of the scene for broadphase
    Track::AxisAlignedBoundingBox bounds = track.get_path().get_bounds();
    bounds.add_border(20.0);
    broadphase = new btAxisSweep3(bounds.get_min(), bounds.get_max(), maxProxies);
    // full collision detection
    collisionConfiguration = new btDefaultCollisionConfiguration();
    dispatcher = new btCollisionDispatcher(collisionConfiguration);
    // solver
    solver = new btSequentialImpulseConstraintSolver;
    // dynamics world
    dynamicsWorld = new btDiscreteDynamicsWorld(dispatcher,broadphase,solver,collisionConfiguration);
    dynamicsWorld->setGravity(btVector3(0,0,-0.98));
    
    /* Floor collision world, with only the surfaces from course_collision_world
     * you can drive on. Mostly usefult to Car, not us.
     * Use a different collision detection so only the floor is detected.
     */
    floor_broadphase = new btAxisSweep3(bounds.get_min(), bounds.get_max(), maxProxies);
    floor_collision_configuration = new btDefaultCollisionConfiguration();
    floor_dispatcher = new btCollisionDispatcher(collisionConfiguration);
    floor_world = new btCollisionWorld(floor_dispatcher, floor_broadphase, floor_collision_configuration);
    
    /* Reserve some space for replay data, so we aren't constantly reallocating
     * a large array.
     */
    replay_events.reserve(100 * 60 * 3 // 100 frames per second, for 3 minutes
                          * 4 * 3); // 4 cars, 3 inputs each per frame.
    // 3 is the maximum number of inputs per frame, may be far less on average.
    
}

World::~World()
{
    delete dynamicsWorld;
    delete solver;
    delete collisionConfiguration;
    delete dispatcher;
    delete broadphase;
    
    delete floor_world;
    delete floor_broadphase;
    delete floor_collision_configuration;
    delete floor_dispatcher;
}

void World::update(unsigned int milliseconds_ellapsed)
{ 
    // milliseconds_remaining is the time in milliseconds that has passed but
    // hasn't been simulated yet.
    // Physics is done in fixed timesteps for determanism.
    // If there are left over milliseconds, we just leave them.
    milliseconds_remaining += milliseconds_ellapsed;
    while (milliseconds_remaining >= milliseconds_per_tick)
    {
        dynamicsWorld->stepSimulation(seconds_per_tick, 1, seconds_per_tick);
        
        // collision sounds
        int numManifolds = dynamicsWorld->getDispatcher()->getNumManifolds();
        for (int i=0;i<numManifolds;i++)
        {
            btPersistentManifold* contactManifold =  dynamicsWorld->getDispatcher()->getManifoldByIndexInternal(i);
            btCollisionObject* obA = static_cast<btCollisionObject*>(contactManifold->getBody0());
            btCollisionObject* obB = static_cast<btCollisionObject*>(contactManifold->getBody1());
        
            int numContacts = contactManifold->getNumContacts();
            for (int j=0;j<numContacts;j++)
            {
                btManifoldPoint& pt = contactManifold->getContactPoint(j);
                if (pt.getDistance()<0.f)
                {
                    const btVector3& ptA = pt.getPositionWorldOnA();
                    const btVector3& ptB = pt.getPositionWorldOnB();
                    const btVector3& normalOnB = pt.m_normalWorldOnB;
                    static SoundBuffer sound_buffer("data/sound/collision.wav");
                    static SoundSource sound_source(sound_buffer);
                    sound_source.set_position(ptA);
                    sound_source.play();
                }
            }
        }
        // check for input (some of the time so AI doesn't take so long).
        // input is additionaly checked by scenes once per rendered frame.
        if (!(tick_number % 4))
        {
            Engine::InputHandler::get_instance().poll();
        }
        // forces set in the tick callback don't seem to work, hence this
        // loop. Call pretick set the forces.
        for (std::set<TickObserver *>::iterator it = observers.begin();
             it != observers.end();
             it++)
        {
            (*it)->posttick();
            (*it)->set_forces();
        }
        tick_number++;
        milliseconds_remaining -= milliseconds_per_tick;
    }
}

btDiscreteDynamicsWorld & World::get_dynamics_world()
{
    return *dynamicsWorld;
}

btCollisionWorld & World::get_floor_world()
{
    return *floor_world;
}

void World::add_tick_observer(TickObserver * tick_observer)
{
    observers.insert(tick_observer);
}

void World::remove_tick_observer(TickObserver * tick_observer)
{
    observers.erase(tick_observer);
}

void World::add_replay_report(const InputReport & report)
{
    switch (report.get_report_type())
    {
        case InputReport::RT_MENU_BACK:
        case InputReport::RT_MENU_DOWN:
        case InputReport::RT_MENU_LEFT:
        case InputReport::RT_MENU_RIGHT:
        case InputReport::RT_MENU_SELECT:
        case InputReport::RT_MENU_UP:
        case InputReport::RT_DISCONNECT:
        case InputReport::RT_BATTERY_LOW:
            // don't record menu inputs or anything else that doesn't affect physics.
            break;
        default:
            // record other inputs.
            replay_events.push_back(ReplayEvent(tick_number, report));
    }
}

World::ReplayEvent::ReplayEvent(unsigned long int tick_number,
                                const InputReport & report)
    :   tick_number(tick_number)
    ,   report(report)
{
}


void World::write_replay_events(std::ostream & stream)
{
    // record replay to file.
    /// @todo record end time.
    /** @todo Record some absolute information so we can correct behaviour of
     * different floating point units making the replay more portable.
     */
    DEBUG_MESSAGE("Writing replay");
    for (std::vector<ReplayEvent>::iterator it = replay_events.begin();
         it != replay_events.end(); it++)
    {
        const ReplayEvent & event = *it;
        const InputReport & report = event.report;
        stream << event.tick_number << " "
               << (unsigned long int)&*(report.get_input_device()) << " "
               << report.get_report_type() << " "
               << report.get_value() << " ";
    }
}

unsigned long int World::get_tick_number()
{
    return tick_number;
}

const Track::Track & World::get_track()
{
    return m_track;
}

}

}
