/** @file racer/Engine/LoadScene.h
 *  @brief Declare the Engine::LoadScene class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef RACER_ENGINE_LOAD_SCENE_H
#define RACER_ENGINE_LOAD_SCENE_H

#include "GameScene.h"
#include "InputHandler.h"
#include "InputDeviceAI.h"
#include "../MainLoop.h"
#include "../UI/BasicFonts.h"
#include "../Graphics/Window.h"

#include <libtrack/Track.h>
#include <libtrack/Texture.h>

#include <GL/gl.h>

#include <boost/shared_ptr.hpp>

#include <vector>
#include <string>
#include <fstream>

namespace Engine
{

template <class T>
class daft_function_object
{
public:
    void operator()(T * in) {};
};

/** Load a game.
 * @tparam class of scene to create. Must be an Engine::Scene constuctable
 * with a std::vector<std::size_t> picking the input devices and a
 * Track::Track for the track.
 */
template <class T = GameScene, class Q = daft_function_object<T> >
class LoadScene : public Engine::Scene
{
public:
    /**  Load a game where the players use specified input devices.
     * @param input_devices InputHandler iterators for devices to use, paired
     * with the car each device's user wants. The order is important.
     * @param filename The filename of the track.
     * @param number_of_ais The number of AI players wanted.
     */
    LoadScene(std::vector<std::pair<InputHandler::iterator, unsigned int> > input_devices,
              std::string filename,
              unsigned int number_of_ais = 0)
        :   blanked(false)
        ,   track_filename(filename)
        ,   input_devices(input_devices)
        ,   function_object(0)
        ,   done(false)
        ,   error(false)
        ,   m_number_of_ais(number_of_ais)
    {
        main_loop = 0;
    }
    
    virtual ~LoadScene()
    {
    }
    
    virtual void take_input(InputReport & report)
    {
        // ignore input, unless there is an error.
        if (error)
        {
            // Try to quit, unless it could be noise causing the event.
            // An analgue joystick could output noise, but pushing it far
            // enough will report a menu movement.
            switch(report.get_report_type())
            {
                case InputReport::RT_MENU_UP:
                case InputReport::RT_MENU_DOWN:
                case InputReport::RT_MENU_LEFT:
                case InputReport::RT_MENU_RIGHT:
                case InputReport::RT_MENU_SELECT:
                case InputReport::RT_MENU_BACK:
                    if (main_loop)
                    {
                        main_loop->pop_scene();
                    };
                    break;
                default:
                    break; // possibly noise.
            };
        }
    }
    
    virtual void update_logic(unsigned int milliseconds_elapsed)
    {
        // wait until the screen has been blanked before trying any long
        // operations.
        if (!blanked) return;
        if (error) return; // doesn't work, don't try again.
        assert(!done);
        // load the track and its theme.
        try
        {
            track_file = boost::shared_ptr<std::ifstream>(new std::ifstream(track_filename.c_str()));
            if (!track_file->good())
            {
                throw std::ios_base::failure("Unable to read track file " + track_filename + ".");
            }
            track_file->exceptions (std::ifstream::eofbit | std::ifstream::failbit | std::ifstream::badbit);
            std::getline(*track_file, theme_filename);
            theme_file = boost::shared_ptr<std::ifstream>(new std::ifstream(theme_filename.c_str()));
            if (!theme_file->good())
            {
                throw std::ios_base::failure("Unable to read theme file " + theme_filename + " required by track " + track_filename + ".");
            }
            theme_file->exceptions (std::ifstream::eofbit | std::ifstream::failbit | std::ifstream::badbit);
            theme = boost::shared_ptr<Track::Theme>(new Track::Theme(*theme_file));
            track = boost::shared_ptr<Track::Track>(new Track::Track(*track_file, *theme));
            track->set_filename(track_filename);
        
            // Cache meshes and textures needed for the track.
            const Track::Path & path = track->get_path();
            typedef boost::graph_traits<Track::Path::Graph>::edge_iterator EdgeIterator;
            std::pair<EdgeIterator, EdgeIterator> edge_range;
            for (edge_range = boost::edges(path.graph);
                 edge_range.first != edge_range.second;
                 edge_range.first++)
            {
                const Track::PathEdge & edge = path.graph[*(edge_range.first)];
                edge.make_cache();
            }
            typedef boost::graph_traits<Track::Path::Graph>::vertex_iterator VertexIterator;
            std::pair<VertexIterator, VertexIterator> vertex_range;
            for (vertex_range = boost::vertices(path.graph);
                 vertex_range.first != vertex_range.second;
                 vertex_range.first++)
            {
                const Track::PathVertex & vertex = path.graph[*(vertex_range.first)];
                vertex.get_segment()->get_graphics_mesh().get_faces().make_cache();
            }
            
            // make the AI graph for the track.
            track->update_ai_mesh();
            
            // Add some AI players
            /// @todo Let a player pick the number of AI players?
            for (unsigned int ai = 0; ai < m_number_of_ais; ai++)
            {
                m_ai_devices.push_back(boost::shared_ptr<InputDeviceAI>(new InputDeviceAI()));
                input_devices.push_back(std::pair<InputHandler::iterator, unsigned int>(
                    m_ai_devices.back()->get_handle(),
                    ai%2)); // ai alterenatly uses both cars
            }
            
            // create the game scene
            game_scene = boost::shared_ptr<T>(new T(input_devices, *track));
            game_scene->attach_main_loop(*main_loop);
            if (function_object)
            {
                (*function_object)(&(*game_scene));
            }
            main_loop->push_scene(*game_scene);
            // now game scene will have finished, quit the load scene too.
            main_loop->pop_scene();
            done = true;
        }
        catch (std::ios_base::failure e)
        {
            std::cerr << "Failed to load track " << track_filename <<":" << std::endl
                      << e.what() << std::endl;
            error = true;
        }
        catch (ThemeChangedError e)
        {
            std::cerr << "Failed to load track (" << track_filename <<")," << std::endl
                      << "  since it requires theme elements which do not exist." << std::endl
                      << "  The linked theme is " << theme_filename << "." << std::endl;
            error = true;
        }
        catch (CorruptedError e)
        {
            std::cerr << "Failed to load track (" << track_filename <<")," << std::endl
                      << " since it is incositant with itself." << std::endl;
            error = true;
        }
        catch (NewVersionError e)
        {
            std::cerr << "Failed to load track (" << track_filename <<")," << std::endl
                      << " since it has been written with a newer version of racer." << std::endl;
            error = true;
        }
        catch (DepreciatedVersionError e)
        {
            std::cerr << "Failed to load track (" << track_filename <<")," << std::endl
                      << " since it contains depreciated features." << std::endl;
            error = true;
        }
        catch (FormError e)
        {
            std::cerr << "Failed to load track (" << track_filename <<")," << std::endl
                      << " since it is not in the expected format." << std::endl;
            error = true;
        }
    }
    
    virtual void draw()
    {
        glClearColor(0, 0, 0, 0);
        glClear(GL_COLOR_BUFFER_BIT);
        blanked = true;
        if (error)
        {
            // Tell user when track cannot be loaded.
            Graphics::Window::get_instance().set_ortho_projection();
            glLoadIdentity();
            glPushMatrix();
            glTranslatef(100, 120, 0);
            UI::BasicFonts::get_instance().big_font.Render("Failed to load track.");
            glPopMatrix();
            glTranslatef(100, 90, 0);
            UI::BasicFonts::get_instance().small_font.Render("See terminal window for more information.");
        }
    }
    
    virtual void do_sound()
    {
    }
    
    /// Give a function object the a reference to the scene when it's ready.
    void set_done_notifier(Q * function_object_in)
    {
        function_object = function_object_in;
    }
private:
    bool blanked;
    /// The filename of the track.
    std::string track_filename;
    boost::shared_ptr<std::ifstream> track_file;
    /// The filename of the theme the track uses
    std::string theme_filename;
    boost::shared_ptr<std::ifstream> theme_file;
    /// Theme the track uses
    boost::shared_ptr<Track::Theme> theme;
    /// The course to play on
    boost::shared_ptr<Track::Track> track;
    
    /// The InputDevices for the computer controlled players.
    std::vector<boost::shared_ptr<Engine::InputDeviceAI> > m_ai_devices;
    
    /// Devices to use for inputs paired with chosen car.
    std::vector<std::pair<InputHandler::iterator, unsigned int> > input_devices;
    
    boost::shared_ptr<T> game_scene;

    /// object to send new scene to.
    Q * function_object;
    
    bool done;
    
    /// True if and only if track cannot be loaded.
    bool error;
    
    /// The number of computer controlled cars requested.
    unsigned int m_number_of_ais;
};

} // namespace Engine

#endif // RACER_ENGINE_LOAD_SCENE_H
