/** @file InputReport.h
 *  @brief Declare the Engine::InputReport class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef INPUTREPORT_H_
#define INPUTREPORT_H_

#include <set>

namespace Engine
{
class InputDevice;
/** A message about a players control over their car or the menu.
 * 
 * Abstracted away is weather the player is human or AI, what input device a
 * human player is using, what the controller layout is like, and if certain
 * responses are analogue (joy stick or accellerometer) or digital (direction
 * pad, keys or buttons).
 */
class InputReport
{
public:
    /* The changed value or action we are reporting.
     */
    enum ReportType
    {
        /// Change to accelerator / break amount
        RT_CHANGE_ACCEL,
        
        /// Change to sideways crabbing control     
        RT_CHANGE_SLIDE,
        
        /// Change to steering wheel angle
        RT_CHANGE_STEERING,
        
        /// select item above in menus
        RT_MENU_UP,
        
        /// select item below in menus
        RT_MENU_DOWN,
        
        /// select item to the left in menus
        RT_MENU_LEFT,
        
        /// select item to the right in menus
        RT_MENU_RIGHT,
        
        /// go forwards in menus, pick the selected option
        RT_MENU_SELECT,
        
        /// go back to the previous menu
        RT_MENU_BACK,
        
        /// Device unplugged / switched off
        RT_DISCONNECT,
        
        /// Battery level warning, emmited by Wiimotes.
        RT_BATTERY_LOW,
        
        /// Maximum value for a report type.
        RT_MAX 
    };
        
    /** Create a report.
     * Input devices should report events as soon as they can. 
     * @param input_device the input device that caused the event. Important for
     * establishing which player this notification applies to.
     * @param type ReportType indicating what has changed.
     * @param value number indicating new strength for type. It should
     * range from -32767 to +32767, where 0 is neutral. It is ignored for some
     * values of type.
     */
    InputReport(const InputDevice * input_device, ReportType type, int value = 0);
    
    virtual ~InputReport();
    
    /** get the report type of this report
     * @return the ReportType enum telling what the report is indicating.
     */ 
    ReportType get_report_type() const;
    
    /** Get a value attached to this report. It is only used for reports of type
     * RT_CHANGE_ACCEL, RT_CHANGE_SLIDE, and RT_CHANGE_STEERING. 
     * @return Value associated with this report. It ranges from -32767 to
     * +32767, where 0 is neutral.
     */
    int get_value() const;
    
    /** Get the InputHandler's iterator of the input_device that created the report.
     */
    std::set<InputDevice *>::iterator get_input_device() const;
private:
    std::set<InputDevice *>::iterator input_device;
    ReportType type;
    int value;
};

}

#endif /*INPUTREPORT_H_*/
