/** @file InputHandler.h
 *  @brief Declare the Engine::InputHandler class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef INPUTHANDLER_H_
#define INPUTHANDLER_H_

#include <set>
#include <vector>
#include "InputDevice.h"
#include "Scene.h"

namespace Engine
{
class InputDeviceKeyboard;
class InputDeviceJoystick;
/** Coordinate all the InputDevices.
 */
class InputHandler
{
public:
    class iterator
        :   public std::set<InputDevice *>::iterator
    {
        public:
            iterator (const std::set<InputDevice *>::iterator & other)
                :   std::set<InputDevice *>::iterator(other)
            {
            }
            iterator ()
                :   std::set<InputDevice *>::iterator()
            {
            }
            bool operator <(const iterator & other) const
            {
                return **this < *other;
            }
    };
    
    virtual ~InputHandler();
    static InputHandler & get_instance();
    void set_scene(Scene & scene);
    void poll();
    
    /// Get an iterator to the first input device.
    iterator begin();
    /// Get an iterator to one past the last input device.
    iterator end();
    
    /// Get the number of devices found.
    std::size_t get_number_of_devices() const;
    
    /** Use an input device.
     * Call remove_device() with the same object or with the returned iterator
     * when done.
     * @return iterator pointing to the new device.
     */
    iterator add_device(InputDevice & device);
    /** Stop using an input device.
     */
    void remove_device(InputDevice & device);
    /** Stop using an input device.
     */
    void remove_device(iterator it);
private:    
    // We only want one InputHandler. Use get_instance to get it.
    InputHandler();
    /// Initalise keyboard and joystick devices.
    void init();
    
    std::set<InputDevice *> devices;
    
    // The scene to send notifications to.
    Scene * scene;
    // devices which always exist
    InputDeviceKeyboard * keyboard;
    
    /// a list of joystick devices avaliable on the system.
    std::vector<InputDeviceJoystick *> joysticks;
};

}

#endif /*INPUTHANDLER_H_*/
