/** @file InputHandler.cpp
 *  @brief Implement the Engine::InputHandler class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "InputHandler.h"

#include "InputDeviceKeyboard.h"
#include "InputDeviceJoystick.h"

#include <Debug.h>

namespace Engine
{

InputHandler::InputHandler()
    :   scene(0)
    ,   keyboard(0)
{
}

void InputHandler::init()
{
    // don't initalise twice.
    if (keyboard) return;
    
    // Add devices which are permanetly attached.
    keyboard = (InputDeviceKeyboard*)-1;
    keyboard = new InputDeviceKeyboard;
    
    // Add all joysticks found.
    SDL_InitSubSystem(SDL_INIT_JOYSTICK);
    int number_of_joysticks = SDL_NumJoysticks();
    DEBUG_MESSAGE("Using " << number_of_joysticks << " joysticks. ");
    joysticks.reserve(number_of_joysticks);
    for (int joystick_index = 0; joystick_index < number_of_joysticks;
         joystick_index++)
    {
        add_device(*(new InputDeviceJoystick(joystick_index)));
        DEBUG_MESSAGE("Using joystick \"" << SDL_JoystickName(joystick_index));
    }
    
    /// @todo check for notifications from new usb & bluetooth devices
}

InputHandler::~InputHandler()
{
    delete keyboard;
    for (std::vector<InputDeviceJoystick *>::iterator it = joysticks.begin();
         it != joysticks.end();
         it++)
    {
        delete *it;
    }
    joysticks.clear();
}

InputHandler & InputHandler::get_instance()
{
    static InputHandler input_handler;
    input_handler.init();
    return input_handler;
}

InputHandler::iterator InputHandler::add_device(InputDevice & device)
{
    if (scene)
    {
        device.set_scene(*scene);
    }
    return devices.insert(&device).first;
}

void InputHandler::remove_device(InputHandler::iterator it)
{
    devices.erase(it);
}

void InputHandler::remove_device(InputDevice & device)
{
    devices.erase(&device);
}

void InputHandler::set_scene(Scene & scene_in)
{
    scene = &scene_in;
    for (iterator it = devices.begin(); it != devices.end(); it++)
    {
        (*it)->set_scene(*scene);
    }
}

void InputHandler::poll()
{
    // poll all devices that need polling for events.
    for (iterator it = devices.begin(); it != devices.end(); it++)
    {
        (*it)->poll();
    }
}

InputHandler::iterator InputHandler::begin()
{
    return devices.begin();
}

InputHandler::iterator InputHandler::end()
{
    return devices.end();
}

std::size_t InputHandler::get_number_of_devices() const
{
    return devices.size();
}


}
