/** @file InputDeviceKeyboard.cpp
 *  @brief Implement the Engine::InputDeviceKeyboard class.
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "InputDeviceKeyboard.h"
#include "InputReport.h"
#include "SDL.h"
#include <Debug.h>
#include "GameObjects/Car.h"

namespace Engine
{

InputDeviceKeyboard::InputDeviceKeyboard()
    :   accelerate_down(false), // assume keys are not held during startup.
        break_down(false),
        steer_left_down(false),
        steer_right_down(false),
        slide_left_down(false),
        slide_right_down(false)
{
    
}

InputDeviceKeyboard::~InputDeviceKeyboard()
{
    
}

int InputDeviceKeyboard::get_acceleration()
{
    if (accelerate_down && !break_down) return 32767;
    else if (break_down && !accelerate_down) return -32767;
    else return 0;
}

int InputDeviceKeyboard::get_slide()
{
    if (slide_left_down && !slide_right_down) return -32767;
    else if (slide_right_down && !slide_left_down) return 32767;
    else return 0;
}

int InputDeviceKeyboard::get_steering()
{
    if (steer_left_down && !steer_right_down) return -32767;
    else if (steer_right_down && !steer_left_down) return 32767;
    else return 0;    
}

void InputDeviceKeyboard::poll()
{
    SDL_Event event;
    SDL_PumpEvents();
    while (SDL_PeepEvents
            (&event, 1, SDL_GETEVENT,
                SDL_EVENTMASK(SDL_KEYUP) | SDL_EVENTMASK(SDL_KEYDOWN)
             ) > 0
          )
    {
        bool new_state = event.type == SDL_KEYDOWN;
        InputReport::ReportType report_type = InputReport::RT_MAX;
        int value(0);
        switch (event.key.keysym.sym)
        {
#ifdef PANDORA
	    case SDLK_HOME:
                report_type = InputReport::RT_CHANGE_ACCEL;
                accelerate_down = new_state;
                value = get_acceleration();
		report(report_type, value);
#endif
            case SDLK_RETURN:
            case SDLK_SPACE:
                report_type = InputReport::RT_MENU_SELECT;
		if (!new_state) return;
                break;

            case SDLK_q:
                report_type = InputReport::RT_CHANGE_SLIDE;
                slide_left_down = new_state;
                value = get_slide();
                break;
            case SDLK_e:
                report_type = InputReport::RT_CHANGE_SLIDE;
                slide_right_down = new_state;
                value = get_slide();
                break;
	    case SDLK_UP:
		report_type = InputReport::RT_MENU_UP;
		if (new_state) report(report_type, value);
#ifdef PANDORA
	    case SDLK_PAGEDOWN:
#endif
            case SDLK_w:
                report_type = InputReport::RT_CHANGE_ACCEL;
                accelerate_down = new_state;
                value = get_acceleration();
                break;
	    case SDLK_DOWN:
		report_type = InputReport::RT_MENU_DOWN;
		if (new_state) report(report_type, value);
#ifdef PANDORA
	    case SDLK_END:
	    case SDLK_PAGEUP:
#endif
            case SDLK_s:
                report_type = InputReport::RT_CHANGE_ACCEL;
                break_down = new_state;
                value = get_acceleration();
                break;
	    case SDLK_LEFT:
		report_type = InputReport::RT_MENU_LEFT;
		if (new_state) report(report_type, value);
            case SDLK_a:
                report_type = InputReport::RT_CHANGE_STEERING;
                steer_left_down = new_state;
                value = get_steering();
                break;
	    case SDLK_RIGHT:
		report_type = InputReport::RT_MENU_RIGHT;
		if (new_state) report(report_type, value);
            case SDLK_d:
                report_type = InputReport::RT_CHANGE_STEERING;
                steer_right_down = new_state;
                value = get_steering();
                break;
#ifdef PANDORA
	    case SDLK_LALT:
#endif
            case SDLK_ESCAPE:
            case SDLK_BACKSPACE:
                report_type = InputReport::RT_MENU_BACK;
		if (!new_state) return;
                break;
            default:
                // unhandled key. Don't report any events.
                return;
                break;
        }
        // Update the scene with a report for the keypress.
        report(report_type, value);
    }
}

}
