/** @file InputDeviceJoystick.cpp
 *  @brief Implement the Engine::InputDeviceJoystick class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "InputDeviceJoystick.h"
#include <Debug.h>
#include "GameObjects/Car.h"

namespace Engine
{

InputDeviceJoystick::InputDeviceJoystick(int index)
    :   joystick_index(index)
    ,   steering(0)
    ,   acceleration(0)
    ,   slide(0)
    ,   slide_left_down(false)
    ,   slide_right_down(false)
{
    SDL_JoystickEventState(SDL_ENABLE);
    assert(SDL_NumJoysticks() > index);
    sdl_joystick_handle = SDL_JoystickOpen(joystick_index);
}

InputDeviceJoystick::~InputDeviceJoystick()
{
    if (SDL_JoystickOpen(joystick_index))
    {
        SDL_JoystickClose(sdl_joystick_handle);
    }
}

int InputDeviceJoystick::get_steering()
{
    return steering;
}

int InputDeviceJoystick::get_acceleration()
{
    return acceleration;
}

int InputDeviceJoystick::get_slide()
{
    if (slide_left_down && !slide_right_down) return -32767;
    else if (slide_right_down && !slide_left_down) return 32767;
    else return slide;
}

void InputDeviceJoystick::poll()
{
    SDL_Event event;
    SDL_PumpEvents();
    
    // Messages about other events to put back on the queue when we've checked
    // them all.
    std::vector<SDL_Event> rejects;
    
    while (SDL_PeepEvents
            (&event, 1, SDL_GETEVENT,
                SDL_EVENTMASK(SDL_JOYAXISMOTION) |
                SDL_EVENTMASK(SDL_JOYBUTTONDOWN) |
                SDL_EVENTMASK(SDL_JOYBUTTONUP)
             ) > 0
          )
    {
        if (event.type == SDL_JOYAXISMOTION)
        {
            if (event.jaxis.which != joystick_index)
            {
                // This event is for a different joystick device.
                // add it to the rejects queue.
                rejects.push_back(event);
            }
            else
            {
                // an axis changed.
                int new_value = event.jaxis.value;
                switch (event.jaxis.axis)
                {
                    case 0:
                        // turn left or right.
                        report(InputReport::RT_CHANGE_STEERING, new_value);
                        if ((steering >= -16384) && (new_value < -16384))
                        {
                            report(InputReport::RT_MENU_LEFT);
                        }
                        if ((steering < 16384) && (new_value >= 16384))
                        {
                            report(InputReport::RT_MENU_RIGHT);
                        }
                        steering = new_value;
                        break;
                    case 1:
                        // acceleration
                        if (new_value == -32768) new_value = -32767;
                        new_value = -new_value;
                        report(InputReport::RT_CHANGE_ACCEL, new_value);
                        // if it crosses the half way point consider it a menu selection too.
                        if ((acceleration >= -16384) && (new_value < -16384))
                        {
                            report(InputReport::RT_MENU_DOWN);
                        }
                        if ((acceleration < 16384) && (new_value >= 16384))
                        {
                            report(InputReport::RT_MENU_UP);
                        }
                        acceleration = new_value;
                        break;
                    case 2:
                        // slide
                        if (new_value == -32768) new_value = -32767;
                        report(InputReport::RT_CHANGE_SLIDE, new_value);
                        slide = new_value;
                        break;
                    default:
                        // unused axis
                        break;
                }
            }
        }
        else
        {
            if (event.jbutton.which != joystick_index)
            {
                // This event is for a different joystick device.
                // add it to the rejects queue.
                rejects.push_back(event);
            }
            else
            {
                bool new_state = event.type == SDL_JOYBUTTONDOWN;
                switch (event.jbutton.button)
                {
                    case 0:
                    case 2:
                        if (new_state)
                        {
                            report(InputReport::RT_MENU_SELECT);
                        }
                        break;
                    case 1:
                    case 3:
                        if (new_state)
                        {
                            report(InputReport::RT_MENU_BACK);
                        }
                        break;
                    case 4:
                        slide_left_down = new_state;
                        report(InputReport::RT_CHANGE_SLIDE, get_slide());
                        break;
                    case 5:
                        slide_right_down = new_state;
                        report(InputReport::RT_CHANGE_SLIDE, get_slide());
                        break;
                    default:
                        // unhandled button. Don't report any events.
                        break;
                }
            }
        }
    }
    // put messages about other joysticks back on the event queue.
    for (std::vector<SDL_Event>::iterator it = rejects.begin();
        it != rejects.end(); it++)
    {
        SDL_PushEvent(&(*it));
    }
}

}
