/** @file InputDeviceAI.cpp
 *  @brief Implement the Engine::InputDeviceAI class.
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "InputDeviceAI.h"
#include <Debug.h>
#include "GameObjects/Car.h"

namespace Engine
{

/** A input device for a computer player.
 * Simulated events occur when an AI decides.
 */
InputDeviceAI::InputDeviceAI()
    :   acceleration(0)
    ,   slide(0)
    ,   steering(0)
    ,   ann(fann_create_from_file("data/ai.net"))
{
    fann_set_activation_function_output(ann, FANN_SIGMOID_SYMMETRIC_STEPWISE);
    fann_set_activation_function_hidden(ann, FANN_SIGMOID_SYMMETRIC_STEPWISE);
}

InputDeviceAI::~InputDeviceAI()
{
    fann_destroy(ann);
}

int InputDeviceAI::get_acceleration()
{
    return acceleration;
}

int InputDeviceAI::get_slide()
{
    return slide;
}

int InputDeviceAI::get_steering()
{
    return steering;
}

void InputDeviceAI::poll()
{
    fann_type sensors[Engine::GameObjects::Car::NUM_SENSORS] ;
    assert(car);
    car->get_sensors(sensors);
    /** @todo Blend with some more manual behaviour as the car approaches the
     * boundary of the nav mesh.
     * This behaviour should check all directions, not just forwards.
     * It should help most with narrow paths.
     * Possible solution:
     * Pick slide based on left and right sensors to move towards the middle.
     * Weighted blend for slide between neural network using minimum sideways
     * distance.
     * Pick stearing based on front left and front right sensors towards middle.
     * Weighted blend for stearing based on minimum distance on forward diagonals.
     * Pick acceleration based on forwards and backwards sensors towards biggest space.
     * Weighted blend for acceleration with neural netowrk using minimum y
     * distance.
     * If the car is facing backwards, replace steering.
     */
    const btScalar min_edge_distance = 0.05; // 5cm or less is full avoision
    const btScalar max_edge_distance = 0.8; // 80cm or more is full ANN.
    const btScalar edge_distance_range = max_edge_distance - min_edge_distance;
    // sensors have the inverse of the distance.
    const btScalar min_edge_sensor = 1.0 / min_edge_distance;
    const btScalar max_edge_sensor = 1.0 / max_edge_distance;
    // slide to avoid edges
    btScalar left_sensor = sensors[Track::SENSOR_COUNT * 1 + Track::SENSOR_NAVIGABLE_DISTANCE];
    btScalar right_sensor = sensors[Track::SENSOR_COUNT * 2 + Track::SENSOR_NAVIGABLE_DISTANCE];
    btScalar edge_avoid_slide = 0;
    btScalar slide_avoid_weight = 0;
    if (left_sensor > right_sensor)
    {
        edge_avoid_slide = -32767;
        slide_avoid_weight = left_sensor;
    }
    else if (right_sensor > left_sensor)
    {
        edge_avoid_slide = 32767;
        slide_avoid_weight = right_sensor;
    }
    // change the weights to give sensible values between 0 and 1.
    if (slide_avoid_weight > min_edge_sensor || !std::isfinite(slide_avoid_weight))
    {
        slide_avoid_weight = 1.0;
    }
    else if (slide_avoid_weight < max_edge_sensor)
    {
        slide_avoid_weight = 0.0;
    }
    else
    {
        slide_avoid_weight = 1.0 - ((1.0 / slide_avoid_weight) - min_edge_distance) / edge_distance_range;
    }
    
    // steer to avoid edges
    btScalar front_left_sensor = sensors[Track::SENSOR_COUNT * 3 + Track::SENSOR_NAVIGABLE_DISTANCE];
    btScalar front_right_sensor = sensors[Track::SENSOR_COUNT * 4 + Track::SENSOR_NAVIGABLE_DISTANCE];
    btScalar edge_avoid_steer = 0;
    btScalar steer_avoid_weight = 0;
    if (front_left_sensor > front_right_sensor)
    {
        edge_avoid_steer = -32767;
        steer_avoid_weight = front_left_sensor;
    }
    else if (front_left_sensor < front_right_sensor)
    {
        edge_avoid_steer = 32767;
        steer_avoid_weight = front_right_sensor;
    }
    // adjust weight to sensible value between 0 and 1.
    if (steer_avoid_weight > min_edge_sensor)
    {
        steer_avoid_weight = 1.0;
    }
    else if (steer_avoid_weight < max_edge_sensor)
    {
        steer_avoid_weight = 0.0;
    }
    else
    {
        steer_avoid_weight = 1.0 - ((1.0 / steer_avoid_weight) - min_edge_distance) / edge_distance_range;
    }
    
    // accelerate/reverse to avoid going straight forwards or backwards over the edge.
    btScalar front_sensor = sensors[Track::SENSOR_NAVIGABLE_DISTANCE];
    btScalar back_sensor = sensors[Track::SENSOR_NAVIGABLE_DISTANCE + (GameObjects::Car::NUM_SENSORS_FEELERS -1)* Track::SENSOR_COUNT];
    btScalar edge_avoid_accel = 0;
    btScalar accel_avoid_weight = 0;
    if (front_sensor > back_sensor)
    {
        edge_avoid_accel = -32767;
        accel_avoid_weight = front_sensor;
    }
    else if (front_sensor < back_sensor)
    {
        edge_avoid_accel = 32867;
        accel_avoid_weight = back_sensor;
    }
    // adjust weight to sensible value between 0 and 1.
    if (accel_avoid_weight > min_edge_sensor)
    {
        accel_avoid_weight = 1.0;
    }
    else if (accel_avoid_weight < max_edge_sensor)
    {
        accel_avoid_weight = 0.0;
    }
    else
    {
        accel_avoid_weight = 1.0 - ((1.0 / accel_avoid_weight) - min_edge_distance) / edge_distance_range;
    }
    
    #if 0
    if (sensors[Track::SENSOR_LAP_DIFFERENTIAL] > 0.0)
    {
    #endif
        fann_type *output;
        output = fann_run(ann, sensors);
        
        for (int i = 0; i < 4; i++)
        {
            output[i] *= 32767.0;
            if (output[i] > 32767) output[i] = 32767;
            if (output[i] < -32767) output[i] = -32767;
        }
        //weighted blend outputs for edge avoision
        output[0] = output[0] * (1.0 - accel_avoid_weight) + edge_avoid_accel * accel_avoid_weight;
        output[1] = output[1] * (1.0 - slide_avoid_weight) + edge_avoid_slide * slide_avoid_weight;
        output[2] = output[2] * (1.0 - steer_avoid_weight) + edge_avoid_steer * steer_avoid_weight;
        if (int(output[0]) != acceleration)
        {
            acceleration = int(output[0]);
            report(InputReport::RT_CHANGE_ACCEL, output[0]);
        }
        
        if (int(output[1]) != slide)
        {
            slide = int(output[1]);
            report(InputReport::RT_CHANGE_SLIDE, output[1]);
        }
        
        if (int(output[2]) != steering)
        {
            steering = int(output[2]);
            report(InputReport::RT_CHANGE_STEERING, output[2]);
        }
    #if 0
    } else if (std::isfinite(sensors[Track::SENSOR_LAP_DIFFERENTIAL]))
    {
        // facing backwards.
        // try to slow down if going forwards
        if (sensors[Engine::GameObjects::Car::NUM_SENSORS_FEELERS * Engine::GameObjects::Car::NUM_FEELER_PROPERTIES + 1] > 0.0)
        {
            acceleration = -32767;
        } else {
            acceleration = 0;
        }
        report(InputReport::RT_CHANGE_ACCEL, acceleration);
        report(InputReport::RT_CHANGE_SLIDE, 0);
        // turn around.
        int ts = sensors[Engine::GameObjects::Car::NUM_SENSORS - 1]*32767;
        if (ts != steering)
        {
            steering = ts;
            report(InputReport::RT_CHANGE_STEERING, ts);
        }
        //DEBUG_MESSAGE("Oh no! Wrong way!");
    } else {
        // driven over the boundary
        if (std::isfinite(sensors[Engine::GameObjects::Car::NUM_FEELER_PROPERTIES * (Engine::GameObjects::Car::NUM_SENSORS_FEELERS - 1) + Track::SENSOR_LAP_DIFFERENTIAL]))
        {
            // backwards is safe
            // reverse hard to reduce change of falling off.
            report(InputReport::RT_CHANGE_ACCEL, -32767);
        }
        if (std::isfinite(sensors[Engine::GameObjects::Car::NUM_FEELER_PROPERTIES * 1 + Track::SENSOR_LAP_DIFFERENTIAL]))
        {
            // +x is available
            report(InputReport::RT_CHANGE_SLIDE, 32767);
            report(InputReport::RT_CHANGE_STEERING, 32767);
        }
        else if (std::isfinite(sensors[Engine::GameObjects::Car::NUM_FEELER_PROPERTIES * 2  + Track::SENSOR_LAP_DIFFERENTIAL]))
        {
            // -x is available
            report(InputReport::RT_CHANGE_SLIDE, -32767);
            report(InputReport::RT_CHANGE_STEERING, -32767);
        }
        //DEBUG_MESSAGE("Argh, I've driven over the edge!");
    }
    #endif
}

void InputDeviceAI::save()
{
    fann_save(ann, "data/ai.net");
}

void InputDeviceAI::mutate()
{
    // set a few connection weights to random values.
    int num_connections = fann_get_total_connections(ann);
    fann_connection cs[num_connections];
    fann_get_connection_array(ann, cs);
    
    // decide how many variables to mutate
    float v = float(rand())/float(RAND_MAX);
    // raise it to the power of three so we get more lower numbers.
    v *= v*v;
    unsigned int variables_mutated = 1 + int(v * (num_connections - 1));
    
    // mutate the variables
    for (unsigned int i = 0; i < variables_mutated; i++)
    {
        int w = rand()/(RAND_MAX/num_connections);
        btScalar scale = std::pow(2.0, rand() / (RAND_MAX / 15) - 5);
        fann_set_weight(ann, cs[w].from_neuron, cs[w].to_neuron,
                float(rand())/float(RAND_MAX)*scale - scale/2.0);
    }
}

void InputDeviceAI::cross(const InputDeviceAI & p1, const InputDeviceAI & p2)
{
    unsigned int num_connections = fann_get_total_connections(ann);
    fann_connection c2[num_connections];
    fann_get_connection_array(p2.ann, c2);
    for (unsigned int i = 0; i < num_connections; i++)
    {
        if (rand()/(RAND_MAX/2))
        {
            fann_set_weight(ann, c2[i].from_neuron, c2[i].to_neuron, c2[i].weight);
        }
    }
}

} // namespace Engine
