/** @file InputDevice.h
 *  @brief Declare the Engine::InputDevice class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef INPUTDEVICE_H_
#define INPUTDEVICE_H_

#include "Scene.h"

namespace Engine
{

namespace GameObjects
{
    class Car;
}
/** A keyboard, mouse, gamepad, joystick, wiimote, player on the other end
 * of a network, or AI.
 * 
 * This is where input device abstraction is done. Different types of
 * InputDevices can be created that are subclasses of this class, then anything
 * wishing to get input from any type of input device can use a consistant
 * interface.
 */
class InputDevice
{
public:
    InputDevice();
    virtual ~InputDevice();
    /** Get a number representing the acceleration or breaking.
     * 
     * @return A number from -32767 to 32767 giving the amount of accelerating
     * or breaking to do: 0 if not accelerating or braking, a negative value
     * for breaking, a positive value for acceleration.
     */
    virtual int get_acceleration() = 0;
    /** Sideways force .
     * 
     * @return a number between -32767 (maximum left) and +32767 (maximum
     * right). 0 indicates no sliding.
     */
    virtual int get_slide() = 0;
    /** Stearing angle
     * @return a number between -32767 (maximum left) and 32767 (maximum
     * right). 0 is straight ahead.
     */
    virtual int get_steering() = 0;
    
    /** Poll for events, if necessary.
     */
    virtual void poll() {};
    
    virtual void set_scene(Scene & scene);
    
    virtual void set_car(GameObjects::Car * car);
    
    std::set<InputDevice *>::iterator get_handle() const;
protected:
    Scene * scene;
    GameObjects::Car * car;
    std::set<InputDevice *>::iterator handler_reference;
    /** Make a report, then submit it the car and scene as required.
     * @param type The classification of the event that occured.
     * @param value The new value for steering, slide, and acceleration changed events.
     */
    void report(InputReport::ReportType type, int value = 0);
};

}

#endif /*INPUTDEVICE_H_*/
