/** @file GameScene.h
 *  @brief Declare the Engine::GameScene class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef GAMESCENE_H_
#define GAMESCENE_H_

#include "Scene.h"
#include "../Graphics/SkyBox.h"
#include "../Graphics/SkyParticles.h"
#include "Physics/World.h"
#include "GameObjects/Car.h"
#include "CarCamera.h"
#include "InputHandler.h"
#include "../UI/PauseMenu.h"

#include <libtrack/OcclusionTester.h>
#include <libtrack/Track.h>
#include <libtrack/Mesh/BulletMesh.h>

#include <sstream>

namespace Engine
{

class GameScene : public Engine::Scene
{
public:
    /**  Create a game where the players use specified input devices.
     * @param input_devices InputHandler iterators to the input devices,
     * paired with the car model to use for each player.
     * @param track The track to play on.
     */
    GameScene(std::vector<std::pair<InputHandler::iterator, unsigned int> > input_devices,
              const Track::Track & track);
    
    virtual ~GameScene();
    
    /** Take input from an input device.
     * @param report An input report, which may be freed immediately after the
     * call returns.
     */
    virtual void take_input(InputReport & report);
    
    /** Process any status changes that occur due to elapsed time.
     * @param milliseconds_elapsed time in milliseconds since last update_logic.
     */
    virtual void update_logic(unsigned int milliseconds_elapsed);
    
    /** Draw the scene using OpenGL commands.
     * Must go from any state to ready for a buffer swap.
     */
    virtual void draw();
    
    /** Update the audio.
     */
    virtual void do_sound();
    
    /// Get the physics world used for the scene.
    Physics::World & get_world();
    
    /** Turn on or off replay saving.
     * Replay saving is turned on by default.
     * When the scene is destroyed, the replay can be saved to a file called
     * last replay.
     * @param value true to save replays, false otherwise.
     */
    void set_save_replay(bool value = true);
private:
    /// Input devices used for cars, in the same order as cars.
    std::vector<std::pair<InputHandler::iterator, unsigned int> > m_input_devices;
    std::stringstream m_replay_header;
    Graphics::SkyBox sky;
    Graphics::SkyParticles m_sky_particles;
    Track::OcclusionTester occlusion_tester;
    std::vector<GameObjects::Car *> cars;
    /// Indices of the human players in cars and m_input_devices.
    std::vector<unsigned int> m_humans;
    /// For each human player, true if they tried to skip the pause before quitting.
    std::vector<bool> car_skip;
    /// Cameras for each human player's car.
    std::vector<CarCamera *> car_cameras;
    btRigidBody * track_body;
    btRigidBody * floor_body;
    boost::shared_ptr<btCollisionShape> track_shape;
    boost::shared_ptr<btCollisionShape> floor_shape;
    
    /// The course to play on
    const Track::Track & track;
    
    Physics::World world;
    
    /// Save replays when the scene is destroyed?
    bool save_replay;
    
    /// frequency of rendering in hertz.
    /// Calcualted from time between last two frames.
    float fps;
    
    /** The ranking of the cars.
     * car_ranks[i] is the rank of the ith car. Rank 0 is first.
     */
    std::vector<int> car_ranks;
    
    /// Draw the overlay that appears when the game is paused.
    void draw_paused_screen();
    /**Draw the window for the specified player. Viewport must be set up first.
     *  @param player the index in the car_cameras array to use.
     *  @param aspect The aspect ratio (width divided by height) of the
     *  viewport.
     */
    void draw_for_player(unsigned int player, float aspect);
    void draw_world(unsigned int player, float aspect);
    void draw_hud(unsigned int player);
    void draw_speedometer(unsigned int player);
    void draw_lap_info(unsigned int player);
    void draw_rank(unsigned int player);
    /// Write 'Disqualified' in the middle of a player's viewport.
    void draw_disqualified(unsigned int player);
    /// Write 'Finished' in the middle of a player's viewport.
    void draw_finished(unsigned int player);
    /// Show the user they are facing the wrong way.
    void draw_reverse_sign(unsigned int player);
    /// Show the countdown at the begining of the race
    void draw_countdown();
    
    /// Set car_ranks.
    void rank_cars();
    
    /// The centre of the starting plane
    btVector3 start_point;
    /// A normalized vector in the direction perpendicular to the starting plane.
    btVector3 start_plane_normal;
    /// The signed distance between the origin and and the starting plane.
    btScalar start_plane_distance;
    
    /// The texture with the speedometer parts.
    Track::Texture * m_speedometer_texture;
    /// The texture with the best lap icon.
    Track::Texture * m_best_lap_texture;
    /// The texture for the border above and below the finish message
    Track::Texture * m_finish_border_texture;
    /// The texture that says "Finished"
    Track::Texture * m_finish_text_texture;
    /// The texture that says "Disqualified"
    Track::Texture * m_disqualified_text_texture;
    /// The texture for the u-turn sign that appears when facing the wrong way
    Track::Texture * m_reverse_texture;
    /// The texture for the countdown at the start of the race
    Track::Texture * m_countdown_texture;
    
    /** Millisecond timer to the start of the race.
     * >= 0 implies race has started.
     */
    int countdown_timer;
    
    /// True if the game is paused.
    bool paused;
    /// The menu to display when paused.
    UI::PauseMenu m_pause_menu;
    /// Unpause the game and reset the pause menu.
    void unpause();
    
    bool show_debug;

#ifndef NDEBUG
    /// Display list for debug draw of the track.
    unsigned int m_debug_list_name;
    /// Texture name for the navigation texture used for debug draw.
    unsigned int m_debug_texture_name;
    Track::DrawableMesh ai_mesh;
    
    /// number of frames processed since the start of the game.
    unsigned int m_total_frames;
    /// number of milliseconds elapsed since the start of the game.
    unsigned int m_total_time;
#endif
};

}

#endif /*GAMESCENE_H_*/
