/** @file GameScene.cpp
 *  @brief Implement the Engine::GameScene class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "GameScene.h"
#include "../Graphics/Window.h"
#include <Debug.h>
#include "InputHandler.h"
#include "InputDeviceAI.h"
#include "ResourceHandler.h"
#include "../UI/BasicFonts.h"
#include "../MainLoop.h"
#include "Audio.h"

#include <libtrack/Mesh/MultiDrawableMesh.h>
#include <libtrack/Texture.h>
#include <libtrack/StartingPosition.h>

#include <cmath>
#include <fstream>
#include <limits>

#include <glibmm/miscutils.h>

#ifndef HAVE_GLES
#include <GL/glu.h>
#include <config.h>
#else
#include <GLES/gl.h>
#include "../Graphics/eglport.h"
#define GLdouble     GLfloat
#define glOrtho      glOrthof
#endif
#ifdef HAVE_FTGL_2_1_2
typedef FTGLTextureFont FTTextureFont;
#endif

namespace Engine
{

/// stores shared textures, indexed by a string name
typedef ResourceHandler<Track::Texture, std::string, std::string> TextureStore;

GameScene::GameScene(std::vector<std::pair<InputHandler::iterator, unsigned int> > input_devices,
                     const Track::Track & track)
    :   m_input_devices(input_devices)
    ,   sky(track.get_theme().get_skybox())
    ,   track(track)
    ,   world(track)
    ,   save_replay(true)
    ,   fps(0)
    ,   countdown_timer(-3000)
    ,   paused(false)
    ,   show_debug(false)
#ifndef NDEBUG
    // prepare ai_mesh's display list so we can call it recusively.
    ,   ai_mesh(track.get_ai_mesh())
    ,   m_total_frames(0)
    ,   m_total_time(0)
#endif
{
    // make sure textures randomly accessed by the hud are cached before the
    // game starts, to prevent mini freezes while the textures are loaded.
    // These remain in memory after the GameScene ends, but are reused next
    // time without loading them again.
    TextureStore & tex_store = TextureStore::get_instance();
    #define tex_load(bind_name, file_name, variable)\
                tex_store.check_load(bind_name, file_name);\
                variable = tex_store.get(bind_name);\
                variable->make_cache()
    tex_load("speedometer.hud.ui", "data/ui/hud/speedometer.png", m_speedometer_texture);
    tex_load("best_lap.hud.ui", "data/ui/hud/best_lap.png", m_best_lap_texture);
    tex_load("border.finish.hud.ui", "data/ui/hud/finish_border.png", m_finish_border_texture);
    tex_load("text.finish.hud.ui", "data/ui/hud/finished_text.png", m_finish_text_texture);
    tex_load("disqualified.hud.ui", "data/ui/hud/disqualified_text.png", m_disqualified_text_texture);
    tex_load("reverse.hud.ui", "data/ui/hud/reverse.png", m_reverse_texture);
    tex_load("countdown.hud.ui", "data/ui/hud/countdown.png", m_countdown_texture);
    #undef tex_load
    /** @todo Replays should save the version since they are not compatible
     * with other versions.
     * Any minor tweak in game mechanics, car paramaters, or courses will
     * break the replay. We don't actually want compatibility, a replay from
     * an old version does not reflect what is possible in a current version,
     * so it is meaningless.
     */
    // first line contains the track's filename
    m_replay_header << track.get_filename() << std::endl;
    // The number of cars.
    m_replay_header << input_devices.size() << " ";
    // We add information about each car as we set it up
    
    // Put cars in the starting positions.
    const Track::Path & path = track.get_path();
    const Track::PathEdge & edge = path.get_edge(path.get_starting_edge());
    start_point = path.find_start_position();
    path.find_start_plane(start_plane_normal, start_plane_distance);
    
    const std::size_t number_of_cars = input_devices.size();
    cars.reserve(number_of_cars);
    /** @todo Don't make cameras for AI cars. */
    car_cameras.reserve(number_of_cars);
    for(unsigned int i = 0; i< number_of_cars; i++)
    {
        // Find the starting position for this car.
        btTransform initial_transform =
            // guess transformation when starting position cannot be found.
            edge.get_transform(float(i) / float(number_of_cars));
        // find the starting position track attachment for this car's rank.
        for (std::vector<boost::shared_ptr<Track::TrackAttachment> >::const_iterator it = edge.get_attachments().begin();
             it != edge.get_attachments().end();
             it++)
        {
            const Track::StartingPosition * pos = dynamic_cast<const Track::StartingPosition *>(&(**it));
            if (pos)
            {
                // cars go in reverse order, so human players go last.
                if (pos->get_rank() == number_of_cars - i - 1)
                {
                    initial_transform = pos->get_global_transform();
                }
            }
        }
        // raise the car above the track slightly.
        initial_transform.setOrigin(initial_transform(btVector3(0.0, 0.0, 0.2)));
        cars.push_back(new GameObjects::Car(world,
                                            initial_transform,
                                            *(input_devices[i].first),
                                            input_devices[i].second,
                                            start_plane_normal,
                                            start_plane_distance,
                                            start_point));
        if (!dynamic_cast<InputDeviceAI*>(*(input_devices[i].first)))
        {
            // Not a computer player (though it could be a replay of one).
            /** @todo In a replay, give one view of a car, and allow switching
             * the car that is visible.
             */
            car_cameras.push_back(new CarCamera(*cars[i], world));
            car_skip.push_back(false);
            m_humans.push_back(i);
        }
        // The pointer to the input device will be used to identify input
        // reports during the game. Store it as a long int instead of hex.
        m_replay_header << (unsigned long int)&*(input_devices[i].first) << " "
        // Record which car model was used.
                        << input_devices[i].second << " ";
        
    }
    btDefaultMotionState motion_state(btTransform(btQuaternion(0, 0, 0, 1),
                                        btVector3(0, 0, 0)));
    track_shape = track.get_collision_shape();
    btRigidBody::btRigidBodyConstructionInfo
                rigid_body_CI(btScalar(0),
                              &motion_state,
                              &(*track_shape),
                              btVector3(0, 0, 0));
    rigid_body_CI.m_restitution = 1.0;
    track_body = new btRigidBody(rigid_body_CI);
    world.get_dynamics_world().addRigidBody(track_body);
    
    // now do the floor shape in the floor world.
    floor_shape = track.get_floor_shape();
    rigid_body_CI.m_collisionShape = &(*floor_shape);
    floor_body = new btRigidBody(rigid_body_CI);
    world.get_floor_world().addCollisionObject(floor_body);
    
    // stage specific light and fog.
    track.get_lighting().initalise();
    
    
#ifndef NDEBUG
    // Create a display list to show the debug version of the track.
    // This shows the navigation mesh, textured by lap position, with the
    // connections between faces drawn over the top.
    
    // Set up texture
    GLubyte t[16] = {0, 15, 31, 47, 63, 79, 95, 111, 127, 143, 159, 175, 191, 207, 223, 239};
    glGenTextures(1, &m_debug_texture_name);
    glBindTexture(GL_TEXTURE_1D, m_debug_texture_name);
    gluBuild1DMipmaps(GL_TEXTURE_1D, GL_LUMINANCE4, 16, GL_LUMINANCE, GL_UNSIGNED_BYTE, t);
    
    ai_mesh.make_cache();
    m_debug_list_name = glGenLists(1);
    glNewList(m_debug_list_name, GL_COMPILE);
        glPushAttrib(GL_CURRENT_BIT | GL_DEPTH_BUFFER_BIT | GL_ENABLE_BIT | GL_TEXTURE_BIT | GL_TRANSFORM_BIT);
            // navigation mesh
            glEnable(GL_TEXTURE_1D);
            glDisable(GL_TEXTURE_2D);
            glBindTexture(GL_TEXTURE_1D, m_debug_texture_name);
            glMatrixMode(GL_TEXTURE);
            glScalef(0.2, 1.0, 1.0);
            ai_mesh.draw();
            glLoadIdentity();
            glDisable(GL_TEXTURE_1D);
            
            // connection lines
            const Track::MeshFaces::Graph & graph = track.get_ai_graph();
            glDisable(GL_DEPTH_TEST);
            glBegin(GL_LINES);
                typedef boost::graph_traits<Track::MeshFaces::Graph>::edge_iterator EdgeIterator;
                std::pair<EdgeIterator, EdgeIterator> edge_range;
                for (edge_range = boost::edges(graph);
                     edge_range.first != edge_range.second;
                     edge_range.first++)
                {
                    const btVector3 & source = graph[boost::source(*(edge_range.first), graph)].face_centre;
                    const btVector3 & target = graph[boost::target(*(edge_range.first), graph)].face_centre;
                    glColor3f(1.0, graph[boost::source(*(edge_range.first), graph)].fv1.texture_coord_u / 1000.0, 0);
                    glVertex3f(source.x(), source.y(), source.z());
                    glVertex3f(target.x(), target.y(), target.z());
                }
            glEnd();
        glPopAttrib();
    glEndList();
#endif
}

GameScene::~GameScene()
{
    // record the replay if requested.
    if (save_replay)
    {
        try
        {
            std::ofstream out;
            out.exceptions(std::ofstream::badbit | std::ofstream::failbit);
            out.open((Glib::get_user_config_dir() + "/racer_last_replay").c_str());
            // Write metadata identifing the scene and cars
            out << m_replay_header.str();
            // Write the input device events that occured during the game.
            world.write_replay_events(out);
        } catch (std::ofstream::failure e)
        {
            DEBUG_MESSAGE("Error while writing replay file.");
        }
    }
    
    world.get_dynamics_world().removeRigidBody(track_body);
    delete track_body;
    world.get_floor_world().removeCollisionObject(floor_body);
    delete floor_body;
    // delete the cars' cameras
    for (std::vector<CarCamera *>::iterator i = car_cameras.begin();
         i != car_cameras.end(); i++)
    {
        delete *i;
    }
    // delete the cars
    for (std::vector<GameObjects::Car *>::iterator i = cars.begin();
         i != cars.end(); i++)
    {
        delete *i;
    }
    
#ifndef NDEBUG
    glDeleteLists(m_debug_list_name, 1);
    glDeleteTextures(1, &m_debug_texture_name);
    DEBUG_MESSAGE("Framerate: " << (float(m_total_frames) / float(m_total_time) * 1000.0));
#endif
}

void GameScene::take_input(InputReport & report)
{
    if (paused)
    {
        m_pause_menu.take_input(report);
        return;
    }
    switch (report.get_report_type())
    {
        case InputReport::RT_MENU_BACK:
#ifndef NDEBUG
            /* In debug builds, toggle debugging overlay when pressing escape
             * or backspace instead of pausing.
             */
            show_debug = !show_debug;
            break;
#endif
        case InputReport::RT_MENU_SELECT:
            {
                std::size_t car = -1;
                for (unsigned int i = 0; i < m_input_devices.size(); i++)
                {
                    if (m_input_devices[i].first == report.get_input_device())
                    {
                        car = i;
                        break;
                    }
                }
                if (car != std::size_t(-1))
                {
                    if (cars[car]->get_disqualified() || cars[car]->get_finished())
                    {
                        car_skip[car] = true;
                    } else {
                        paused = true;
                        Audio::get_instance().pause();
                    }
                } else {
                    // unused controller.
                    paused = true;
                    Audio::get_instance().pause();
                }
            }
            break;
        case InputReport::RT_DISCONNECT:
            // automatically pause the game if connection to a device is lost.
            paused = true;
        case InputReport::RT_BATTERY_LOW:
            /**@todo Give some sort of warning for low batteries.
             * We need direct wii remote support first.
             */
            break;
        default:
            // all car actions are passed to the car.
            break;
    }
}

void GameScene::update_logic(unsigned int milliseconds_elapsed)
{
#ifndef NDEBUG
    m_total_frames++;
    m_total_time += milliseconds_elapsed;
#endif
    fps = 1000.0 / float(milliseconds_elapsed);
    /*Set limits for quality adjustment.
     * We want to ensure a reasonable enough quality for the surrounding
     * track so the player can see the walls they are approching even
     * with a slow refresh rate.
     * Also, if facing empty space on a fast machine, the quality could
     * increase way to much, making it very slow to get back to sensible
     * value when they turn round.
     * Within the limits, quality is adjusted to target 60fps.
     */
    btScalar max_quality = 10000000000.0; // 10 billion
    ///@todo adjust this per theme?
    ///@todo Keep this proportional to viewport size?
    btScalar min_quality = 1.5;
    /**@todo See if the rate of adaption should be adjusted.
     * Too high, and it might cause a lot of flickering,
     * to low and it might cause slow refresh rates for too long.
     */
    if (fps > 60.0)
    {
         Track::global_quality *= 1.0 + (fps-60.0) * 0.015625;
         if (Track::global_quality > max_quality)
         {
             Track::global_quality = max_quality;
         }
    }
    else if (fps < 60.0)
    {
        Track::global_quality *= 1.0 - ((60.0 - fps) * 0.015625);
        if (Track::global_quality < min_quality)
        {
            Track::global_quality = min_quality;
        }
    }
#ifndef NDEBUG
    if (m_total_frames % 64 == 0) ///@todo remove- debug
    {
        DEBUG_MESSAGE("Quality level is " << Track::global_quality);
    }
#endif
    if (paused)
    {
        if (m_pause_menu.want_to_quit())
        {
            // escape or backspace hit while pause menu showing.
            unpause();
        } else {
            switch (m_pause_menu.get_response())
            {
                case UI::PauseMenu::R_NONE:
                    // wait for user to respond to pause menu.
                    break;
                case UI::PauseMenu::R_CONTINUE:
                    // continue the game.
                    unpause();
                    break;
                case UI::PauseMenu::R_QUIT:
                    // quit selected, then confirm pressed (space or enter).
                    // resume the paused audio so it functions correctly
                    // if something else wants to use it.
                    Audio::get_instance().resume();
                    // abort this game.
                    main_loop->pop_scene();
                    break;
            }
        }
        return;
    }
    if (track.get_theme().get_use_sky_particles())
    {
        m_sky_particles.update(milliseconds_elapsed);
    }
    if (countdown_timer <=500)
    {
        countdown_timer += milliseconds_elapsed;
        // don't do physics simulation until race starts.
        if (countdown_timer < 0) return;
    }
    world.update(milliseconds_elapsed);
    
    // if we are done with non computer controlled cars, quit the scene.
    bool quit = true;
    for (unsigned int i = 0; i < m_humans.size(); i++)
    {
        int car_index = m_humans[i];
        bool this_car = cars[car_index]->get_disqualified() ||
                        cars[car_index]->get_finished();
        if (this_car)
        {
            // done after a few seconds or if manually skipped.
            if (   (world.get_tick_number() - cars[car_index]->get_finish_time() < 300)
                && !car_skip[car_index])
            {
                quit = false;
            }
        }
        else
        {
            quit = false;
        }
    }
    if (quit)
    {
        main_loop->pop_scene();
    }
}

void GameScene::draw()
{
    // Find the correct ranking of cars to display.
    rank_cars();
    
    std::size_t camera_count = car_cameras.size();
    /** @todo only recalculate this when the screen size changes or on the
     * first run.
     */
    // find reasonable way to split the viewports.
    // We pick the layout with the aspect ratio closest to 4:3. Given multiple
    // choices, we use the layout with the most screen utilisation (least area
    // spent on unused viewports).
    unsigned int height = Graphics::Window::get_instance().get_height();
    unsigned int width = Graphics::Window::get_instance().get_width();
    // try each possible number of columns
    unsigned int best_rows(1);
    unsigned int best_columns(1);
    float best_utilisation(0.0);
    float best_aspect(0.0);
    float best_aspect_wrongness(height * width);
    for (unsigned int rows = 1; rows <= camera_count; rows++)
    {
        unsigned int columns = (camera_count + rows - 1) / rows;
        float aspect =   (float(width) / float(columns))
                       / (float(height) / float(rows));
        float utilisation = float(camera_count) / float(rows * columns);
        float aspect_wrongness = 2.35 - aspect;
        if (aspect_wrongness < 0.0)
        {
            aspect_wrongness = -aspect_wrongness;
        }
        bool better = false;
        if (best_aspect_wrongness > aspect_wrongness)
        {
            better = true;
        }
        else if (    best_aspect_wrongness == aspect_wrongness
                  && utilisation > best_utilisation)
        {
            // The aspect ratio is just as good, but this has less screen
            // space used for unused viewports.
            better = true;
        }
        if (better)
        {
            best_rows = rows;
            best_columns = columns;
            best_utilisation = utilisation;
            best_aspect = aspect;
            best_aspect_wrongness = aspect_wrongness;
        }
    }
    // now render each viewport with the choosen layout.
    glEnable(GL_SCISSOR_TEST);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_TEXTURE_2D);
    glEnable(GL_CULL_FACE);
    unsigned int row = 0;
    unsigned int column = 0;
    unsigned int row_size = height / best_rows;
    unsigned int column_size = width / best_columns;
    unsigned int player = 0;
    assert(best_aspect > 0);
    for (std::vector<CarCamera *>::iterator it = car_cameras.begin();
         it != car_cameras.end(); it++)
    {
        glViewport(column * column_size,  row * row_size,
                   column_size, row_size);
        glScissor(column * column_size,  row * row_size,
                  column_size, row_size);
        /** @todo 'player' is only going to work if the human players (with
         * cameras and viewports) are before all the computer players in the
         * car list.
         * Otherwise the HUD will refer to the wrong car.
         */
        draw_for_player(player, best_aspect);
        column++;
        player++;
        if (column == best_columns)
        {
            row++;
            column = 0;
        }
    }
    // blank the unused windows.
    
    while(player < best_rows * best_columns)
    {
        glViewport(column * column_size,  row * row_size,
                   column_size, row_size);
        glScissor(column * column_size,  row * row_size,
                  column_size, row_size);
        glClear(GL_COLOR_BUFFER_BIT);
        player++;
        column++;
        if (column == best_columns)
        {
            row++;
            column = 0;
        }
    }
    glDisable(GL_SCISSOR_TEST);
    
    // Indicate when paused
    if (paused)
    { 
        draw_paused_screen();
    }
    
    // show the fps counter in adebug build.
#ifndef NDEBUG
    char msg[10] = "f/s: 000";
    // take the average frames per second over the last 60 frames.
    const int number_of_readings = 60;
    static int readings[number_of_readings] = {0.0};
    static int which_reading = 0;
    static int total = 0;
    total += int(fps);
    total -= readings[which_reading];
    readings[which_reading] = int(fps);
    which_reading++;
    if (which_reading == number_of_readings)
    {
        which_reading = 0;
    }
    int fpsi = total > 999 * number_of_readings ? 999 : total / number_of_readings;
    msg[5] = fpsi / 100 + '0';
    msg[6] = (fpsi / 10) % 10 + '0';
    msg[7] = fpsi % 10 + '0';
    Graphics::Window::get_instance().set_ortho_projection();
    glLoadIdentity();
    glTranslatef(Graphics::Window::get_instance().get_width() - 150, 15, 0);
    FTTextureFont & font = UI::BasicFonts::get_instance().small_font;
    glDisable(GL_DEPTH_TEST);
    glColor4f(0.0f, 0.0f, 0.0f, 0.5f);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glDisable(GL_TEXTURE_2D);
    glBegin(GL_QUADS);
        glVertex2f(-5, -5);
        glVertex2f(50, -5);
        glVertex2f(50, 12);
        glVertex2f(-5, 12);
    glEnd();
    glEnable(GL_TEXTURE_2D);
    if (fpsi > 60)
    {
        glColor4f(0.0f, 1.0f, 0.0f, 1.0f); // green, good refresh rate
    }
    else if (fpsi > 45)
    {
        glColor3ub(255, 255, 0); // yellow, mediocre refresh rate.
    }
    else if (fpsi > 30)
    {
        glColor3ub(255, 127, 0); // orange, poor refresh rate
    }
    else
    {
        glColor3ub(255, 0, 0); // red, very bad refresh rate.
    }
    font.Render(msg);
    glColor3ub(255, 255, 255);
    glEnable(GL_DEPTH_TEST);
    glDisable(GL_BLEND);
    glLoadIdentity();
#endif
}

void GameScene::draw_paused_screen()
{
    glViewport(0,
               0,
               Graphics::Window::get_instance().get_width(),
               Graphics::Window::get_instance().get_height());
    Graphics::Window::get_instance().set_ortho_projection();
    glLoadIdentity();
    glDisable(GL_DEPTH_TEST);
    // darken screen to reduce burn in
    glDisable(GL_TEXTURE_2D);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glColor4f(0.0, 0.0, 0.0, 0.5);
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        float x = Graphics::Window::get_instance().get_width() + 1.0;
        float y = Graphics::Window::get_instance().get_height() + 1.0;
        glVertex2f(-1, -1);
        glVertex2f(x, -1);
        glVertex2f(x, y);
        glVertex2f(-1, y);
    glEnd();
#else
        float x = Graphics::Window::get_instance().get_width() + 1.0;
        float y = Graphics::Window::get_instance().get_height() + 1.0;
	GLfloat vtx1[] = {-1, -1,	x, -1,	x, y,	-1, y};
 
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(2, GL_FLOAT, 0, vtx1);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
	glDisableClientState(GL_VERTEX_ARRAY);
#endif
    // show menu in the middle of the screen.
    glEnable(GL_TEXTURE_2D);
    glColor4f(1, 1, 1, 1);
    glTranslatef(int(x/2), int(y/2), 0.0);
    glDisable(GL_CULL_FACE);
    m_pause_menu.draw();
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    glDisable(GL_BLEND);
}

void GameScene::draw_for_player(unsigned int player, float aspect)
{
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(65.0, aspect, 0.03, 1000.0);
    glMatrixMode(GL_MODELVIEW);
    glClear(GL_DEPTH_BUFFER_BIT);
    
    draw_world(player, aspect);
    
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    if (aspect > 1.0)
    {
        // screen wider than it is tall, add extra horizontal space
        glOrtho(-aspect, aspect, -1, 1, -1, 1);
    } else {
        // screen taller than it is wide, add extra vertical space.
        GLdouble aspect_inv = 1.0 / aspect;
        glOrtho(-1, 1, -aspect_inv, aspect_inv, -1, 1);
    }
    glMatrixMode(GL_MODELVIEW);
       
    draw_hud(player);
}

void GameScene::draw_world(unsigned int player, float aspect)
{
    glLoadIdentity();
    car_cameras[player]->full_transform();
    car_cameras[player]->update_occlusion_tester(occlusion_tester, aspect);
    
    // set up light and fog
    track.get_lighting().prepare_render(); 
    track.get_lighting().turn_on();
    if (!show_debug)
    {
        const Track::Path & path = track.get_path();
        path.conditional_draw(occlusion_tester);
    }
    
    // draw sky box using only the camera rotation
    // (no translation to fake infinite distance)
    track.get_lighting().turn_off();
    glPushMatrix();
        glLoadIdentity();
        car_cameras[player]->rotation_transform();
        sky.draw();
    glPopMatrix();    
    track.get_lighting().turn_on();
    
    // Cars
    // Need to be drawn over track & sky because of the blended engine flame.
    for (unsigned int i = 0; i < cars.size(); i++)
    { 
        cars[i]->conditional_draw(occlusion_tester);
    }
    if (track.get_theme().get_use_sky_particles())
    {
        m_sky_particles.draw(*(car_cameras[player]));
    }
    track.get_lighting().turn_off();
    // (Debug) draw the ai navigation graph.
#ifndef NDEBUG
    if (show_debug)
    {
        glCallList(m_debug_list_name);
    }
#endif
}

void GameScene::draw_hud(unsigned int player)
{
    // Heads up display overlayed over the game.
    // We leave a border of 0.1 (5%) around the screen edges to keep in the
    // safe area incase of playing on a TV.
    glLoadIdentity();
    glEnable(GL_BLEND);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_CULL_FACE);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    
    if (cars[player]->get_finished())
    {
        draw_finished(player);
    }
    else if (cars[player]->get_disqualified())
    {
        draw_disqualified(player);
    }
    else
    {
        draw_speedometer(player);
        draw_lap_info(player);
        draw_rank(player);
        int backwards = cars[player]->get_facing_backwards();
        if (backwards)
        {
            glColor4f(1.0f, 1.0f, 1.0f, (GLfloat)backwards/255.0f);
            draw_reverse_sign(player);
            glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
        }
        if (countdown_timer < 500)
        {
            draw_countdown();
        }
    }
    
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    glDisable(GL_BLEND);
}

void GameScene::draw_speedometer(unsigned int player)
{
    // size: 0.4, middle: 0.
    btScalar speed = cars[player]->get_speed() * 10.0;
    m_speedometer_texture->bind();
    glColor4f(1.0, speed / 80.0 + 0.5, speed / 150.0 + 0.5, 1.0f);
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        // background
        glTexCoord2f(0.0,  1.0); glVertex2f(0.5, -0.9);
        glTexCoord2f(0.0,  0.5); glVertex2f(0.5, -0.5);
        glTexCoord2f(0.5,  0.5); glVertex2f(0.9, -0.5);
        glTexCoord2f(0.5,  1.0); glVertex2f(0.9, -0.9);
    glEnd();
#else
	GLfloat vtx1[] = {0.5, -0.9,	0.5, -0.5,	0.9, -0.5,	0.9, -0.9};
	GLfloat tex1[] = {0.0,  1.0,	0.0,  0.5,	0.5,  0.5,	0.5,  1.0};
 
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
 
	glVertexPointer(2, GL_FLOAT, 0, vtx1);
	glTexCoordPointer(2, GL_FLOAT, 0, tex1);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
#endif
    // needle
    glTranslatef(0.7, -0.7, 0.0);
    glRotatef(speed * 2.25 + 180, 0.0, 0.0, -1.0);
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        glTexCoord2f(0.0,  0.5); glVertex2f(-0.2, -0.2);
        glTexCoord2f(0.0,  0.0); glVertex2f(-0.2, 0.2);
        glTexCoord2f(0.5,  0.0); glVertex2f(0.2, 0.2);
        glTexCoord2f(0.5,  0.5); glVertex2f(0.2, -0.2);
    glEnd();
#else
	GLfloat vtx2[] = {-0.2, -0.2,	-0.2, 0.2,	0.2, 0.2,	0.2, -0.2};
	GLfloat tex2[] = {0.0,  0.5,	0.0,  0.0,	0.5,  0.0,	0.5,  0.5};
 
	glVertexPointer(2, GL_FLOAT, 0, vtx2);
	glTexCoordPointer(2, GL_FLOAT, 0, tex2);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
#endif
    glLoadIdentity();
    glColor4f(1.0, speed / 60.0, speed / 100.0, 1.0f);
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        // needle cover
        glTexCoord2f(0.5,  1.0); glVertex2f(0.5, -0.9);
        glTexCoord2f(0.5,  0.5); glVertex2f(0.5, -0.5);
        glTexCoord2f(1.0,  0.5); glVertex2f(0.9, -0.5);
        glTexCoord2f(1.0,  1.0); glVertex2f(0.9, -0.9);
    glEnd();
#else
	GLfloat tex3[] = {0.5,  1.0,	0.5,  0.5,	1.0,  0.5,	1.0,  1.0};
 
	glVertexPointer(2, GL_FLOAT, 0, vtx1);
	glTexCoordPointer(2, GL_FLOAT, 0, tex3);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
#endif
    // text stating the speed
    std::stringstream speedo_text;
    speedo_text.precision(1);
    speedo_text << std::fixed << speed << " m/s";
    glTranslatef(0.565, -0.865, 0.0);
    glScalef(0.002, 0.002, 1);
    FTTextureFont & font = UI::BasicFonts::get_instance().big_font;
    glColor4f(1.0, 1.0, 1.0, 0.7);
    font.Render(speedo_text.str().c_str());
    glLoadIdentity();
    m_speedometer_texture->bind();
    glColor4f(1.0, 1.0, 1.0, 1.0f);
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        // number cover
        glTexCoord2f(0.5,  0.5); glVertex2f(0.5, -0.9);
        glTexCoord2f(0.5,  0.0); glVertex2f(0.5, -0.5);
        glTexCoord2f(1.0,  0.0); glVertex2f(0.9, -0.5);
        glTexCoord2f(1.0,  0.5); glVertex2f(0.9, -0.9);
    glEnd();
#else
	GLfloat tex4[] = {0.5,  0.5,	0.5,  0.0,	1.0,  0.0,	1.0,  0.5};
 
	glVertexPointer(2, GL_FLOAT, 0, vtx1);
	glTexCoordPointer(2, GL_FLOAT, 0, tex4);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);

	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
#endif
}

void GameScene::draw_lap_info(unsigned int player)
{
    // display lap counter
    // darken corner.
    glDisable(GL_TEXTURE_2D);
    glColor4f(0.0, 0.0, 0.0, 0.5);
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        glVertex2f(0.5, 0.7);
        glVertex2f(0.5, 100.0);
        glVertex2f(100.0, 100.0);
        glVertex2f(100.0, 0.7);
    glEnd();
#else
	GLfloat vtx1[] = {0.5, 0.7,	0.5, 100.0,	100.0, 100.0,	100.0, 0.7};
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(2, GL_FLOAT, 0, vtx1);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
#endif
    glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
    glEnable(GL_TEXTURE_2D);
    
    // display message
    std::stringstream lap_text;
    lap_text << "Lap " << cars[player]->get_lap() << " of 3";
    glTranslatef(0.5, 0.8, 0.0);
    glScalef(0.75/256.0, 0.75/256.0, 1);
    FTTextureFont & font = UI::BasicFonts::get_instance().big_font;
    font.Render(lap_text.str().c_str());
    glLoadIdentity();
    
    glDisable(GL_TEXTURE_2D);
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_LINE_LOOP);
        glVertex2f(0.55, 0.76);
        glVertex2f(0.9, 0.76);
        glVertex2f(0.9, 0.78);
        glVertex2f(0.55, 0.78);
    glEnd();
#else
	GLfloat vtx2[] = {0.55, 0.76,	0.9, 0.76,	0.9, 0.78,	0.55, 0.78};
	glVertexPointer(2, GL_FLOAT, 0, vtx2);
	glDrawArrays(GL_LINE_LOOP,0,4);
#endif

    float x_end = 0.55 + 0.35 * cars[player]->get_lap_distance() / track.get_lap_length();
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        glVertex2f(0.55, 0.76);
        glVertex2f(x_end, 0.76);
        glVertex2f(x_end, 0.78);
        glVertex2f(0.55, 0.78);
    glEnd();
#else
	GLfloat vtx3[] = {0.55, 0.76,	x_end, 0.76,	x_end, 0.78,	0.55, 0.78};
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(2, GL_FLOAT, 0, vtx3);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
	glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glEnable(GL_TEXTURE_2D);
    if (cars[player]->get_lap() > 1)
    {
        std::stringstream time_text;
        long unsigned int time = cars[player]->get_last_lap_ticks();
        time_text << "Last " << time / 6000 << ":" << (time % 6000) / 100
                  << "." << ((time / 10) % 10) << (time % 10);
        glTranslatef(0.5, 0.7, 0.0);
        glScalef(0.75/256.0, 0.75/256.0, 1);
        font.Render(time_text.str().c_str());
        glLoadIdentity();
        // if the lap time is the best of all players, draw an icon next to it.
        unsigned long int best_time = std::numeric_limits<unsigned long int>::max();
        for (std::vector<GameObjects::Car *>::const_iterator it = cars.begin();
             it != cars.end(); it++)
        {
            unsigned long int car_best = (*it)->get_best_lap_ticks();
            if (best_time > car_best)
            {
                best_time = car_best;
            }
        }
        if (time == best_time)
        {
            // draw icon.
            m_best_lap_texture->bind();
#if 0 //!defined(HAVE_GLES)
            glBegin(GL_QUADS);
                glTexCoord2f(0.0,  1.0); glVertex2f(0.4, 0.675);
                glTexCoord2f(0.0,  0.0); glVertex2f(0.4, 0.775);
                glTexCoord2f(1.0,  0.0); glVertex2f(0.5, 0.775);
                glTexCoord2f(1.0,  1.0); glVertex2f(0.5, 0.675);
            glEnd();
#else
	GLfloat vtx4[] = {0.4, 0.675,	0.4, 0.775,	0.5, 0.775,	0.5, 0.675};
	GLfloat tex4[] = {0.0,  1.0,	0.0,  0.0,	1.0,  0.0,	1.0,  1.0};

	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glVertexPointer(2, GL_FLOAT, 0, vtx4);
	glTexCoordPointer(2, GL_FLOAT, 0, tex4);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
#endif
        }
    }
}

void GameScene::draw_rank(unsigned int player)
{
    glDisable(GL_TEXTURE_2D);
    glColor4f(0, 0, 0, .75);
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        glVertex2f(-0.1, 0.8);
        glVertex2f( 0.1, 0.8);
        glVertex2f( 0.1, 100.0);
        glVertex2f(-0.1, 100.0);
    glEnd();
#else
	GLfloat vtx1[] = {-0.1, 0.8,	0.1, 0.8,	0.1, 100.0,	-0.1, 100.0};
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(2, GL_FLOAT, 0, vtx1);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
	glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glEnable(GL_TEXTURE_2D);
    glColor4f(1.0, 1.0, 1.0, 1.0);
    
    std::stringstream rank_text;
    int rank = car_ranks[player] + 1;
    rank_text << rank;
    // work out what suffix to use (st, nd, rd, or th)
    int rank_mod100 = rank % 100;
    int rank_mod10 = rank % 10;
    // catch 11th, 12th, and 13th, so they don't become 11st, 12nd, and 13rd.
    if (rank_mod100 >= 4 && rank_mod100 <= 20) rank_text << "th";
    else if (rank_mod10 == 1) rank_text << "st";
    else if (rank_mod10 == 2) rank_text << "nd";
    else if (rank_mod10 == 3) rank_text << "rd";
    else rank_text << "th";
    glTranslatef(-0.1, 0.8, 0.0);
    glScalef(0.004, 0.004, 1);
    FTTextureFont & font = UI::BasicFonts::get_instance().big_font;
    font.Render(rank_text.str().c_str());
    glLoadIdentity();
}

void GameScene::draw_disqualified(unsigned int player)
{
    glDisable(GL_TEXTURE_2D);
    unsigned long int ticks_since = world.get_tick_number() - cars[player]->get_finish_time();
    glColor4f(0, 0, 0, 1.0 - 1.0 / (float(ticks_since) / 100.0));
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        glVertex2f(-100.0, -100.0);
        glVertex2f( 100.0, -100.0);
        glVertex2f( 100.0,  100.0);
        glVertex2f(-100.0,  100.0);
    glEnd();
#else
	GLfloat vtx1[] = {-100.0, -100.0,	100.0, -100.0,		100.0,  100.0,		-100.0,  100.0};
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(2, GL_FLOAT, 0, vtx1);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
#endif
    glEnable(GL_TEXTURE_2D);
    glColor4f(1.0, 1.0, 1.0, 1.0);
    
    m_disqualified_text_texture->bind();
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        glTexCoord2f(0, 1); glVertex2f(-1.0, -0.25);
        glTexCoord2f(1, 1); glVertex2f( 1.0, -0.25);
        glTexCoord2f(1, 0); glVertex2f( 1.0,  0.25);
        glTexCoord2f(0, 0); glVertex2f(-1.0,  0.25);
    glEnd();
#else
	GLfloat vtx2[] = {-1.0, -0.25,	1.0, -0.25,	1.0,  0.25,	-1.0,  0.25};
	GLfloat tex2[] = {0, 1,		1, 1,		1, 0,		0, 0};
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glTexCoordPointer(2, GL_FLOAT, 0, tex2);
	glVertexPointer(2, GL_FLOAT, 0, vtx2);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);
#endif
}

void GameScene::draw_finished(unsigned int player)
{
    // black background
    glDisable(GL_TEXTURE_2D);
    glColor4f(0.0, 0.0, 0.0, 0.75);
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        glVertex2f(-100.0, -0.3);
        glVertex2f( 100.0, -0.3);
        glVertex2f( 100.0,  0.3);
        glVertex2f(-100.0,  0.3);
    glEnd();
#else
	GLfloat vtx1[] = {-100.0, -0.3,		100.0, -0.3,		100.0,  0.3,		-100.0,  0.3};
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(2, GL_FLOAT, 0, vtx1);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
	glDisableClientState(GL_VERTEX_ARRAY);
#endif
    
    // checkered border
    glEnable(GL_TEXTURE_2D);
    m_finish_border_texture->bind();
    glColor4f(1.0, 1.0, 1.0, 1.0);
    glMatrixMode(GL_TEXTURE);
    // make it move
    glTranslatef(float(world.get_tick_number()) * 0.003, 0, 0);
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        glTexCoord2f(-1000, 1); glVertex2f(-100.0, 0.3);
        glTexCoord2f( 1000, 1); glVertex2f( 100.0, 0.3);
        glTexCoord2f( 1000, 0); glVertex2f( 100.0, 0.4);
        glTexCoord2f(-1000, 0); glVertex2f(-100.0, 0.4);
        
        glTexCoord2f( 1000, 1); glVertex2f(-100.0, -0.3);
        glTexCoord2f(-1000, 1); glVertex2f( 100.0, -0.3);
        glTexCoord2f(-1000, 0); glVertex2f( 100.0, -0.4);
        glTexCoord2f( 1000, 0); glVertex2f(-100.0, -0.4);
    glEnd();
#else
	GLfloat vtx2[] = {-100.0, 0.3,	100.0, 0.3,	100.0, 0.4,	-100.0, 0.4};
	GLfloat vtx3[] = {-100.0, -0.3,	100.0, -0.3,	100.0, -0.4,	-100.0, -0.4};
	GLfloat tex2[] = {-1000, 1,	1000, 1,	1000, 0,	-1000, 0};
	GLfloat tex3[] = {1000, 1,	-1000, 1,	-1000, 0,	1000, 0};
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glVertexPointer(2, GL_FLOAT, 0, vtx2);
	glTexCoordPointer(2, GL_FLOAT, 0, tex2);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
	glVertexPointer(2, GL_FLOAT, 0, vtx3);
	glTexCoordPointer(2, GL_FLOAT, 0, tex3);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
#endif
    glLoadIdentity();
    glMatrixMode(GL_MODELVIEW);
    
    // text that says "Finished" in the middle.
    m_finish_text_texture->bind();
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        glTexCoord2f(0, 1); glVertex2f(-1.0, -0.25);
        glTexCoord2f(1, 1); glVertex2f( 1.0, -0.25);
        glTexCoord2f(1, 0); glVertex2f( 1.0,  0.25);
        glTexCoord2f(0, 0); glVertex2f(-1.0,  0.25);
    glEnd();
#else
	GLfloat vtx4[] = {-1.0, -0.25,	1.0, -0.25,	1.0,  0.25,	-1.0,  0.25};
	GLfloat tex4[] = {0, 1,		1, 1,		1, 0,		0, 0};

	glVertexPointer(2, GL_FLOAT, 0, vtx4);
	glTexCoordPointer(2, GL_FLOAT, 0, tex4);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
#endif
    glDisable(GL_TEXTURE_2D);
    
    // white screen flash
    unsigned long int ticks_since = world.get_tick_number() - cars[player]->get_finish_time();
    glColor4f(1.0, 1.0, 1.0, 1.0 / (float(ticks_since) / 10.0));
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        glVertex2f(-100.0, -100.0);
        glVertex2f( 100.0, -100.0);
        glVertex2f( 100.0,  100.0);
        glVertex2f(-100.0,  100.0);
    glEnd();
#else
	GLfloat vtx5[] = {-100.0, -100.0,	100.0, -100.0,	100.0,  100.0,	-100.0,  100.0};
	glVertexPointer(2, GL_FLOAT, 0, vtx5);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
	glDisableClientState(GL_VERTEX_ARRAY);
#endif
    glEnable(GL_TEXTURE_2D);
    glColor4f(1.0, 1.0, 1.0, 1.0);
    
    // text displaying the time taken.
    std::stringstream time_text;
    unsigned long int time = cars[player]->get_finish_time();
    time_text << "Time: " << (time / 6000)
              << ((time / 6000 == 1) ? " minute, " : " minutes, ")
              << (time / 100) % 60 << "."
              << (time / 10) % 10 << (time % 10) << " seconds";
    glTranslatef(-0.9, -0.27, 0.0);
    glScalef(0.004, 0.004, 1);
    FTTextureFont & font = UI::BasicFonts::get_instance().big_font;
    font.Render(time_text.str().c_str());
    glLoadIdentity();
}

void GameScene::draw_reverse_sign(unsigned int player)
{
    m_reverse_texture->bind();
#if 0 //!defined(HAVE_GLES)
    glBegin(GL_QUADS);
        glTexCoord2f(0.0, 1.0); glVertex2f(-0.3, 0.0);
        glTexCoord2f(1.0, 1.0); glVertex2f( 0.3, 0.0);
        glTexCoord2f(1.0, 0.0); glVertex2f( 0.3, 0.6);
        glTexCoord2f(0.0, 0.0); glVertex2f(-0.3, 0.6);
    glEnd();
#else
	GLfloat vtx1[] = {-0.3, 0.0,	0.3, 0.0,	0.3, 0.6,	-0.3, 0.6};
	GLfloat tex1[] = {0.0, 1.0,	1.0, 1.0,	1.0, 0.0,	0.0, 0.0};
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glVertexPointer(2, GL_FLOAT, 0, vtx1);
	glTexCoordPointer(2, GL_FLOAT, 0, tex1);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);
#endif
}

void GameScene::draw_countdown()
{
    m_countdown_texture->bind();
    unsigned int tick = 3000+countdown_timer;
    // which sign to show (0->3, 1->2, 2->1, 3->Go)
    unsigned int n = tick/1000;
    float xl = float(n) / 4.0;
    float xh = xl + 0.25;
    // time taken on this sign in seconds
    float s_time = float (tick - n * 1000) / 1000.0;
    glPushMatrix();
        if (n == 3)
        {
            s_time *= 2.0;
            float scale = 1.0 + sqrt(s_time) * 2.0;
            glScalef(scale, scale, 1.0);
        }
#if 1 //defined(HAVE_GLES)
	GLfloat vtx1[] = {-0.3, -0.3,	0.3, -0.3,	0.3, 0.3,	-0.3, 0.3};
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glVertexPointer(2, GL_FLOAT, 0, vtx1);
#endif
        if (s_time > 0.875)
        {
            //glColor4f(1.0, 1.0, 1.0, s_time*16.0-15.0);
            float scale_t = 8.0+s_time*-8.0;
            float h_scale = 1.0/(scale_t);
            float v_scale = scale_t * scale_t;
            glScalef(h_scale, v_scale, 1.0);
            // white and grey
#if 0 //!defined(HAVE_GLES)
            glBegin(GL_QUADS);
                glTexCoord2f(xl, 0.5); glVertex2f(-0.3, -0.3);
                glTexCoord2f(xh, 0.5); glVertex2f( 0.3, -0.3);
                glTexCoord2f(xh, 0.0); glVertex2f( 0.3, 0.3);
                glTexCoord2f(xl, 0.0); glVertex2f(-0.3, 0.3);
            glEnd();
#else
	GLfloat tex1[] = {xl, 0.5,	xh, 0.5,	xh, 0.0,	xl, 0.0};
	glTexCoordPointer(2, GL_FLOAT, 0, tex1);
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
#endif
        } else {
            // coloured outline, dark shadow
#if 0 //!defined(HAVE_GLES)
            glBegin(GL_QUADS);
                glTexCoord2f(xl, 1.0); glVertex2f(-0.3, -0.3);
                glTexCoord2f(xh, 1.0); glVertex2f( 0.3, -0.3);
                glTexCoord2f(xh, 0.5); glVertex2f( 0.3, 0.3);
                glTexCoord2f(xl, 0.5); glVertex2f(-0.3, 0.3);
            glEnd();
#else
	GLfloat tex2[] = {xl, 1.0,	xh, 1.0,	xh, 0.5,	xl, 0.5};
	glTexCoordPointer(2, GL_FLOAT, 0, tex2);
#endif
        }
#if 1 //defined(HAVE_GLES)
	glDrawArrays(GL_TRIANGLE_FAN,0,4);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);
#endif

    glPopMatrix();
}

void GameScene::do_sound()
{
    // The changes in audio object's state are done in update_logic
    // OpenAL needs to be told the result, though:
    Audio::get_instance().commit();
}

Physics::World & GameScene::get_world()
{
    return world;
}

void GameScene::set_save_replay(bool value)
{
    save_replay = value;
}

void GameScene::rank_cars()
{
    car_ranks.resize(cars.size());
    // sort cars by lap position + lap length * number of laps completed.
    std::multimap<btScalar, std::size_t> car_positions;
    
    // Add a bit on to the lap length. The lap length is the shortest route
    // around, so a car may be further around the lap than the lap is long,
    // but we don't want to count this in the position so cars don't get
    // ranked behind a car on the previous lap.
    btScalar lap_length = track.get_lap_length() + 100.0;
    
    for (std::size_t index = 0; index < cars.size(); index++)
    {
        GameObjects::Car & car = *cars[index];
        btScalar position;
        if (car.get_disqualified())
        {
            position = 0;
        } else {
            position = car.get_lap_distance() + car.get_lap() * lap_length;
        }
        car_positions.insert(std::pair<btScalar, std::size_t>(position, index));
    }
    // rank cars in reverse order of car_positions, as it is sorted from the
    // lowest distance around to course to the highest: highest is the leader.
    int rank = cars.size() - 1;
    for (std::multimap<btScalar, std::size_t>::iterator it = car_positions.begin();
         it != car_positions.end();
         it++)
    {
        car_ranks[it->second] = rank;
        rank--;
    }
    
}

void GameScene::unpause()
{
    // menu is already giving a response. Reset it so we can show it again.
    m_pause_menu.reset();
    paused = false;
    // start the sound
    Audio::get_instance().resume();
}

}
