/** @file Car.h
 *  @brief Declare the Engine::GameObjects::Car class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef CAR_H_
#define CAR_H_

#include <btBulletDynamicsCommon.h>

#include <libtrack/AABBDrawable.h>
#include <libtrack/Texture.h>
#include <libtrack/Mesh/BulletMesh.h>
#include <libtrack/NearTrack.h>

#include "../Physics/World.h"
#include "../Physics/TickObserver.h"
#include "../InputReport.h"
#include "../Audio.h"

namespace Engine
{
    class InputDevice;

/// Objects which affect the gameplay.
namespace GameObjects
{

/** A vehicle that can be controled by a player using an input device.
 */
class Car
    :   public Track::AABBDrawable
    ,   public Physics::TickObserver
    ,   private Track::NearTrack 
{
    Car(Car & car);
    Car & operator=(Car & car);
public:
    /// The number of directions the cpu car directional sensors read
    const static unsigned int NUM_SENSORS_FEELERS = 8;
    /// The number of values of data each directional feeler has.
    const static unsigned int NUM_FEELER_PROPERTIES = Track::SENSOR_COUNT;
    /// The total number of sensor values.
    const static unsigned int NUM_SENSORS = NUM_SENSORS_FEELERS * NUM_FEELER_PROPERTIES + 14;
    
    /** Construct the car and place it into the game.
     * @param world The Physics world of the game.
     * @param start The transformation for the starting position.
     * @todo Get a proper starting position.
     * @param input_device The input device to bind the car to.
     * @param car_model The mesh and texture of the car. Must be 0 or 1.
     * @param plane_normal The unit normal of the start/finish plane.
     * @param plane_distance The signed distance between the start/finish plane and the origin.
     * @param start_point The centre of the start/finish line.
     */
    Car(Physics::World & world, btTransform start, InputDevice * input_device,
        unsigned int car_model, const btVector3 & plane_normal,
        const btScalar & plane_distance, const btVector3 & start_point);
    virtual ~Car();
    virtual void draw() const;
    virtual Track::AxisAlignedBoundingBox get_bounds() const;
    
    /// Take input from a device.
    void take_input(InputReport & report);
    
    /// apply driving forces to the physics engine's representation of the car.
    virtual void set_forces();
    
    /// update the physics
    virtual void posttick();
    
    void get_transform(btTransform & transform);
    
    /// Return the lap the car is on.
    signed int get_lap();
    
    /// Return the number of ticks taken to complete the last lap.
    unsigned long int get_last_lap_ticks();
    
    /// Return the number of ticks taken to complete the fastest lap
    unsigned long int get_best_lap_ticks();
    
    /// Return the speed the car is travelling at in decimeters per second.
    btScalar get_speed();
    
    /// Return the distance from the start of the lap.
    btScalar get_lap_distance() const;
    
    /// Return true if three laps have been completed.
    bool get_finished() const;
    /// Return true if the car has been disqualifed.
    bool get_disqualified() const;
    /** Return the time taken to complete the course or get disqualified.
     * Precondition: get_finished() or get_disqualified() returns true.
     * @return time in ticks taken to complete three laps or get disqualified.
     */
    unsigned long int get_finish_time() const;
    
    /** Find if car is facing the wrong way.
     * If facing the right way, returns 0.
     * Otherwise returns a value between 1 and 255 indicating how wrong the
     * directions is.
     */
    int get_facing_backwards() const;
    
    /** Find a suitable height for a camera.
     * Takes into account the slope of the track infront of the car.
     */
    btScalar get_camera_height() const;
    
    /** Return the relative distance to the goal in various directions.
     * Gives the difference between the distance to the goal of the car and
     * the difference to the goal of other points: 1m infront of the car,
     * 1m behind the car, 1m to the left of the car, 1m to the right of the car
     * @param results array where calculated values will be written.
     */
    void get_sensors(float results[NUM_SENSORS]) const;
private:
    // graphical representation
    Track::Texture *texture;
    Track::BulletMesh *mesh;
    /// The texture used for the engine flame effect.
    Track::Texture *flame_texture;
    /// The mesh used for the engine flame effect.
    Track::BulletMesh *flame_mesh;
    // physical representation
    Physics::World & world;
    btRigidBody * rigid_body;
    InputDevice * input_device;
    btVector3 force, torque, local_force, local_torque;
    
    /** True when sticking to a floor, false if falling.
     */
    bool floor_stick;
    
    /// Which car mesh and texture to use.
    unsigned int car_model;
    
    /// the lap the car is on
    signed int lap;
    /// The highest numbered lap ever reached.
    signed int max_lap;
    /// The tick number when starting the last lap.
    unsigned long int lap_start_ticks;
    /// The number of ticks taken to complete the last lap.
    unsigned long int last_lap_ticks;
    /// The number of ticks taken to complete the fastest lap
    unsigned long int best_lap_ticks;
    
    /// True if infront of the start line.
    bool infront_of_start;
    
    const btVector3 & plane_normal;
    const btScalar & plane_distance;
    const btVector3 & start_point;
    
    /// energy built up when breaking.
    btScalar elastic_potential_energy;
    /// amount of energy built up while breaking in the last frame.
    btScalar tick_energy_absorbed;
    /// The amount stored of enery released last tick.
    btScalar used_energy_boost;
    
    /// number of ticks since last boost.
    unsigned int boost_timer;
    
    /// True if 3 laps have been completed.
    bool m_course_complete; 
    /// The time in ticks taken to complete three laps.
    unsigned long int m_course_complete_time;
    /** True if the car has been removed from the course.
     * If m_course_complete is set, it finished,
     * otherwise it has been disqualified.
     */
    bool m_removed;
    
    /// The transformation used when the car crossed the finish line.
    btTransform m_last_transform;
    
    /// Remove the car from the course.
    void remove();
    
    /// The sound effect for the engine accelerating noise.
    boost::shared_ptr<SoundSource> m_engine_sound_source;
    /// The sound effect when the car is moving independant of acceleration.
    boost::shared_ptr<SoundSource> m_drag_sound_source;
    /// The sound effect when the car is braking.
    boost::shared_ptr<SoundSource> m_break_sound_source;
    /// The sound effect for the booster noise.
    boost::shared_ptr<SoundSource> m_boost_sound_source;
    /// The sound effect for releasing energy stored while breaking.
    boost::shared_ptr<SoundSource> m_release_energy_sound_source;
    /// The sound effect for sliding sideways.
    boost::shared_ptr<SoundSource> m_slide_sound_source;
    
    /// Set the volume, pitch, position and velocity of the sound sources.
    void update_sounds();
};

}

}

#endif /*CAR_H_*/
