/** @file Car.cpp
 *  @brief Implement the Engine::GameObjects::Car class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "Car.h"
#include <GL/gl.h>
#include <btBulletDynamicsCommon.h>
#include <btBulletCollisionCommon.h>
#include <BulletCollision/CollisionShapes/btTriangleShape.h>
#include "../InputDevice.h"
#include "../ResourceHandler.h"
#include <Debug.h>
#include <cmath>
#include <limits>
#include <libtrack/TrackBooster.h>

// uncomment and play (or watch replay) with a single human player
// to make artificial neural network training data file.
// Train the neural network to this file with racer_train_ai.
// After that, you can disable it again and run racer t track to improve on
// the neural network by racing mutated AI players against each other.
//#define TRAIN_AI

#ifdef TRAIN_AI
#include <fstream>
#endif
#ifdef HAVE_GLES
#define GL_CLAMP     GL_CLAMP_TO_EDGE
#endif

const btVector3 feeler_vectors[Engine::GameObjects::Car::NUM_SENSORS_FEELERS] = {
                btVector3( 0,  1, 0),
                btVector3( 1,  0, 0),
                btVector3(-1,  0, 0),
                btVector3( 1,  1, 0).normalized(),
                btVector3(-1,  1, 0).normalized(),
                btVector3(-0.2, 1, 0).normalized(),
                btVector3( 0.2, 1, 0).normalized(),
                btVector3( 0, -1, 0)};

const btScalar _force_scale = 1.5;

/// stearing torque scale per car model. High values imply car turns easily.
const btScalar _car_stearing_scale[2] = {0.02, 0.03};
/// stearing torque in midair per car model.
const btScalar _car_stearing_scale_midair[2] = {0.01, 0.015};
/** control over car sliding. Higher values give strong manual sliding and
 * strong recovery from sliding when cornering.
 */
const btScalar _car_slide_scale[2] = {1.0, 1.5};


/* Correction for stupid bouncing my cars up in the air when they drive over the
 * internal edges of flat triangulated sections of the road.
 */

void contact_added_callback_obj (btManifoldPoint& cp,
                                 const btCollisionObject* colObj,
                                 int partId, int index)
{
        (void) partId;
        (void) index;
        const btCollisionShape *shape = colObj->getCollisionShape();

        if (shape->getShapeType() != TRIANGLE_SHAPE_PROXYTYPE) return;
        const btTriangleShape *tshape =
               static_cast<const btTriangleShape*>(colObj->getCollisionShape());


        const btCollisionShape *parent = colObj->getRootCollisionShape();
        if (parent == NULL) return;
        if (parent->getShapeType() != TRIANGLE_MESH_SHAPE_PROXYTYPE) return;

        btTransform orient = colObj->getWorldTransform();
        orient.setOrigin( btVector3(0.0f,0.0f,0.0f ) );

        btVector3 v1 = tshape->m_vertices1[0];
        btVector3 v2 = tshape->m_vertices1[1];
        btVector3 v3 = tshape->m_vertices1[2];

        btVector3 normal = (v2-v1).cross(v3-v1);

        normal = orient * normal;
        normal.normalize();

        btScalar dot = normal.dot(cp.m_normalWorldOnB);
        btScalar magnitude = cp.m_normalWorldOnB.length();
        normal *= dot > 0 ? magnitude : -magnitude;

        cp.m_normalWorldOnB = normal;
}

bool contact_added_callback (btManifoldPoint& cp,
                             const btCollisionObject* colObj0,
                             int partId0, int index0,
                             const btCollisionObject* colObj1,
                             int partId1, int index1)
{
        contact_added_callback_obj(cp, colObj0, partId0, index0);
        contact_added_callback_obj(cp, colObj1, partId1, index1);
        //std::cout << to_ogre(cp.m_normalWorldOnB) << std::endl;
        return true;
}

extern ContactAddedCallback      gContactAddedCallback;

namespace Engine
{

namespace GameObjects
{

// shorter names for getting the car texture and mesh resources.
typedef ResourceHandler<Track::Texture, int, std::string> TextureStore;
typedef ResourceHandler<Track::BulletMesh, int, Track::BulletMesh::ConstructionInformation> MeshStore;

Car::Car(Physics::World & world, btTransform start, InputDevice * input_device,
         unsigned int car_model, const btVector3 & plane_normal,
        const btScalar & plane_distance, const btVector3 & start_point)
    :   NearTrack(world.get_track().get_ai_graph(), start.getOrigin())
    ,   world(world),
        input_device(input_device),
        force(0, 0, 0),
        torque(0, 0, 0),
        local_force(0, 0, 0),
        local_torque(0, 0, 0),
        floor_stick(false),
        car_model(car_model)
    ,   lap(0)
    ,   max_lap(1)
    ,   lap_start_ticks(0)
    ,   best_lap_ticks(std::numeric_limits<unsigned long int>::max())
    ,   plane_normal(plane_normal)
    ,   plane_distance(plane_distance)
    ,   start_point(start_point)
    ,   elastic_potential_energy(2<<27)
    ,   tick_energy_absorbed(0)
    ,   used_energy_boost(0)
    ,   boost_timer(100)
    ,   m_course_complete(false)
    ,   m_removed(false)
{
    input_device->set_car(this);
    
    // load a texture and mesh for the car, if not already loaded.
    const unsigned int mesh_generator_bits =
                            Track::BulletMesh::genererator_convex_hull_bit;
    if (car_model == 0)
    {
        TextureStore::get_instance().check_load(0,
                                                "data/cars/1/craft_full.png");
        MeshStore::get_instance().check_load(0,
                                             std::make_pair("data/cars/1/mesh",
                                                            mesh_generator_bits));
    }
    else
    {
        TextureStore::get_instance().check_load(1,
                                                "data/cars/2/craft_full.png");
        MeshStore::get_instance().check_load(1,
                                             std::make_pair("data/cars/2/mesh",
                                                            mesh_generator_bits));
    }
    TextureStore::get_instance().check_load(-1, "data/generic/flame_grad.png");
    flame_texture = TextureStore::get_instance().get(-1);
    flame_texture->bind();
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    
    MeshStore::get_instance().check_load(-1,
                                             std::make_pair("data/generic/engine_flame",
                                                            0));
    flame_mesh = MeshStore::get_instance().get(-1);
    mesh = MeshStore::get_instance().get(car_model);
    texture = TextureStore::get_instance().get(car_model);
    btConvexHullShape* shape = mesh->get_convex_hull_shape();
    shape->setMargin(btScalar(0.04));
    btScalar mass(500.0);
    btVector3 inertia(0,0,0);
    shape->calculateLocalInertia(mass, inertia);
    
    btRigidBody::btRigidBodyConstructionInfo rigid_body_CI(mass,
                                                           0,
                                                           shape,
                                                           inertia);
    rigid_body_CI.m_linearDamping = btScalar(0.1);
    rigid_body_CI.m_angularDamping = btScalar(0.9992);
    rigid_body_CI.m_friction = btScalar(0.4);
    rigid_body_CI.m_restitution = 0.25;
    rigid_body_CI.m_startWorldTransform = start;
    rigid_body = new btRigidBody(rigid_body_CI);
    
    rigid_body->setSleepingThresholds(btScalar(0.0),    btScalar(0.0));  
    
    world.get_dynamics_world().addRigidBody(rigid_body);
    
    gContactAddedCallback = contact_added_callback;
    
    rigid_body->setCollisionFlags(rigid_body->getCollisionFlags() |
    btCollisionObject::CF_CUSTOM_MATERIAL_CALLBACK);
    
    world.add_tick_observer(this);
    
    infront_of_start = false;
    
    // sound
    static SoundBuffer engine_sound_buffer("data/sound/engine.wav");
    m_engine_sound_source = boost::shared_ptr<SoundSource>(new SoundSource(engine_sound_buffer));
    m_engine_sound_source->set_looping();
    
    static SoundBuffer drag_sound_buffer("data/sound/car_drag.wav");
    m_drag_sound_source = boost::shared_ptr<SoundSource>(new SoundSource(drag_sound_buffer));
    m_drag_sound_source->set_looping();
    
    static SoundBuffer break_sound_buffer("data/sound/car_break.wav");
    m_break_sound_source = boost::shared_ptr<SoundSource>(new SoundSource(break_sound_buffer));
    m_break_sound_source->set_looping();
    
    static SoundBuffer boost_sound_buffer("data/sound/boost.wav");
    m_boost_sound_source = boost::shared_ptr<SoundSource>(new SoundSource(boost_sound_buffer));
    m_boost_sound_source->set_gain(2.0);
    
    static SoundBuffer release_energy_sound_buffer("data/sound/car_release_energy.wav");
    m_release_energy_sound_source = boost::shared_ptr<SoundSource>(new SoundSource(release_energy_sound_buffer));
    m_release_energy_sound_source->set_looping();
    
    static SoundBuffer slide_sound_buffer("data/sound/slide.wav");
    m_slide_sound_source = boost::shared_ptr<SoundSource>(new SoundSource(slide_sound_buffer));
    m_slide_sound_source->set_looping();
    
    update_sounds();
    m_release_energy_sound_source->play();
    m_engine_sound_source->play();
    m_drag_sound_source->play();
    m_break_sound_source->play();
    m_slide_sound_source->play();
    
    // we should be in a reasonable starting position
    assert(get_position().distance2(rigid_body->getCenterOfMassPosition()) < 225.0);
}

Car::~Car()
{
    if (!m_removed) remove();
    world.remove_tick_observer(this);
    // Stop the car's InputDevice from sending reports to this car.
    input_device->set_car(0);
}

Track::AxisAlignedBoundingBox Car::get_bounds() const
{
    if (m_removed)
    {
        // no bounds, nothing is drawn.
        return Track::AxisAlignedBoundingBox();
    }
    /** @todo Maybe testing a bounding box transformed is better than testing
     *  the axis aligned bounding box of a transformed axis aligned bounding
     *  box.
     */
    btTransform transform = rigid_body->getCenterOfMassTransform();
    return mesh->get_bounds().transform(transform);
}


void Car::draw() const
{
    if (m_removed) return;
    texture->bind();
    glPushMatrix();
        // get the position and angle from the physics engine.
        btTransform transform = rigid_body->getCenterOfMassTransform();
        btScalar transform_matrix[16];
        transform.getOpenGLMatrix(transform_matrix);
        glMultMatrixf(transform_matrix);
        // additionally, roll depending on the slide.
        glRotatef(force.getX()*0.0005, 0.0, 1.0, 0.0);
        mesh->draw();
        if (used_energy_boost > 0)
        {
            // show released energy by making the car glow.
            world.get_track().get_lighting().turn_off();
            glBlendFunc(GL_SRC_ALPHA, GL_ONE);
            glEnable(GL_BLEND);
            glDepthFunc(GL_LEQUAL);
            glDisable(GL_TEXTURE_2D);
            glColor4f(used_energy_boost * 0.0000001, used_energy_boost * 0.00000005, used_energy_boost * 0.00000001, 1.0f);
            mesh->draw();
            glEnable(GL_TEXTURE_2D);
            glDisable(GL_BLEND);
            glDepthFunc(GL_LESS);
            glColor4f(1.0f, 1.0f,1.0f,1.0f);
        }
        // engine flames
        // The flame is not affected by lighting, double sided, doesn't
        // change the depth buffer, doesn't darken the scene behind
        // when blended over, and reflects the textures alpha value.
        world.get_track().get_lighting().turn_off();
        glEnable(GL_BLEND);
        glDisable(GL_CULL_FACE);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE);
        glDepthMask(GL_FALSE);
        glMatrixMode(GL_TEXTURE);
        glPushMatrix();
            // Move the flame effect randomly to make it flicker.
            glTranslatef(0.0, float(std::rand()%256) * 0.0004, 0.0);
            // intensity is dependant on how hard you are accelerating.
            // there is no flame if not accelerating, breaking, or going
            // backwards.
            // The flame is brighter just after a boost.
            glColor4f(1.0, 1.0, 1.0, ((100.0 - boost_timer) / 200.0 + 0.5) * (force.getY() / 32768.0));
            flame_texture->bind();
            
            glMatrixMode(GL_MODELVIEW);
            float b_scale = (100.0 - boost_timer) * 0.05 + 1.0;
            switch (car_model)
            {
                case 0:
                    glPushMatrix();
                        glTranslatef(0.046, -0.231, 0.032);
                        glScalef(0.03, 0.03*b_scale, 0.03);
                        flame_mesh->draw();
                    glPopMatrix();
                    glPushMatrix();
                        glTranslatef(-0.046, -0.231, 0.032);
                        glScalef(0.03, 0.03*b_scale, 0.03);
                        flame_mesh->draw();
                    glPopMatrix();
                    break;
                case 1:
                    glPushMatrix();
                        glTranslatef(0.151, -0.295, 0.0);
                        glScalef(0.047, 0.047*b_scale, 0.047);
                        flame_mesh->draw();
                    glPopMatrix();
                    glPushMatrix();
                        glTranslatef(-0.151, -0.295, 0.0);
                        glScalef(0.047, 0.047*b_scale, 0.047);
                        flame_mesh->draw();
                    glPopMatrix();
                    break;
            }
            // reset OpenGL settings.
            glMatrixMode(GL_TEXTURE);
        glPopMatrix();
        glMatrixMode(GL_MODELVIEW);
        world.get_track().get_lighting().turn_on();
        glEnable(GL_CULL_FACE);
        glDepthMask(GL_TRUE);
        glDisable(GL_BLEND);
    glPopMatrix();
#ifndef NDEBUG
    glDisable(GL_TEXTURE_2D);
    glBegin(GL_LINE_STRIP);
        glVertex3f(get_position().x(), get_position().y(), get_position().z());
        glVertex3f(transform.getOrigin().x(),
                   transform.getOrigin().y(),
                   transform.getOrigin().z());
    glEnd();
    glBegin(GL_LINES);
        glColor4f(0.0f,1.0f,0.0f,1.0f);
        const Track::MeshFaces::Graph & graph = world.get_track().get_ai_graph();
        const btVector3 *v = graph[get_face_descriptor()].vertex_positions;
        glVertex3f(v[0].x(), v[0].y(), v[0].z());
        glVertex3f(v[1].x(), v[1].y(), v[1].z());
        glColor4f(0,0,1,1);
        glVertex3f(v[0].x(), v[0].y(), v[0].z());
        glVertex3f(v[2].x(), v[2].y(), v[2].z());
    glEnd();
    glColor4f(1,1,1,1);
    
    // AI sensors- disabled because it is slightly slower and distracting
#if 0
    btScalar r[NUM_SENSORS];
    get_sensors(r);
    // feeler distances
    glDisable(GL_DEPTH_TEST);
    glEnable(GL_TEXTURE_2D);
    glBegin(GL_LINES);
        
        for (unsigned int i = 0; i < NUM_SENSORS_FEELERS; i++)
        {
            glTexCoord2f(0, 0);
            glVertex3f(transform.getOrigin().x(),
                   transform.getOrigin().y(),
                   transform.getOrigin().z());
            btScalar distance = 1.0 / r[i * NUM_FEELER_PROPERTIES];
            btVector3 d =transform(feeler_vectors[i] * distance);
            glTexCoord2f(distance, distance);
            glVertex3f(d.x(), d.y(), d.z());
        }
    glEnd();
    glEnable(GL_DEPTH_TEST);
#endif // 0
    glEnable(GL_TEXTURE_2D);
#endif // debug
}

void Car::take_input(InputReport & report)
{
    switch (report.get_report_type())
    {
        case InputReport::RT_CHANGE_ACCEL:
            /// Change to accelerator / break amount
            force.setY(btScalar(report.get_value()));
            break;
        case InputReport::RT_CHANGE_SLIDE:
            /// Change to sideways crabbing control
            force.setX(btScalar(report.get_value()));
            break;
        case InputReport::RT_CHANGE_STEERING:
            /// Change to steering wheel angle
            torque.setZ(btScalar(-report.get_value()));
            break;
        default:
            // ignore menu commands, etc.
            break;
    }
    // record input for replay
    world.add_replay_report(report);
    update_sounds();
}

void Car::set_forces()
{
    rigid_body->clearForces();
    rigid_body->applyCentralForce(local_force);
    rigid_body->applyTorque(local_torque);  
}

void Car::posttick()
{
    if (m_removed) return;
#ifdef TRAIN_AI
    static std::ofstream data_file("ANN_training_data");
    static bool train_data_init = false;
    if (!train_data_init)
    {
        // artifical neural network training data file should begin with the
        // number of tests, the number of inputs, and the number of outputs.
        // We don't know the number of tests yet, but the rest we can provide:
        // The tests is the number of lines subtract 1 (this one), halved (in&out)
        data_file << "#tests "<< NUM_SENSORS << " 3\n";
        train_data_init = true;
    }
    // Save sensor data and response pairs for AI neural network training data.
    // The inputs to the neural network are the sensor data.
    // every tick is a bit excessive, so provide less data:
    if (world.get_tick_number() % 4 == 3)
    {
        float results[NUM_SENSORS];
        get_sensors(results);
        // infinite or NaN values might upset training.
        // Also, these are mostly produced when the edge avoision blending
        // gives no weight to the neural network anyway, so it doesn't make
        // sense to use a neural network on it.
        bool good = true;
        for (unsigned int i = 0; i < NUM_SENSORS; i++)
        {
            good &= std::isfinite(results[i]);
        }
        good &= results[4] > 0.0;
        if (good)
        {
            for (unsigned int i = 0; i < NUM_SENSORS; i++)
            {
                data_file << results[i] << ' ';
            }
            data_file << '\n';
            // The outputs are acceleration, slide, and steering,
            // scaled to the range -1 to 1:
            data_file << force.y()/32767.0 << ' ' << force.x()/32767.0 << ' ' << torque.z()/-32767.0 << '\n';
        }
    }
#endif
    
    btTransform transform = rigid_body->getCenterOfMassTransform();
    btTransform rotation_transform(transform);
    rotation_transform.setOrigin(btVector3(0, 0, 0));
    
    move_towards(transform.getOrigin());
    
    // passed start finish line?
    bool new_infront_of_start = transform.getOrigin().dot(plane_normal) + plane_distance > 0;
    if (new_infront_of_start != infront_of_start)
    {
        // is this near enough to the start finish line?
        /** @todo better lap detection.
         * We count it if the car is within 17 units of the start/finish line.
         * It is fiesable that this is wrong in some cases (e.g. on a mobius
         * stip this would count the lap twice.)
         */
        if (transform.getOrigin().distance2(start_point) < 289)
        {
            if (new_infront_of_start)
            {
                lap++;
                // has the user completed this lap before?
                if (lap > max_lap)
                {
                    // properly completed a lap.
                    unsigned long int tick = world.get_tick_number();
                    last_lap_ticks = tick - lap_start_ticks;
                    lap_start_ticks = tick;
                    max_lap = lap;
                    DEBUG_MESSAGE("Car " << this << " completed a lap in " << last_lap_ticks / 6000 << " min, " << (last_lap_ticks / 100) % 60 << " sec, " << last_lap_ticks % 100 << "0 msec");
                    if (best_lap_ticks > last_lap_ticks)
                    {
                        // new lap record for this session and car.
                        best_lap_ticks = last_lap_ticks;
                    }
                    if (lap == 4)
                    {
                        // end of the race
                        m_course_complete = true;
                        remove();
                        return;
                    }
                }
            } else {
                lap--;
            }
        }
        infront_of_start = new_infront_of_start;
        //DEBUG_MESSAGE("Car " << this << " is on lap " << lap);
    }
    
    // near booster?
    if (boost_timer >= 40)
    {
        const Track::Track & track = world.get_track();
        const Track::Path & path = track.get_path();
        typedef boost::graph_traits<Track::Path::Graph>::edge_iterator EdgeIterator;
        std::pair<EdgeIterator, EdgeIterator> edge_range;
        for (edge_range = boost::edges(path.graph);
             edge_range.first != edge_range.second;
             edge_range.first++)
        {
            const Track::PathEdge & edge = path.graph[*(edge_range.first)];
            typedef std::vector<boost::shared_ptr<Track::TrackAttachment> > Vector;
            const Vector & attachments = edge.get_attachments();
            for (Vector::const_iterator it = attachments.begin();
                 it != attachments.end();
                 it++)
            {
                const Track::TrackBooster * booster = dynamic_cast<const Track::TrackBooster*>(&(**it));
                if (booster)
                {
                    btScalar distance2 = booster->get_global_transform().
                                getOrigin().distance2(transform.getOrigin());
                    if (distance2 < 1.0)
                    {
                        // boost
                        boost_timer = 0;
                        m_boost_sound_source->play();
                        //DEBUG_MESSAGE("Boosted");
                    }
                }
            }
        }
        if (boost_timer < 100) boost_timer++;
    }
    else
    {
        boost_timer++;
    }
    
    // find where the line from the thruster in direction of force goes in world space.
    btVector3 check_vector = transform(btVector3(0.0, 0.0, -5.547));
    btVector3 start_vector = transform(btVector3(0.0, 0.0, -0.047));
    
    // raycast out to see if we find a surface.
    btCollisionWorld::ClosestRayResultCallback thruster_ray_callback(start_vector,
                                                                     check_vector);
    world.get_floor_world().rayTest(
                                        start_vector,
                                        check_vector,
                                        thruster_ray_callback);
    btScalar floor_distance = 200.0;
    const btScalar desired_height = 0.22;
    if (thruster_ray_callback.hasHit())
    {
        btVector3 & surface_normal = thruster_ray_callback.m_hitNormalWorld;
        floor_distance = surface_normal.dot(
                        start_vector - thruster_ray_callback.m_hitPointWorld);
        
        // stick to constant height above floor.
        if (floor_distance <= desired_height)
        {
            if (!floor_stick)
            {
                //DEBUG_MESSAGE("Sticking car to floor.");
            }
            floor_stick = true;
            rigid_body->setGravity(btVector3(0.0, 0.0, 0.0));
        }
        if (floor_stick)
        {
            // move to an almost constant height above the floor.
            // The /2.0 on the end is to reduce the effect, so distance varies
            // based on more realistic physics.
            btScalar mult = (desired_height - floor_distance) / 2.0 / 2.0;
            rigid_body->translate(surface_normal * mult);
            
            // now remove vertical motion.
            btVector3 linear_velocity = rigid_body->getLinearVelocity();
            linear_velocity = rotation_transform.inverse()(linear_velocity);
            /// @todo recycle some vertical momentum as forwards momentum
            linear_velocity.setZ(0.0);
            linear_velocity = rotation_transform(linear_velocity);
            rigid_body->setLinearVelocity(linear_velocity);
        }
        // Rotate so floor's normal is in the same direction as car's z axis.
        // find change in roll
        btVector3 vec_car_axis = rotation_transform(btVector3(0.0, 1.0, 0.0));
        btScalar cross_length = vec_car_axis.cross(surface_normal).length();
        // Make numerically stable: asin expects values <= 1, but sometimes
        // The cross product returns values greater than 1 (even though it
        // shouldn't).
        if (cross_length > 1.0) cross_length = 1.0;
        btScalar roll = cross_length ? M_PI / 2.0 - std::asin(cross_length) : 0.0;
        // reverse if the angle is on the other side
        if (vec_car_axis.dot(surface_normal) < 0.0)
        {
            roll = -roll;
        }
        
        // find change in pitch
        vec_car_axis = rotation_transform(btVector3(1.0, 0.0, 0.0));
        cross_length = vec_car_axis.cross(surface_normal).length();
        if (cross_length > 1.0) cross_length = 1.0;
        btScalar pitch = cross_length ? M_PI / 2.0 - std::asin(cross_length) : 0.0;
        if (vec_car_axis.dot(surface_normal) < 0.0)
        {
            pitch = -pitch;
        }
        btQuaternion rotation = rigid_body->getWorldTransform().getRotation();
        /* Smooth rotation so it is not so forceful before applying it.
         * This reduces problems with sharp bends and uneven surfaces, and 
         * makes it possible to fly off the crest of hills.
         */
        rotation *= btQuaternion(pitch / 4.0, -roll / 4.0, 0.0);
        rigid_body->getWorldTransform().setRotation(rotation);
        
        // turning left and right.
        btVector3 torque_vector = btVector3(0.0, 0.0,
            torque.getZ() * _car_stearing_scale[car_model] * _force_scale);
        local_torque = rotation_transform(torque_vector);
    } else {
        // allow reduced stearing in mid air.
        btVector3 torque_vector = btVector3(0.0, 0.0,
            torque.getZ() * _car_stearing_scale_midair[car_model] * _force_scale);
        local_torque = rotation_transform(torque_vector);
        // note that you carry on in roughly the same direction.
        // The sliding physics make it slightly more useful.
        if (floor_stick)
        {
            rigid_body->setGravity(btVector3(0.0, 0.0, -9.8));
            floor_stick = false;
            //DEBUG_MESSAGE("Car left floor.");
        }
    }
    // prevent sliding when not pressing slide key, and break when not accelerating.
    btVector3 unforce(-rotation_transform.inverse()(rigid_body->getLinearVelocity()) * 600.0);
    const btScalar sideways_velocity = -unforce.getX();
    const btScalar forward_velocity = -unforce.getY();
    /// @todo do this smoothly for anolgue input.
    unforce.setX(unforce.getX() * (1.0 - force.getX()/32767.0));
    unforce.setY(unforce.getY() * (1.0 - force.getY()/32767.0));
    // clamp to maximum achievable using other slide key or reverse, and don't change vertical motion.
    unforce.setMin(btVector3(32767, 32767, 0));
    unforce.setMax(btVector3(-32767, -16383, 0));
    
    // swap sideways velocity for forward velocity to help cornering
    // when sliding in the right direction, we want to reduce the effect though.
    btScalar slide_assist = sideways_velocity * force.getX();
    // If sliding the wrong way, don't reduce the effect.
    if (slide_assist < 0.0) slide_assist = 0.0;
    // If sliding the right way really fast, put a limit on the speed.
    if (slide_assist > 35.0) slide_assist = 35.0;
    unforce.setX((unforce.x() - sideways_velocity * (50.0 - slide_assist)) * _car_slide_scale[car_model]);
    
    if (sideways_velocity < 0)
    {
        unforce.setY(unforce.y() - sideways_velocity * 15.0);
    } else {
        unforce.setY(unforce.y() + sideways_velocity * 15.0);
    }
    /* additional break force when moving forwards or backwards while the
     * accelerate / reverse control is pushing the other way.
     */
    tick_energy_absorbed  = 0;
    if ((forward_velocity < 0.0 && force.y() > 0) ||
        (forward_velocity > 0.0 && force.y() < 0.0))
    {
        unforce.setY(unforce.y() + force.y() * 8.0);
        
        // Store energy lost while breaking.
        if (force.y() < 0)
        {
            tick_energy_absorbed = force.y() * forward_velocity * -0.05;
            elastic_potential_energy +=  tick_energy_absorbed;
        }
    }
    
    // additional force to use for accelerating varies with forward speed.
    // With a low speed the force is higher, providing greater acceleration.
    btScalar accel_multiplier = 5400000.0 / (forward_velocity * forward_velocity + 1800000.0);
    btScalar forward_accel_force = force.getY() * accel_multiplier;
    unforce.setY(unforce.getY() + forward_accel_force);
    
    // something similar for sliding sideways. Sharper, but half as strong.
    btScalar slide_accel_multiplier = 150000.0 / (sideways_velocity * sideways_velocity + 100000.0);
    // always use maximum if sliding the wrong way
    if (sideways_velocity * force.getX() < 0)
    {
        slide_accel_multiplier = 1.5;
    }
    btScalar slide_accel_force = force.getX() * slide_accel_multiplier;
    unforce.setX(unforce.getX() + slide_accel_force);
    
    // release stored energy while accelerating.
    used_energy_boost = 0.0;
    if (force.y() > 0)
    {
        btScalar transfer_force = force.y() * 3.0;
        used_energy_boost = transfer_force * forward_velocity * 0.01;
        if (used_energy_boost > elastic_potential_energy)
        {
            used_energy_boost = elastic_potential_energy;
            transfer_force = used_energy_boost / forward_velocity / 0.01;
        }
        unforce.setY(unforce.y() + transfer_force);
        elastic_potential_energy -= used_energy_boost;
    }
    
    // provide additional force when boosting.
    if (boost_timer < 100.0)
    {
        unforce.setY(unforce.getY() + float(force.getY()) * (float(100 - boost_timer) / 10.0));
    }
    
    if (floor_stick)
    {
        local_force = rotation_transform((force+unforce) * 0.012 * _force_scale);
    } else {
        local_force = rotation_transform((force * btVector3(2.0, 1.0, 1.0) + unforce) * 0.006 * _force_scale);
    }
    
    // Disqualify cars that have fallen off the course, or missed a section of
    // the track. If NearTrack cannot find a position within 15m of the track,
    // the car is disqualified.
    if (get_position().distance2(transform.getOrigin()) > 225.0)
    {
        remove();
    }
    
    update_sounds();
}

void Car::get_transform(btTransform & transform)
{
    if (m_removed)
    {
        transform = m_last_transform;
        return;
    }
    transform = rigid_body->getCenterOfMassTransform();
}

signed int Car::get_lap()
{
    return lap;
}

unsigned long int Car::get_last_lap_ticks()
{
    return last_lap_ticks;
}

unsigned long int Car::get_best_lap_ticks()
{
    return best_lap_ticks;
}

btScalar Car::get_speed()
{
    if (m_removed) return 0.0;
    return rigid_body->getLinearVelocity().length();
}

btScalar Car::get_lap_distance() const
{
    const Track::MeshFaces::FaceGraph face = m_graph[get_face_descriptor()];
    return face_u_interpolation(face, get_position());
}

void Car::remove()
{
    m_last_transform = rigid_body->getCenterOfMassTransform();
    m_removed = true;
    world.get_dynamics_world().removeRigidBody(rigid_body);
    delete rigid_body;
    m_course_complete_time = world.get_tick_number();
    m_engine_sound_source->stop();
    m_slide_sound_source->stop();
    m_release_energy_sound_source->stop();
    m_drag_sound_source->stop();
    m_break_sound_source->stop();
}

bool Car::get_finished() const
{
    return m_course_complete;
}

bool Car::get_disqualified() const
{
    return m_removed && (!m_course_complete);
}

unsigned long int Car::get_finish_time() const
{
    return m_course_complete_time;
}

int Car::get_facing_backwards() const
{
    if (m_removed)
    {
        // not facing any direction if removed from the scene.
        return 0;
    } else {
        // current position.
        const Track::MeshFaces::FaceGraph & face = m_graph[get_face_descriptor()];
        btVector3 current_coords = get_position();
        btScalar current_lap_position = face_u_interpolation(face, current_coords);
        
        // find point 15 meters infront of the car's centre.
        // The large distance means small bits where the path is backwards
        // like the entrance to the longest option in a split is ignored.
        btTransform transform = rigid_body->getCenterOfMassTransform();
        btVector3 facing_coords = transform(btVector3(0, 15.0, 0));
        Track::NearTrack front(*this);        
        front.move_towards(facing_coords);
        facing_coords = front.get_position();
        if (   facing_coords.dot(plane_normal) + plane_distance > 0.0
            && current_coords.dot(plane_normal) + plane_distance <= 0.0
            && facing_coords.distance2(start_point) < 289.0)
        {
            // The nose is infont of the finish line, but centre isn't.
            // The car must be facing the right way, even though the lap
            // position of the nose is less than the lap position of the
            // centre.
            return 0;
        }
        const Track::MeshFaces::FaceGraph & face2 = m_graph[front.get_face_descriptor()];
        btScalar facing_lap_position = face_u_interpolation(face2, facing_coords);
        btScalar difference = (current_lap_position - facing_lap_position) * 4.0 * 255;
        if (difference <= 0)
        {
            return 0;
        }
        else  if (difference >= 255)
        {
            return 255;
        }
        else
        {
            return int(difference);
        }
    }
}

btScalar Car::get_camera_height() const
{
    if (m_removed) return 0.3;
    // Use the car's NearTrack to get a point on the road infront of the car.
    // find point 12 meters infront of the car's centre.
    btTransform transform = rigid_body->getCenterOfMassTransform();
    btVector3 facing_coords = transform(btVector3(0, 12.0, 0));
    Track::NearTrack front(*this);        
    front.move_towards(facing_coords);
    facing_coords = front.get_position();
    // Now find the difference in height between this point and the car.
    facing_coords = transform.inverse()(facing_coords);
    btScalar height = facing_coords.getZ();
    // if the point is below, move the camera up to get a better view, if it
    // is above, move the camera down.
    btScalar cam_height = 0.35 - height * 0.15;
    if (cam_height >= 0) return cam_height;
    return 0;
}

void Car::get_sensors(float results[NUM_SENSORS]) const
{
    if (m_removed) return;
    
    // Feeler sensors
    
    // These return the available distance and relative position along the lap in
    // several directions.
    btTransform transform = rigid_body->getCenterOfMassTransform();
    // transformation between a global vector and the vector relative to the
    // car's rotation (y forwards along the car).
    btTransform rotate = transform;
    rotate.setOrigin(btVector3(0, 0, 0));
    for (unsigned int i = 0; i < NUM_SENSORS_FEELERS; i++)
    {
        scan_direction(rotate(feeler_vectors[i]*64.0), &(results[i*NUM_FEELER_PROPERTIES]));
    }

    // Non feeler sensors.
    btTransform unrotate = rotate.inverse();
    btVector3 current_coords = get_position();
    // sensors for motion: linear velocity and angular velocity.
    // remove rotation of the car to make it meaningful.
    const unsigned int nonfeeler_start = NUM_SENSORS_FEELERS*NUM_FEELER_PROPERTIES;
    const btVector3 velocity = unrotate(rigid_body->getLinearVelocity());
    results[nonfeeler_start] = velocity.x();
    results[nonfeeler_start + 1] = velocity.y();
    results[nonfeeler_start + 2] = velocity.z();
    const btVector3 ang_vel = unrotate(rigid_body->getAngularVelocity());
    results[nonfeeler_start + 3] = ang_vel.x() / 10.0;
    results[nonfeeler_start + 4] = ang_vel.y() / 10.0;
    results[nonfeeler_start + 5] = ang_vel.z();
    // stored energy
    results[nonfeeler_start + 6] = elastic_potential_energy / float(2 << 27);
    assert(results[nonfeeler_start + 6] > -1);
    // difference between position on the road and car position.
    const btVector3 position_diff = unrotate(rigid_body->getCenterOfMassPosition() - current_coords);
    results[nonfeeler_start + 7] = position_diff.x();
    results[nonfeeler_start + 8] = position_diff.y();
    results[nonfeeler_start + 9] = position_diff.z();
    // boost strength- 1 just after boost, 0 when no effect (after a second).
    results[nonfeeler_start + 10] = (100.0 - boost_timer) * 0.01;
    if (results[nonfeeler_start + 10] < 0) results[nonfeeler_start + 10] = 0;
    if (results[nonfeeler_start + 10] > 1) results[nonfeeler_start + 10] = 1;
    // gravity in z direction - so it knows when it is flying.
    results[nonfeeler_start + 11] = rigid_body->getGravity().z() / 9.8;
    
    // work out a possible direction for steering from the longest feeler distance.
    // since distances are inverted, we find the lowest number.
    unsigned int best_direction = 0;
    btScalar min = std::numeric_limits<btScalar>::max();
    // ignore the backwards one.
    for (unsigned int i = 0; i < NUM_SENSORS_FEELERS-1; i++)
    {
        if (min > results[i * NUM_FEELER_PROPERTIES + Track::SENSOR_NAVIGABLE_DISTANCE])
        {
            min = results[i * NUM_FEELER_PROPERTIES + Track::SENSOR_NAVIGABLE_DISTANCE];
            best_direction = i;
        }
    }
    // which direction to steer this way?
    if (feeler_vectors[best_direction].x() > 0.0)
    {
        results[nonfeeler_start + 12] = 1;
    }
    else if (feeler_vectors[best_direction].x() < 0.0)
    {
        results[nonfeeler_start + 12] = -1;
    }
    else // go straight ahead.
    {
        results[nonfeeler_start + 12] = 0;
    }
    // if it points backwards, do the opposite.
    if (results[best_direction * NUM_FEELER_PROPERTIES + Track::SENSOR_LAP_DIFFERENTIAL] < 0)
    {
        results[nonfeeler_start + 12] *= -1;
    }
    
    // work out a possible steering direction from the gradient of lap
    // distance of all the feelers.
    // this moves the car towards the shortest path, but not necessarily
    // the fastest line to take.
    // find the most positive gradient's direction.
    btScalar max = std::numeric_limits<btScalar>::min();
    for (unsigned int i = 0; i < NUM_SENSORS_FEELERS; i++)
    {
        if (max < results[i * NUM_FEELER_PROPERTIES + Track::SENSOR_LAP_DIFFERENTIAL])
        {
            max = results[i * NUM_FEELER_PROPERTIES + Track::SENSOR_LAP_DIFFERENTIAL];
            best_direction = i;
        }
    }
    // which direction to steer this way?
    if (feeler_vectors[best_direction].x() > 0.0)
    {
        results[nonfeeler_start + 13] = 1;
    }
    else if (feeler_vectors[best_direction].x() < 0.0)
    {
        results[nonfeeler_start + 13] = -1;
    }
    else
    {
        if (best_direction == 7)
        {
            // driving the wrong way.
            if (results[1 * NUM_FEELER_PROPERTIES + 4] > results[2 * NUM_FEELER_PROPERTIES + Track::SENSOR_LAP_DIFFERENTIAL])
            {
                // positive x better than negative x. Turn that way to face
                // the right direction.
                results[nonfeeler_start + 13] = 1;
            }
            else
            {
                results[nonfeeler_start + 13] = -1;
            }
        } else {
            // keep going straight ahead.
            results[nonfeeler_start + 13] = 0;
        }
    }
    
    ///@todo direction and distance to nearest booster(s).
    ///@todo direction and distance to nearest other car(s).
    ///@todo See if the scales can be removed once the AI is good enough.
    ///@todo See if any sensor can be removed once the AI is good enough.
}

void Car::update_sounds()
{
    // The sounds are stopped when the car goes.
    if (m_removed) return;
    
    // set the sounds' properties
    
    m_engine_sound_source->set_position(rigid_body->getCenterOfMassTransform().getOrigin());
    m_engine_sound_source->set_velocity(rigid_body->getLinearVelocity());
    m_engine_sound_source->set_gain(std::abs(force.y()/603456.0) + 0.1);
    m_engine_sound_source->set_pitch(0.5 + 0.1 * (rigid_body->getLinearVelocity().length()));
    
    m_drag_sound_source->set_position(rigid_body->getCenterOfMassTransform().getOrigin());
    m_drag_sound_source->set_velocity(rigid_body->getLinearVelocity());
    m_drag_sound_source->set_gain(rigid_body->getLinearVelocity().length() / 4800.0);
    m_drag_sound_source->set_pitch(0.5 + 0.12 * (rigid_body->getLinearVelocity().length()));
    
    m_break_sound_source->set_position(rigid_body->getCenterOfMassTransform().getOrigin());
    m_break_sound_source->set_velocity(rigid_body->getLinearVelocity());
    m_break_sound_source->set_gain(tick_energy_absorbed / float(2 << 23));
    m_break_sound_source->set_pitch(elastic_potential_energy / float(2 << 27));
    
    m_boost_sound_source->set_position(rigid_body->getCenterOfMassTransform().getOrigin());
    m_boost_sound_source->set_velocity(rigid_body->getLinearVelocity());
    
    if (used_energy_boost > 0)
    {
        m_release_energy_sound_source->set_position(rigid_body->getCenterOfMassTransform().getOrigin());
        m_release_energy_sound_source->set_velocity(rigid_body->getLinearVelocity());
        m_release_energy_sound_source->set_gain(5.0 * used_energy_boost / float(2<<23));
        m_release_energy_sound_source->set_pitch(1.0 + used_energy_boost / float(2<<21));
    } else {
        m_release_energy_sound_source->set_gain(0);
    }
    
    m_slide_sound_source->set_position(rigid_body->getCenterOfMassTransform().getOrigin());
    m_slide_sound_source->set_velocity(rigid_body->getLinearVelocity());
    ALfloat slide_gain = std::abs(force.x())/303456.0;
    m_slide_sound_source->set_gain(slide_gain);
}

}

}
