/** @file CarCamera.cpp
 *  @brief Implement the Engine::CarCamera class. 
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "CarCamera.h"
#include <GL/gl.h>
#include <GL/glu.h>

#include <cmath>

#include "../Graphics/Window.h"
#include "Audio.h"

class MyQuaternion : public btQuaternion
{
public:
    MyQuaternion(const btQuaternion & quaternion)
        :   btQuaternion(quaternion)
    {
    }
    
    // This function is based on code from Bullet, released under this license:  
    /* Copyright (c) 2003-2006 Gino van den Bergen / Erwin Coumans  http://continuousphysics.com/Bullet/

    This software is provided 'as-is', without any express or implied warranty.
    In no event will the authors be held liable for any damages arising from the use of this software.
    Permission is granted to anyone to use this software for any purpose, 
    including commercial applications, and to alter it and redistribute it freely, 
    subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you must not claim that you wrote the original software. If you use this software in a product, an acknowledgment in the product documentation would be appreciated but is not required.
    2. Altered source versions must be plainly marked as such, and must not be misrepresented as being the original software.
    3. This notice may not be removed or altered from any source distribution.
    */
    /// @todo this is in development version of Bullet. Remove this class and use it when released.
    btVector3 getAxis() const
    {
        btScalar s_squared = btScalar(1.) - btPow(m_floats[3], btScalar(2.));
        if (s_squared < btScalar(10.) * SIMD_EPSILON) //Check for divide by zero
        {
            return btVector3(1.0, 0.0, 0.0);  // Arbitrary
        }
        btScalar s = btSqrt(s_squared);
        return btVector3(m_floats[0] / s, m_floats[1] / s, m_floats[2] / s);
    }

};

namespace Engine
{

CarCamera::CarCamera(GameObjects::Car & car, Physics::World & world)
    :   car(car),
        world(world)
    ,   m_velocity(btVector3(0.0, 0.0, 0.0))
{
    world.add_tick_observer(this);
    
    // set inital camera position.
    btTransform car_transform;
    car.get_transform(car_transform);
    btTransform car_rotation_transform = car_transform;
    car_rotation_transform.setOrigin(btVector3(0, 0, 0));
    location = car_transform(btVector3(0.0, -0.70, car.get_camera_height()));
    up = car_rotation_transform(btVector3(0.0, 0.0, 1.0));
    centre = car_transform(btVector3(0.0, 1.0, 0.1));
    // We need the listener set so the sound is right before the game starts.
    posttick();
}

CarCamera::~CarCamera()
{
    world.remove_tick_observer(this);
}

void CarCamera::full_transform()
{
    gluLookAt(location.x(), location.y(), location.z(),
              centre.x(), centre.y(), centre.z(),
              up.x(), up.y(), up.z()); 
}

void CarCamera::rotation_transform()
{
    gluLookAt(0.0, 0.0, 0.0,
              centre.x() - location.x(), centre.y() - location.y(), centre.z() - location.z(),
              up.x(), up.y(), up.z());
}

void CarCamera::posttick()
{
    const float blend = 0.3;
    btTransform car_transform;
    car.get_transform(car_transform);
    btTransform car_rotation_transform = car_transform;
    car_rotation_transform.setOrigin(btVector3(0, 0, 0));
    // find a new place behind the car looking at it.
    btVector3 new_position(car_transform(btVector3(0.0, -0.70, car.get_camera_height())));
    btVector3 new_up(car_rotation_transform(btVector3(0.0, 0.0, 1.0)));
    btVector3 new_centre = car_transform(btVector3(0.0, 1.0, 0.1));
    
    // blend new position with the last camera position to soften motion.
    m_velocity = location;
    location = location.lerp(new_position, blend);
    m_velocity -= location;
    up = up.lerp(new_up, blend);
    up.normalize();
    centre = centre.lerp(new_centre, blend);
    
    // set sound listener to this position, orientation, and velocity.
    m_listener.set_velocity(m_velocity * 100.0);
    m_listener.set_position(location, centre - location, up);
}

void CarCamera::update_occlusion_tester(Track::OcclusionTester & occlusion_tester, btScalar aspect) const
{
    // front plane
    ///@todo The front plane is actually 0.03 units infront of the camera.
    
    btVector3 front_normal = (centre - location).normalized();
    // find the distance to it along this normal.
    btScalar front_distance = (-location).dot(front_normal);
    occlusion_tester.plane_vectors[0] = front_normal;
    occlusion_tester.plane_distances[0] = front_distance;
    
    btVector3 sideways = up.cross(front_normal).normalized();
    btVector3 real_up = sideways.cross(front_normal);
    
    // side planes
    // fovy = 65 degrees = 1.134464014 radians.
    btScalar half_fovy = 0.567232006;
    const Graphics::Window window = Graphics::Window::get_instance();
    btScalar half_fovx = std::atan(std::tan(half_fovy) * aspect);

    btVector3 left_normal = real_up.cross(btTransform(btQuaternion(real_up, -half_fovx))(front_normal)).normalized();
    occlusion_tester.plane_vectors[1] = left_normal;
    occlusion_tester.plane_distances[1] = (-location).dot(left_normal);
    
    btVector3 right_normal = -real_up.cross(btTransform(btQuaternion(real_up, half_fovx))(front_normal)).normalized();
    occlusion_tester.plane_vectors[2] = right_normal;
    occlusion_tester.plane_distances[2] = (-location).dot(right_normal);
    
    btVector3 top_normal = sideways.cross(btTransform(btQuaternion(sideways, -half_fovy))(front_normal)).normalized();
    occlusion_tester.plane_vectors[3] = top_normal;
    occlusion_tester.plane_distances[3] = (-location).dot(top_normal);
    
    btVector3 bottom_normal = -sideways.cross(btTransform(btQuaternion(sideways, half_fovy))(front_normal)).normalized();
    occlusion_tester.plane_vectors[4] = bottom_normal;
    occlusion_tester.plane_distances[4] = (-location).dot(bottom_normal);
    
    occlusion_tester.camera_position = centre;
}

const btVector3 & CarCamera::get_location() const
{
    return location;
}

const btVector3 & CarCamera::get_velocity() const
{
    return m_velocity;
}

}
