/** @file racer/Engine/Audio.h
 *  @brief Declare the Engine::Audio class, and the related classes.
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef ENGINE_AUDIO_H
#define ENGINE_AUDIO_H

#include <AL/alut.h>
#include <LinearMath/btVector3.h>
#include <LinearMath/btTransform.h>
#include <boost/shared_ptr.hpp>
#include <vector>

namespace Engine
{

class SoundSource;
class SoundListener;
class SoundInstance;
class ALSoundSource;

/** Singleton for managing sound effects and music.
*/
class Audio
{
    private:
        Audio();
    public:
        ~Audio();
        static Audio & get_instance();
        /// Pause all playing sound effects
        void pause();
        /// Resume all paused sound effects.
        void resume();
        /** Return the paused state.
         * @return true if resume() was called after the last call to pause(),
         * or if pause() has not been called. Returns false otherwise.
         */
        bool get_paused();
        /** Commit all changes to the listeners and sound sources.
         * Sounds will begin to play at their set gains and pitches.
         * Should be called once per tick after sounds have been set using
         * SoundSources and Listeners.
         */
        void commit();
        /** Remember a listener.
         * Used when a listener is constructed.
         */
        void attach_listner(SoundListener * listener);
        /** Remember a sound source
         * Used when a SoundSource is constructed.
         */
        void attach_source(SoundSource * source);
        /** Forget about a listener.
         * Used when the listener is destructing. 
         */
        void forget_listener(SoundListener * listener);
        /** Forget about a sound source
         */
        void forget_source(SoundSource * source);
    private:
        /** The playing sound instances for positional audio.
         * Sorted by listner then SoundSource, since the listners are created
         * and destroyed less often, and they can cause the most moves when
         * created or destroyed.
         */
        std::vector<boost::shared_ptr<SoundInstance> > m_instances;
        /** Listeners
         * Ordered as in m_instances.
         */
        std::vector<SoundListener *> m_listeners;
        /** SoundSources
         * Ordered as in m_instances.
         */
        std::vector<SoundSource *> m_sources;
        /** OpenAL sources.
         * There is a hardware / sound driver limit on the number available,
         * and also performance problems with large numbers of sounds.
         * Therefore we have a fixed limited number of OpenAL sources. Only
         * the most audiable sounds are played.
         */
        std::vector<boost::shared_ptr<ALSoundSource> > m_al_sources;
        
        /** The speed of sound for doppler calculations.
         * It isn't used to delay distant sounds, only change the pitch when
         * something moves.
         */
        ALfloat m_speed_of_sound;
        
        /// True if all sounds have been paused.
        bool m_paused;
};

/** A sound loaded from a file.
 * A single SoundBuffer can be used by multiple SoundSources.
 * It is a c++ wrapper around an OpenAL buffer.
 */
class SoundBuffer
{
    public:
        /// Create from file
        SoundBuffer(const char * filename);
        ~SoundBuffer();
        // get the openAL buffer name.
        ALuint get_al_buffer() const;
    private:
        ALuint m_buffer;
        // no copying or assignment
        SoundBuffer(const SoundBuffer & source);
        SoundBuffer & operator=(const SoundBuffer & source);
};

/** A point where sound is heard from.
 * There should be one for each camera.
 * Unlike OpenAL, there can be more than one listener.
 */
class SoundListener
{
    friend class Audio;
    public:
        SoundListener();
        ~SoundListener();
        /** Set the position and orientation of the listener.
         * The forward and up vectors will be normalised.
         * @param position The location of the listener in world space.
         * @param forward The direction the listener is facing.
         * @param up The direction the listener considers up.
         */
        void set_position(btVector3 position, btVector3 forward, btVector3 up);
        /** Set the velocity of the listner.
         * Used for doppler shift calculation.
         * @param velocity Velocity of the distance in world space in spacial
         * units per second.
         */
        void set_velocity(btVector3 velocity);
    private:
        btVector3 m_position;
        btVector3 m_forward;
        btVector3 m_up;
        btVector3 m_velocity;
        btTransform m_inverse_transform;
        // no copying or assignment
        SoundListener(const SoundListener & source);
        SoundListener & operator=(const SoundListener & source);
};

/** Plays a sound.
 * Makes a sound when play() is called. Deleting stops the noise.
 * First load the sound you want using a SoundBuffer.
 * 
 * This is not directly a wrapper for openAL's sources.
 * A SoundSource creates as many OpenAL sources as there are SoundListners.
 * OpenAL is fixed to one listner.
 */
class SoundSource
{
    friend class Audio;
    public:
        /** Create attached to a specific buffer.
         * The buffer contains the sound that will be played, so don't delete
         * it until you have deleted the SoundSource.
         */
        SoundSource(const SoundBuffer & buffer);
        ~SoundSource();
        /** Make the sound either stop when it finishes or loop continuously.
         * @param loop true to repeat, false to play once.
         */
        void set_looping(bool loop = true);
        
        /// Set the global position where the sound originates from.
        void set_position(btVector3 position);
        /// Set the velocity of the sound for doppler shift.
        void set_velocity(btVector3 velocity);
        /** Set how loud the noise is.
         * @param gain Gain for the sound. Must be positive.
         */
        void set_gain(ALfloat gain);
        
        void set_pitch(ALfloat pitch);
        
        /// Start making the noise
        void play();
        
        /// Stop making the noise.
        void stop();
        
        const SoundBuffer & get_buffer() const;
    private:
        const SoundBuffer & m_buffer;
        
        bool m_looping;
        ALfloat m_gain;
        ALfloat m_pitch;
        btVector3 m_position;
        btVector3 m_velocity;
        bool m_playing;
        /** True if OpenAL started playing, false if it should start from the
         *  begining.
         */
        bool m_started;
        // no copying or assignment
        SoundSource(const SoundSource & source);
        SoundSource & operator=(const SoundSource & source);
};

/** A sound heard by a SoundListener because of a SoundSource.
 * May or may not be actually played.
 * Only the most audioable sounds are fed into OpenAL.
 * The less audiable ones are ignored for performance reasons.
 */
class SoundInstance
{
    public:
        /** Create associating with an OpenAL buffer object's name.
         */
        SoundInstance(ALuint buffer);
        ~SoundInstance();
        /// Return the buffer name given on creation.
        ALuint get_buffer() const;
        /// return true if an ALSoundSource is being used.
        bool get_assigned() const;
        /// Get the index value passed when assigned
        unsigned int get_index() const;
        /** Use an ALSoundSource.
         * After this, the ALSoundSource is not assigned to any SoundInstance
         * it was assigned to before.
         * This will make the sound actually audiable.
         */
        void assign(ALSoundSource & source, unsigned int index);
        /// Stop using an ALSoundSource.
        void unassign();
        /** Set how loud the sound should be played.
         * 1 is normal volume, 0.5 is -6dB, 0.25 is -25dB, etc.
         * 0 is silent.
         */
        void set_gain(ALfloat gain);
        /// return the gain set by set_gain()
        ALfloat get_gain() const;
        /// Set the scale for the pitch of the sound (1 is normal)
        void set_pitch(ALfloat pitch);
        /// Set the position of the sound relative to the viewer.
        void set_position(btVector3 position);
        /** Set whether the sound should repeat continuosly.
         * A looping sound source can be assigned an ALSoundSource at any
         * time. A sound source which doesn't loop that could not be assigned
         * an ALSoundSource will never be heard.
         * The looping value shouldn't be changed while playing and attached
         * to a ALSoundSource. You can call this function with the previous
         * state again though, it will just do nothing.
         */
        void set_looping(bool looping = true);
        /// Return true iff the sound should repeat continuosly.
        bool get_looping() const;
        
        /// Play the sound.
        void play();
        /// Stop the sound from playing.
        void stop();
        /// Return true iff the sound is playing.
        bool get_playing() const;
        
        /// Update the assigned ALSoundSource
        void update_al();
    private:
        ALSoundSource * m_source;
        unsigned int m_source_index;
        ALuint m_buffer;
        ALfloat m_gain;
        ALfloat m_pitch;
        btVector3 m_position;
        bool m_looping;
        bool m_playing;
        bool m_assigning;
};

/** Wrapper for an OpenAL source.
 */
class ALSoundSource
{
    public:
        ALSoundSource();
        ~ALSoundSource();
        ALuint get_name();
        /// Get the SoundInstance that was last assigned, if any.
        SoundInstance * get_instance();
        /// Assign a SoundInstance.
        void assign(SoundInstance & si);
        /** Stop playing any sound, and forget the last assigment.
         */
        void unassign();
    private:
        ALuint m_name;
        SoundInstance * m_instance;
        // no copying or assignment
        ALSoundSource (const ALSoundSource & source);
        ALSoundSource & operator=(const ALSoundSource & source);
};

class AudioError
{
};

}// namespace Engine

#endif // ndef ENGINE_AUDIO_H

