/** @file racer/Engine/Audio.cpp
 *  @brief Implement the Engine::Audio class and related classes.
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "Audio.h"

#include <iostream>
#include <set>
#include <algorithm>

#include <Debug.h>

namespace Engine
{

Audio::Audio()
    // Scale is about 1 distance unit = 6 meters.
    // so speed of sound in air ~ 330 m/s / 6 = 55 units per second.
    // in space the speed of sound should be much lower, but it sounds weird.
    :   m_speed_of_sound (300.0)
    ,   m_paused(false)
{
    // Open the default audio device
    alutInit(0, 0);
    alDistanceModel(AL_NONE);
    // Allocate some ALSoundSources.
    for (unsigned int i = 0; i < 30; i++)
    {
        try
        {
            m_al_sources.push_back(boost::shared_ptr<ALSoundSource>(new ALSoundSource));
        }
        catch (AudioError t)
        {
            // Probably hit some hardware limit.
            // Nothing to worry about, we will just have less
            // simultaneously playing sounds than normal.
            return;
        }
    }
    
}

Audio::~Audio()
{
    alutExit();
}

Audio & Audio::get_instance()
{
    static Audio * instance = new Audio();
    return *instance;
}

void Audio::pause()
{
    m_paused = true;
    // pause all playing sources
    alcSuspendContext(alcGetCurrentContext());
    for (std::vector<boost::shared_ptr<ALSoundSource> >::iterator it = m_al_sources.begin();
         it != m_al_sources.end();
         it++)
    {
        ALuint name = (**it).get_name();
        ALint state;
        alGetSourcei(name, AL_SOURCE_STATE, &state);
        if (state == AL_PLAYING)
        {
            alSourcePause(name);
        }
    }
    alcProcessContext(alcGetCurrentContext());
}

void Audio::resume()
{
    m_paused = false;
    // resume all paused sources.
    alcSuspendContext(alcGetCurrentContext());
    for (std::vector<boost::shared_ptr<ALSoundSource> >::iterator it = m_al_sources.begin();
         it != m_al_sources.end();
         it++)
    {
        ALuint name = (**it).get_name();
        ALint state;
        alGetSourcei(name, AL_SOURCE_STATE, &state);
        if (state == AL_PAUSED)
        {
            alSourcePlay(name);
        }
    }
    alcProcessContext(alcGetCurrentContext());
}

bool Audio::get_paused()
{
    return m_paused;
}

class SoundInstancePtrGainCompare
{
    public:
        bool operator () (SoundInstance * sound_instance_1,
                          SoundInstance * sound_instance_2)
        {
            return sound_instance_1->get_gain() <
                   sound_instance_2->get_gain();
        }
};

void Audio::commit()
{
    // Update each ALSoundSource's values to match listener and source
    std::vector<boost::shared_ptr<SoundInstance> >::iterator instance_it = m_instances.begin();
    std::vector<SoundListener *>::iterator listener_it;
    std::vector<SoundSource *>::iterator source_it;
    // This will store the sounds we want to be heard.
    std::multiset<SoundInstance *, SoundInstancePtrGainCompare> audiable_sounds;
    
    // Anything at this gain or below is not mixed.
    ALfloat min_gain = 0.0;
    for (source_it = m_sources.begin(); source_it != m_sources.end(); source_it++)
    {
        SoundSource & source = **source_it;
        for (listener_it = m_listeners.begin(); listener_it != m_listeners.end(); listener_it++)
        {
            SoundListener & listener = **listener_it;
            SoundInstance & instance = (**instance_it);
            ALfloat distance = listener.m_position.distance(source.m_position);
            // gain has inverse attenuation.
            // The gain behaviour in openAL makes the inverse square rule wrong.
            ALfloat gain = source.m_gain/distance;
            instance.set_gain(gain);
            
            // find the position local to the viewer.
            btVector3 position = listener.m_inverse_transform(source.m_position);
            instance.set_position(position);
            
            // find the pitch from the relative velocity
            btVector3 relative_position = source.m_position - listener.m_position;
            ALfloat s_listener = relative_position.dot(source.m_velocity);
            ALfloat s_source = relative_position.dot(listener.m_velocity);
            ALfloat pitch = (m_speed_of_sound + s_listener) / (m_speed_of_sound + s_source) * source.m_pitch;
            if (pitch <= 0) pitch = 0.001;
            instance.set_pitch(pitch);
            
            instance.set_looping(source.m_looping);
            
            // playing state.
            // Non looping sounds that have to low a gain at the start are
            // skipped.
            if (   source.m_playing
                && !source.m_started
                && (source.m_looping || gain > min_gain))
            {
                instance.play();
            }
            else if (!(source.m_playing) || !(source.m_looping || source.m_started))
            {
                instance.stop();
            }
            
            if (instance.get_playing() && gain > min_gain)
            {
                // should be audiable.
                audiable_sounds.insert(&instance);
            }
            
            instance_it++;
        }
        // has been started for all listeners, it shouldn't be restarted next
        // time as that would cause it to play from the beginning again if it
        // wasn't supposed to loop.
        if (source.m_playing)
        {
            source.m_started = true;
        }
    }
    
    // queue up the changes and submit them all at once
    // (if supported, otherwise this is a no-op.)
    alcSuspendContext(alcGetCurrentContext());
    
    // We want to play the loudest m_al_sources.size() sounds.
    unsigned int count = 0;
    std::vector<bool> available(m_al_sources.size(), true);
    std::vector<SoundInstance *> remaining;
    // update the loudest m_al_sources.size() sounds that have already been
    // assigned an ALSoundSource.
    std::multiset<SoundInstance *, SoundInstancePtrGainCompare>::reverse_iterator it = audiable_sounds.rbegin();
    for (;
         it != audiable_sounds.rend() && count < m_al_sources.size();
         it++, count++)
    {
        if ((**it).get_assigned())
        {
            (**it).update_al();
            available[(**it).get_index()] = false;
        } else {
            // We need to find a less audiable sound to replace.
            remaining.push_back(*it);
        }
    }
    // The remaining non looping sound instances should be stopped, so that
    // they don't play out of time when a ALSoundSource becomes available.
    for (; it != audiable_sounds.rend(); it++)
    {
        if (!(**it).get_looping())
        {
            (**it).stop();
        }
    }
    // Reassign ALSoundSources for the remaining SoundInstances.
    unsigned int alindex = 0;
    for (std::vector<SoundInstance *>::iterator it = remaining.begin();
         it != remaining.end(); it++)
    {
        while (!available[alindex]) {alindex++;}
        assert(alindex < m_al_sources.size());
        (**it).assign(*(m_al_sources[alindex]), alindex);
        available[alindex] = false;
    }
    // unassign any remaining sources.
    while (alindex < m_al_sources.size())
    {
        if (available[alindex])
        {
            m_al_sources[alindex]->unassign();
        }
        alindex++;
    }
    
    alcProcessContext(alcGetCurrentContext());
    
    // If a non looping SoundSource has no playing instances, stop it.
    std::size_t source_index = 0;
    std::size_t diff = m_sources.size();
    for (source_it = m_sources.begin(); source_it != m_sources.end(); source_it++, source_index++)
    {
        SoundSource & source = **source_it;
        if (source.m_playing && !source.m_looping)
        {
            // set playing to true if there is an instance that is playing it.
            bool playing = false;
            for (std::size_t i = source_index;
                 i < m_instances.size();
                 i += diff)
            {
                const SoundInstance & instance = *(m_instances[i]);
                if (instance.get_assigned())
                {
                    if (instance.get_playing())
                    {
                        playing = true;
                        break;
                    }
                }
            }
            // no playing instance.
            // stop trying to play the sound.
            if (!playing) source.m_playing = false;
        }
    }
    
    // We could do this after every call, but it costs a lot in performance.
    ALenum error = alGetError();
    if (error != AL_NO_ERROR)
    {
        std::cerr << "OpenAL error: "
                  << alutGetErrorString(error) << "\n";
        DEBUG_MESSAGE("here.");
        throw AudioError();
    }
}

void Audio::attach_listner(SoundListener * listener)
{
    // add a OpenAL source for each SoundSource.
    // There should be one after each listener group of the existing ones.
    std::size_t frequency = m_listeners.size();
    m_instances.reserve((frequency+1)*m_instances.size());
    std::vector<boost::shared_ptr<SoundInstance> >::iterator it = m_instances.begin();
    for (std::size_t i = 0; i < m_sources.size(); i++)
    {
        it += frequency;
        it = m_instances.insert(it,
                                boost::shared_ptr<SoundInstance>(new SoundInstance(m_sources[i]->get_buffer().get_al_buffer())));
        it++;
    }
    m_listeners.push_back(listener);
}

/** @todo The mapping between Audio::SoundSources and OpenAL sound sources
 * isn't scalable. Try using a fixed number n of OpenAL sources and switching
 * to the loudest n Audio::SoundSources. Try about 30. On my system, having
 * many sound sources causes an error. Having more than a few causes terribly
 * low refresh rate.
 */

void Audio::attach_source(SoundSource * source)
{
    // add an OpenAL source for each SoundListener
    // they can all go at the end of the existing vector.
    m_sources.push_back(source);
    
    std::size_t amount = m_listeners.size();
    ALint al_buffer = source->get_buffer().get_al_buffer();
    for (std::size_t i = 0; i < amount; i++)
    {
        m_instances.push_back(boost::shared_ptr<SoundInstance>(new SoundInstance(al_buffer)));
    }
    assert(m_sources.size() * m_listeners.size() == m_instances.size());
}

void Audio::forget_listener(SoundListener * listener)
{
    // remove an openAL source for each SoundSource
    // They are spaced evenly
    // It is more efficient to remove them backwards, since less will be moved
    // for the deletes near the beginning of the list.
    std::size_t frequency = m_listeners.size();
    
    // The index of the listner is used as offsets in m_alsources too.
    std::size_t index = 0;
    std::vector<SoundListener *>::iterator l_it;
    for (l_it = m_listeners.begin(); *l_it != listener; l_it++)
    {
        index++;
    }
    
    for (std::size_t i = m_sources.size() - 1; i != std::size_t(-1) ; i--)
    {
        std::vector<boost::shared_ptr<SoundInstance> >::iterator it =
            m_instances.begin() + (index + frequency * i);
        m_instances.erase(it);
    }
    
    m_listeners.erase(l_it);
    assert(m_sources.size() * m_listeners.size() == m_instances.size());
    
    // commit to this, it will stop ALSoundSources using buffers that could
    // be deleted.
    commit();
}

void Audio::forget_source(SoundSource * source)
{
    // remove a SoundSource for each SoundListner
    std::size_t index = 0;
    std::vector<SoundSource *>::iterator source_it;
    for (source_it = m_sources.begin(); *source_it != source; source_it++)
    {
        index++;
    }
    std::size_t scale = m_listeners.size();
    m_instances.erase(m_instances.begin()+scale*index,
                      m_instances.begin()+scale*(index+1));
    m_sources.erase(source_it);
    assert(m_sources.size() * m_listeners.size() == m_instances.size());
}

SoundBuffer::SoundBuffer(const char * filename)
{
    // make sure audio has been initialised
    Audio::get_instance();
    m_buffer = alutCreateBufferFromFile (filename);
    if (m_buffer == AL_NONE)
    {
        std::cerr << "Cannot create SoundBuffer:\nalutCreateBufferFromFile("
                  << filename << ") caused: "
                  << alutGetErrorString(alutGetError()) << "\n";
        DEBUG_MESSAGE("here.");
        throw AudioError();
    }
}

SoundBuffer::~SoundBuffer()
{
    alDeleteBuffers(1, &m_buffer);
    ALenum error = alGetError();
    if (error != AL_NO_ERROR)
    {
        std::cerr << "Cannot delete SoundBuffer:\nalDeleteBuffers caused: "
                  << alutGetErrorString(error) << "\n";
        DEBUG_MESSAGE("here.");
        throw AudioError();
    }
}

ALuint SoundBuffer::get_al_buffer() const
{
    return m_buffer;
}

SoundListener::SoundListener()
    :   m_position(btVector3(0, 0, 0))
    ,   m_forward(btVector3(0, 1, 0))
    ,   m_up(btVector3(0, 0, 1))
    ,   m_velocity(btVector3(0, 0, 0))
{
    Audio::get_instance().attach_listner(this);
}

SoundListener::~SoundListener()
{
    Audio::get_instance().forget_listener(this);
}

void SoundListener::set_position(btVector3 position, btVector3 forward, btVector3 up)
{
    m_position = position;
    m_forward = forward;
    m_up = up;
    btScalar matrix[16];
    forward.normalize();
    up.normalize();
    btVector3 cross = forward.cross(up);
    matrix[0] = cross.x(); matrix[1] = up.x(); matrix[2] = -forward.x(); matrix[3] = 1.0;
    matrix[4] = cross.y(); matrix[5] = up.y(); matrix[6] = -forward.y(); matrix[7] = 1.0;
    matrix[8] = cross.z(); matrix[9] = up.z(); matrix[10] = -forward.z(); matrix[11] = 1.0;
    matrix[12] = position.x(); matrix[13] = position.y(); matrix[14] = position.z();
    matrix[15] = 1.0;
    
    m_inverse_transform.setFromOpenGLMatrix(matrix);
    m_inverse_transform = m_inverse_transform.inverse();
}

void SoundListener::set_velocity(btVector3 velocity)
{
    m_velocity = velocity;
}

SoundSource::SoundSource(const SoundBuffer & buffer)
    :   m_buffer(buffer)
    ,   m_looping(false)
    ,   m_gain(1.0)
    ,   m_pitch(1.0)
    ,   m_position(btVector3(0, 0, 0))
    ,   m_velocity(btVector3(0, 0, 0))
    ,   m_playing(false)
    ,   m_started(false)
{
    Audio::get_instance().attach_source(this);
}

SoundSource::~SoundSource()
{
    Audio::get_instance().forget_source(this);
}

void SoundSource::set_looping(bool loop)
{
    m_looping = loop;
}

bool SoundInstance::get_looping() const
{
    return m_looping;
}

void SoundSource::set_position(btVector3 position)
{
    m_position = position;
}

void SoundSource::set_velocity(btVector3 velocity)
{
    m_velocity = velocity;
}

void SoundSource::set_gain(ALfloat gain)
{
    m_gain = gain;
}

void SoundSource::set_pitch(ALfloat pitch)
{
    m_pitch = pitch;
}

void SoundSource::play()
{
    m_playing = true;
    m_started = false;
}

void SoundSource::stop()
{
    m_playing = false;
}

const SoundBuffer & SoundSource::get_buffer() const
{
    return m_buffer;
}

SoundInstance::SoundInstance(ALuint buffer)
    :   m_source(0)
    ,   m_buffer(buffer)
    ,   m_gain(1)
    ,   m_pitch(1)
    ,   m_position(btVector3(0, 0, 0))
    ,   m_looping(false)
    ,   m_playing(false)
    ,   m_assigning(false)
{
}

SoundInstance::~SoundInstance()
{
    // unassign any assigned sound source.
    if (m_source)
    {
        m_source->unassign();
    }
}

bool SoundInstance::get_assigned() const
{
    return bool(m_source);
}

ALuint SoundInstance::get_buffer() const
{
    return m_buffer;
}

void SoundInstance::assign(ALSoundSource & source, unsigned int index)
{
    m_source = &source;
    m_source_index = index;
    
    // this will call update_al.
    m_assigning = true;
    m_source->assign(*this);
    m_assigning = false;
}

void SoundInstance::unassign()
{
    m_source = 0;
    if (!m_looping)
    {
        // if unassigned, a non looping sound cannot be continued.
        m_playing = false;
    }
}

void SoundInstance::set_gain(ALfloat gain)
{
    m_gain = gain;
}

ALfloat SoundInstance::get_gain() const
{
    return m_gain;
}

void SoundInstance::set_pitch(ALfloat pitch)
{
    m_pitch = pitch;
}

void SoundInstance::set_position(btVector3 position)
{
    m_position = position;
}

void SoundInstance::set_looping(bool looping)
{
    m_looping = looping;
}


void SoundInstance::play()
{
    if (m_playing)
    {
        // already playing, restart the sound.
        if (m_source)
        {
            alSourcePlay(m_source->get_name());
        }
    }
    m_playing = true;
}

void SoundInstance::stop()
{
    m_playing = false;
}

bool SoundInstance::get_playing() const
{
    return m_playing;
}

void SoundInstance::update_al()
{
    if (!m_source) return;
    
    ALuint al_source_name = m_source->get_name();
    alSourcef(al_source_name, AL_GAIN, m_gain);
    alSource3f(al_source_name, AL_POSITION,
            m_position.x(), m_position.y(), m_position.z());
    alSourcef(al_source_name, AL_PITCH, m_pitch);
        ALenum error = alGetError();
    if (error != AL_NO_ERROR)
    {
        std::cerr << "OpenAL error: "
                  << alutGetErrorString(error) << "\n";
        DEBUG_MESSAGE("here.");
        throw AudioError();
    }
    // Looping shouldn't change while playing.
    // It is actually handled by ALSoundSource when first attached.
    
    if (!m_looping && !m_assigning)
    {
        // if the (not looped) sound finishes, give up the slot.
        ALint state;
        alGetSourcei(al_source_name, AL_SOURCE_STATE, &state);
        if (state == AL_STOPPED)
        {
            m_playing = false;
            unassign();
        }
    }
}

unsigned int SoundInstance::get_index() const
{
    return m_source_index;
}

ALSoundSource::ALSoundSource()
    :   m_instance(0)
{
    alGenSources (1, &m_name);
    ALenum error = alGetError();
    if (error != AL_NO_ERROR)
    {
        std::cerr << "Cannot create SoundSource:\nalGenSources caused: "
                  << alutGetErrorString(error) << "\n";
        DEBUG_MESSAGE("here.");
        throw AudioError();
    }
}

ALSoundSource::~ALSoundSource()
{
    alDeleteSources(1, &m_name);
    ALenum error = alGetError();
    if (error != AL_NO_ERROR)
    {
        std::cerr << "Cannot delete SoundSource:\nalDeleteSources caused: "
                  << alutGetErrorString(error) << "\n";
        DEBUG_MESSAGE("here.");
        throw AudioError();
    }
}

ALuint ALSoundSource::get_name()
{
   return m_name;
}

SoundInstance * ALSoundSource::get_instance()
{
    return m_instance;
}

void ALSoundSource::unassign()
{
    if (!m_instance)
    {
        return;
    }
    else
    {
        alSourceStop(m_name);
        ALenum error = alGetError();
        if (error != AL_NO_ERROR)
        {
            std::cerr << "OpenAL error: "
                      << alutGetErrorString(error) << "\n";
            DEBUG_MESSAGE("here.");
            throw AudioError();
        }
        m_instance->unassign();
        m_instance = 0;
        // Incase the buffer gets deleted, stop using it.
        alSourcei (m_name, AL_BUFFER, 0);
    }
}

void ALSoundSource::assign(SoundInstance & si)
{
    if (m_instance)
    {
        // stop using the previous assignment.
        unassign();
    }
    m_instance = &si;
    ALenum error = alGetError();
    if (error != AL_NO_ERROR)
    {
        std::cerr << "OpenAL error: "
                  << alutGetErrorString(error) << "\n";
        DEBUG_MESSAGE("here.");
        throw AudioError();
    }
    alSourcei(m_name, AL_BUFFER, m_instance->get_buffer());
    //DEBUG_MESSAGE("Assigned buffer " << m_instance->get_buffer() << " to source " << m_name);
    error = alGetError();
    if (error != AL_NO_ERROR)
    {
        std::cerr << "OpenAL error: "
                  << alutGetErrorString(error) << "\n";
        DEBUG_MESSAGE("here.");
        throw AudioError();
    }
    // get the instance to set the gain, pitch, and location.
    m_instance->update_al();
    // it won't set the looping by itself though:
    alSourcei(m_name, AL_LOOPING, m_instance->get_looping() ? AL_TRUE : AL_FALSE);
        error = alGetError();
    if (error != AL_NO_ERROR)
    {
        std::cerr << "OpenAL error: "
                  << alutGetErrorString(error) << "\n";
        DEBUG_MESSAGE("here.");
        throw AudioError();
    }    
    if (Audio::get_instance().get_paused())
    {
        alSourcePause(m_name);
    }
    else
    {
        alSourcePlay(m_name);
    }
        error = alGetError();
    if (error != AL_NO_ERROR)
    {
        std::cerr << "OpenAL error: "
                  << alutGetErrorString(error) << "\n";
        DEBUG_MESSAGE("here.");
        throw AudioError();
    }
}

}// namespace Engine

