/** @file AITrainer.h
 *  @brief Declare the Engine::AITrainer class. 
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef AI_TRAINER_H
#define AI_TRAINER_H

#include "Scene.h"
#include "Physics/World.h"
#include "GameObjects/Car.h"
#include "InputHandler.h"
#include "InputDeviceAI.h"

#include <libtrack/Track.h>
#include <libtrack/Mesh/BulletMesh.h>

#include <fstream>

namespace Engine
{

class AITrainer
{
public:
    /**  Train AI by racing variants against each other.
     * @param track_filename The filename of the track to use relative to
     where the data directory is.
     */
    AITrainer(std::string track_filename);
    virtual ~AITrainer();
    void hello();
private:
    /// Create cars in their initial positions.
    void reset_cars();
    /// Run the simulation for a while.
    void simulate();
    /// Score the cars and store the order in car_ranks.
    void rank_cars();
    /** Edit the car's neural networks.
     * The top 5 cars are left alone, the next 5 cars have some random
     * variations added, and the rest are reconstructed from parts of the top
     * 5.
     */
    void mutate();
    
    /// Get the score for the requested car. Higher is better.
    btScalar car_score(unsigned int car);
    
    /// number of cars to use.
    unsigned int number_of_cars;
    /// number of game ticks in a match
    unsigned int ticks_simulated;
    /// number of matches to try
    unsigned int max_iterations;
    /// how many matches have been run
    unsigned int iteration;
    
    /// The InputDevices for the computer controlled players.
    std::vector<boost::shared_ptr<Engine::InputDeviceAI> > m_ai_devices;
    
    std::vector<GameObjects::Car *> cars;
    
    std::ifstream track_file;
    std::string theme_filename;
    std::ifstream theme_file;
    Track::Theme theme;
    /// The course to test on
    Track::Track track;
    
    
    Physics::World * world;
    boost::shared_ptr<btRigidBody> track_body;
    boost::shared_ptr<btRigidBody> floor_body;
    boost::shared_ptr<btCollisionShape> track_shape;
    boost::shared_ptr<btCollisionShape> floor_shape;
    
    /** The ranking of the cars.
     * car_ranks[0] is the index of the best performing car.
     */
    std::vector<int> car_ranks;
    /// The scores the cars have in there initial positions.
    std::vector<btScalar> starting_scores;
    
    /// The centre of the starting plane
    btVector3 start_point;
    /// A normalized vector in the direction perpendicular to the starting plane.
    btVector3 start_plane_normal;
    /// The signed distance between the origin and and the starting plane.
    btScalar start_plane_distance;
};

}

#endif // AI_TRAINER_H
