/** @file AITrainer.cpp
 *  @brief Implement the Engine::AITrainer class. 
 *  @author James Legg
 */
/* Copyright © 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#include "AITrainer.h"

#include <libtrack/StartingPosition.h>
#include <iostream>
#include <Debug.h>

/** Read a line from a c++ standard library input stream.
 * @param input stream to read a line from.
 * @return The read line as a c++ standard library string object.
 */
std::string read_line(std::istream & input)
{
    std::string line;
    std::getline(input, line);
    return line;
}

namespace Engine
{

AITrainer::AITrainer(std::string track_filename)
    :   number_of_cars(30)
    ,   ticks_simulated(2000)
    ,   max_iterations(150)
    ,   track_file(track_filename.c_str())
    ,   theme_filename(read_line(track_file))
    ,   theme_file(theme_filename.c_str())
    ,   theme(theme_file)
    ,   track (track_file, theme)
{
    // make the AI graph for the track.
    track.update_ai_mesh();
    btDefaultMotionState motion_state(btTransform(btQuaternion(0, 0, 0, 1),
                                        btVector3(0, 0, 0)));
    track_shape = track.get_collision_shape();
    btRigidBody::btRigidBodyConstructionInfo
                rigid_body_CI(btScalar(0),
                              &motion_state,
                              &(*track_shape),
                              btVector3(0, 0, 0));
    rigid_body_CI.m_restitution = 1.0;
    track_body = boost::shared_ptr<btRigidBody>(new btRigidBody(rigid_body_CI));
    
    // now do the floor shape in the floor world.
    floor_shape = track.get_floor_shape();
    rigid_body_CI.m_collisionShape = &(*floor_shape);
    floor_body = boost::shared_ptr<btRigidBody>(new btRigidBody(rigid_body_CI));
    
    starting_scores.resize(number_of_cars);
    cars.reserve(number_of_cars);
    m_ai_devices.reserve(number_of_cars);
    
    // Add some AI cars
    for (unsigned int ai = 0; ai < number_of_cars; ai++)
    {
        m_ai_devices.push_back(boost::shared_ptr<InputDeviceAI>(new InputDeviceAI()));
    }
    
    // Do the simulation
    for (iteration = 0; iteration < max_iterations; iteration++)
    {
        world = new Physics::World(track);
        world->get_dynamics_world().addRigidBody(&(*track_body));
        world->get_floor_world().addCollisionObject(&(*floor_body));
        reset_cars();
        ///@todo special mutate for the first iteration?
        simulate();
        rank_cars();
        mutate();
        delete world;
    }
    // save the ai net for the best car.
    m_ai_devices[car_ranks[0]]->save();
}

void AITrainer::reset_cars()
{
    cars.clear();
    const Track::Path & path = track.get_path();
    const Track::PathEdge & edge = path.get_edge(path.get_starting_edge());
    start_point = path.find_start_position();
    path.find_start_plane(start_plane_normal, start_plane_distance);
    for (unsigned int ai = 0; ai < number_of_cars; ai++)
    {
        // Find the starting position for this car.
        btTransform initial_transform =
            // guess transformation when starting position cannot be found.
            edge.get_transform(float(ai) / float(number_of_cars));
        // find the starting position track attachment for this car's rank.
        for (std::vector<boost::shared_ptr<Track::TrackAttachment> >::const_iterator it = edge.get_attachments().begin();
             it != edge.get_attachments().end();
             it++)
        {
            const Track::StartingPosition * pos = dynamic_cast<const Track::StartingPosition *>(&(**it));
            if (pos)
            {
                if (pos->get_rank() == ai)
                {
                    initial_transform = pos->get_global_transform();
                }
            }
        }
        // raise the car above the track slightly.
        initial_transform.setOrigin(initial_transform(btVector3(0.0, 0.0, 0.2)));
        cars.push_back(new GameObjects::Car(*world,
                                            initial_transform,
                                            &(*m_ai_devices[ai]),
                                            ai%2,// ai alterenatly uses both cars
                                            start_plane_normal,
                                            start_plane_distance,
                                            start_point));
        starting_scores[ai] = car_score(ai);
    }
}

void AITrainer::simulate()
{
    // simulate some game time
    world->update(ticks_simulated * 10);
}

void AITrainer::rank_cars()
{
    std::multimap<btScalar, unsigned int> scores;
    for (unsigned int i = 0; i < number_of_cars; i++)
    {
        btScalar score = car_score(i) - starting_scores[i];
        // DEBUG_MESSAGE("Car " << i << " scored " << score);
        scores.insert(std::pair<btScalar, unsigned int>(score, i));
    }
    // was insertion sorted, but we want highest score first.
    car_ranks.resize(number_of_cars - 1);
    unsigned int rank = number_of_cars - 1;
    for (std::multimap<btScalar, unsigned int>::iterator it = scores.begin();
         it != scores.end();
         it++)
    {
        car_ranks[rank] = it->second;
        rank--;
    }
    assert(rank == (unsigned int) -1);
    if (!(iteration % 5))
    {
        std::cout << "Iteration " << iteration << " score " << scores.rbegin()->first << '\n';
    }
}

btScalar AITrainer::car_score(unsigned int i)
{
    // score first based on distance traveled around the course.
    btScalar score = cars[i]->get_lap_distance() +
            (cars[i]->get_lap() * track.get_lap_length() + 100.0);
    if (cars[i]->get_finished())
    {
        // bonus meter per tick for finishing all 3 laps before the end of
        // the simulation.
        score += ticks_simulated - cars[i]->get_finish_time();
        DEBUG_MESSAGE("Car " << i << " finished 3 laps in " << cars[i]->get_finish_time() << " ticks.");
    } else if (cars[i]->get_disqualified())
    {
        // Penalty: deduct a meter per tick for falling off before the end
        // of the simulation.
        score -= ticks_simulated - cars[i]->get_finish_time();
        DEBUG_MESSAGE("Car " << i << " was disqualified in " << cars[i]->get_finish_time() << " ticks.");
    }
    return score;
    /**@todo For good competitive AIs, rank is more important than speed.
     * If it knocks all rivals off, it can take as long as it likes to
     * finish.
     */
}

void AITrainer::mutate()
{
    // Leave cars ranked 0-4 alone.
    // cars ranked 5-9 get a little variation.
    for (unsigned int r = 5; r < 10; r++)
    {
        m_ai_devices[car_ranks[r]]->mutate();
    }
    // the rest gets reconstructed from two of the top 5
    for (unsigned int r = 10; r < number_of_cars; r++)
    {
        // pick two different classes to cross
        /*int p1 = rand() / (RAND_MAX / 5);
        int p2 = p1;
        while (p2 == p1)
        {
            p2 = rand() / (RAND_MAX / 5);
        }*/
        
        // cross a unique pair from the top 5
        int p1 = (r-10)%5;
        int p2 = (r-10)/4;
        if (p2 >= p1) p2++;
        assert(number_of_cars == 30);
        m_ai_devices[car_ranks[r]]->cross(*m_ai_devices[car_ranks[p1]],
                                          *m_ai_devices[car_ranks[p2]]);
    }
}

AITrainer::~AITrainer()
{
}

}

