#!BPY
# This Python file uses the following encoding: utf-8

"""
Name: 'Racer theme Exporter'
Blender: 248
Group: 'Export'
Tooltip: 'Export meshes to a racer theme'
"""

##@file python/racer_theme_export.py
#@brief Provide a Blender exporter for track themes.
#@author James Legg

# Copyright © 2009, 2010 James Legg.
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

## @package racer_theme_export
#  Export meshes in a scene to a racer theme.
#
# There are four blender meshes required for each segment. They each need to
# have the same base name in order to be linked, and a tail to specify which
# of the 4 meshes they are.
# - A mesh object name ending in _g is used for graphics.
# - A mesh object name ending in _a is used for AI.
# - A mesh object name ending in _w is used for wall collisions.
# - A mesh object name ending in _f is used for floor physics, so it can be
#   driven on.
# additionally, lower level of detail meshes can be provided for the
# graphics meshes. These should have the base name with _h, _i, _j, used
# in order and consecutively. (_h is a mesh slightly lower in detail
# than _g, _i is lower in detail than _h, etc, until no more levels are
# provided.)
# Each graphics mesh at each level of detail needs a texture, which
# should have the name as the mesh object it relates to followed by .png
#
# The points where edges can be connected are given by the positions of
# curves objects with a name ending in _0 to _9. The same prefix as the other
# meshes must be used to link them.
# The curve data should be shared and used for each different cross section.
# The positions and angles of the curve objects relative to the graphics mesh
# object is used to find where the connections should be.
#
# The positive y direction is treated as forwards, the positive x direction is
# sideways across the track to the right, and the positive z direction is
# vertically up from the track. This is important when you want to use a mesh
# along an edge.
#
# The objects used for an edge will be bent into a curve to fit the edge's
# path. You must not place faces in such a way that holes appear when this
# happens. A normal corner will be OK when the mesh is well subdivided along
# the y direction, but the road surface should also be divided in the x
# direction if the road twists around the y axis.
#
# The physics floor mesh can be more subdivided than the graphical mesh to
# reduce wobbly / slow patches in twisted road.

import Blender
import bpy

## Write the theme in the current scene to a file.
# @param filename The filename to write to.
def write(filename):
    print "Opening file..."
    out = file(filename, "w")
    #sce= bpy.data.scenes.active
    print "Finding data..."
    # check for a text containg the path.
    found = False
    for text in Blender.Text.Get() :
        if text.getName() == "Texture path":
                texture_path = text.asLines()[0]
                found = True
    if not found:
        # no path, guess from the filename.
        texture_path = 'data/' + filename.rpartition('/')[2]
        texture_path_text = Blender.Text.New("Texture path")
        texture_path_text.write(texture_path)
        Blender.Draw.PupMenu("Info|I'm guessing the texture path is '" + texture_path + "'. If this is incorrect, change the path in the text buffer called 'Texture path' and export again.")
    export = {}
    piece_names = [];
    mesh_codes_b = ['_g', '_a', '_w', '_f']
    mesh_codes = set(mesh_codes_b)
    for ob in Blender.Scene.GetCurrent().objects:
        if ob.getType() == 'Mesh' or ob.getType() == 'Curve':
            name = ob.name
            #print "    Adding " + name
            export[name] = ob
            for code in mesh_codes:
                if name[len(name)-2:] == code:
                    #print "        code " + code + ", base " + name[:len(name)-2]
                    piece_names[:0] = [name[:len(name)-2]]
    #only output each piece once.
    piece_names = set(piece_names);
    print "Names to export: "
    print piece_names
    # check valid
    print "Checking data..."
    for piece in piece_names:
        for name in [piece + x for x in mesh_codes]:
            #print "    Trying to find " + name
            if not name in export:
                #moan and return, doesn't exist
                print "    Cannot find " + name + ", aborting."
                Blender.Draw.PupMenu("Error|Expected a mesh object called " + name + ", but could not find it in this scene.")
                return;
    #valid
    print "Writing data..."
    # scheme stuff
    # skybox data
    out.write ("6 ")
    out.write("%s %s %s %s %s %s " % (format_string(texture_path + "/sky.png.+z"),
                                      format_string(texture_path + "/sky.png.-z"),
                                      format_string(texture_path + "/sky.png.-x"),
                                      format_string(texture_path + "/sky.png.+x"),
                                      format_string(texture_path + "/sky.png.-y"),
                                      format_string(texture_path + "/sky.png.+y")))
    out.write('%i ' % (len(piece_names)))
    curve_data_names = {}
    for piece in piece_names:
        # Write the graphics meshes.
        # Find the graphics levels of detail created
        lods = {}
        # most detailed level first
        for name in [piece + '_' + x for x in "ghijklmnopqrstuv"]:
            if name in export:
                lods[name] = export[name]
        # the number of levels of detail we found
        out.write('%i ' % len(lods.keys()))
        # Each graphics mesh level of detail
        for name in [piece + '_' + x for x in "ghijklmnopqrstuv"]:
            if name in export:
                # texture placeholder
                ##@todo get the texture assigned to the mesh, and complain earlier if there is more than one.
                # currently we just take combine the guessed path, the name of the object, and ".png".
                out.write("%s " % format_string(texture_path + '/' + name + ".png"))
                write_mesh(out, lods[name])
        
        #Each required mesh except for graphics. (ai / walls / floor).
        for name in [piece + x for x in ['_a', '_w', '_f']]:
            #print "    Writing " + name
            write_mesh(out, export[name])
        #human readable name
        print piece
        out.write("%s " % format_string(piece))
        # Output the SegmentConnections. First find some:
        # They will be curves with the same base name.
        connections = {}
        for name in [piece + '_' + x for x in "1234567890"]:
            if name in export:
                connections[name] = export[name]
        # the number of connections.
        out.write('%i ' % len(connections.keys()))
        # location offset due to graphics mesh.
        location_offset = export[piece + '_g'].getLocation()
        for con_name in connections:
            connection = connections[con_name]
            # The cross_Section_id. This is an id unique to all curve data names.
            curve_data_name = connection.data.getName()
            if not curve_data_name in curve_data_names:
                curve_data_names[curve_data_name] = len(curve_data_names.keys())
            out.write('%i ' % curve_data_names[curve_data_name])
            # The position. This is offset by the graphics meshes origin.
            position = connection.getLocation('worldspace')
            out.write('%r %r %r ' % (position[0] - location_offset[0], position[1] - location_offset[1], position[2] - location_offset[2]))
            # Write rotation as a quaternion
            euler = connection.getEuler('worldspace').copy()
            # Convert to degrees first
            euler.x *= 57.295779513
            euler.y *= 57.295779513
            euler.z *= 57.295779513
            quaternion = euler.toQuat()
            out.write('%r %r %r %r\t' % (quaternion.x, quaternion.y, quaternion.z, quaternion.w))
    ## @todo UI for SkyParticles
    # For now, manually change the 0 at the end of the file to 1 to
    # get the firey solar wind effect.
    out.write(' 0 ')
    print "Done."
    print "[End Racer theme exporter]"

## Write an indvidual mesh's data to an open file.
# @param out The output file.
# @param ob The object to write about.
def write_mesh(out, ob):
        mesh = ob.getData(mesh=True)
        #bounding box
        #ignored for now.
        #box=ob.boundingBox
        #out.write('%f %f %f\t%f %f %f\n' % (box[0].x, box[0].y, box[0].z, box[7].x, box[7].y, box[5].z))
        #number of vertices
        out.write('%i\n' % (len(mesh.verts)))
        #each vertex
        for vertex in mesh.verts:
            # coordinates
            out.write('%r %r %r\n' % (vertex.co.x, vertex.co.y, vertex.co.z))
        #number of faces - quads count as two
        face_count = 0
        for face in mesh.faces:
            if len(face.verts) == 4:
                face_count += 2
            else:
                face_count += 1
        out.write('%i\n' % (face_count))
        #each face
        for face in mesh.faces:
            #Split quads into two triangles.
            #For triangles, just use the first.
            for vert_index in [0,1,2]:
                write_vert(out, vert_index, face, mesh.faceUV)
            if len(face.verts) == 4:
                for vert_index in [0,2,3]:
                    write_vert(out, vert_index, face, mesh.faceUV)

##Write the output for a particular vertex.
# If the vertex's face does not have texture coordinates, set faceUV to false.
# Texture coordinates will be written as if they were (0,0) in this case.
#@param out the opened file to write to
#@param vert_index an index within the a face
#@param face the face that contains the vertex.
#@param faceUV a boolean value indicating if the face has UV coordinates.
def write_vert(out, vert_index, face, faceUV):
    vert = face.verts[vert_index]
    #normals
    if face.smooth:
            out.write('%r %r %r\t' % (vert.no.x, vert.no.y, vert.no.z))
    else:
            out.write('%r %r %r\t' % (face.no.x, face.no.y, face.no.z))
    #uvs
    if faceUV:
        out.write('%r %r\n' % (face.uv[vert_index].x, 1.0-face.uv[vert_index].y))
    else:
        #daft uv coordinates to get around the non uv mapped faces.
        out.write('0 0\n')
    #vertex index
    out.write('%i\t' % (vert.index))

## Convert a string so it can be loaded by string_from_stream().
# Version 1. Writes 1, the length, and the characters, seperated by spaces.
#@param string string to write.
def format_string(string):
    return ('1 %i %s' % (len(string), string))

print "[Begin Racer theme exporter]"
print "Asking for filename..."
#TODO default to a file in ~/.racer/themes/ so it appears in the editor.
Blender.Window.FileSelector(write, "Export")

