/** @file libtrack/stream_loader.h
 *  @brief Declare some functions to help with loading information from c++
 * stream objects.
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/
#ifndef LIBTRACK_STREAM_LOADER_H_
#define LIBTRACK_STREAM_LOADER_H_

#include <vector>
#include <istream>

#include  "Drawable.h"

#include <boost/shared_ptr.hpp>

#include <LinearMath/btTransform.h>

#include <Debug.h>

/** Fill a std::vector<boost::shared_ptr<T> > with refernce counted pointers to
 * T objects created from a stream object.
 * 
 * If the constructor of T takes the stream output from the objects' operator<<,
 * then Track::write_vector_to_stream and Track::fill_vector_from_stream can be
 * used to save and restore the state of a std::vector of objects.
 * @warning This breaks on strings with separators. Use
 * @code fill_string_vector_from_stream @endcode instead.
 * @tparam T The type of object listed in the file. It must have a constructor
 * that takes a std::istream and pulls from it the information it needs.
 * @param vector The vector to fill.
 * @param stream The stream containing the information about the vector.
 */
template <class T>
void fill_vector_from_stream(std::vector<boost::shared_ptr<T> > & vector,
                             std::istream & stream)
{
    std::size_t number_of_elements;
    stream >> number_of_elements;
    DEBUG_MESSAGE("Reading vector of " << number_of_elements << " starting at " << stream.tellg());
    vector.reserve(number_of_elements);
    DEBUG_MESSAGE("Reading elements");
    for (unsigned int i = 0; i < number_of_elements; i++)
    {
        boost::shared_ptr<T> element(new T(stream));
        DEBUG_MESSAGE("Read element " << i << ", now at " << stream.tellg());
        vector.push_back(element);
    }
    DEBUG_MESSAGE("Finished reading vector");
}

/** Fill a std::vector<boost::shared_ptr<T> > with refernce counted pointers to
 * T objects created from a stream object.
 * 
 * If the constructor of T takes the stream output from the objects' operator<<,
 * then Track::write_vector_to_stream and Track::fill_vector_from_stream can be
 * used to save and restore the state of a std::vector of objects.
 * @tparam T The type of object listed in the file. It must have a constructor
 * that takes a std::istream and pulls from it the information it needs.
 * @param vector The vector to fill.
 * @param stream The stream containing the information about the vector.
 * @tparam S Type of the additional data to pass to the constructor of T objects.
 * @param additional Additional data to pass to the constructor of T objects.
 */
template <class T, class S>
void fill_vector_from_stream(std::vector<boost::shared_ptr<T> > & vector,
                             std::istream & stream,
                             S & additional)
{
    std::size_t number_of_elements;
    stream >> number_of_elements;
    DEBUG_MESSAGE("Reading vector of " << number_of_elements << " starting at " << stream.tellg());
    vector.reserve(number_of_elements);
    DEBUG_MESSAGE("Reading elements");
    for (unsigned int i = 0; i < number_of_elements; i++)
    {
        boost::shared_ptr<T> element(new T(stream, additional));
        DEBUG_MESSAGE("Read element " << i << ", now at " << stream.tellg());
        vector.push_back(element);
    }
    DEBUG_MESSAGE("Finished reading vector");
}

/** Write a std::vector<boost::shared_ptr<T> >'s contents to a stream
 * 
 * If the constructor of T takes the stream output from the objects' operator<<,
 * then Track::write_vector_to_stream and Track::fill_vector_from_stream can be
 * used to save and restore the state of a std::vector of objects.
 * @warning This breaks on strings with separators. Use
 * @code write_string_vector_to_stream @endcode instead.
 * @tparam T The type of object pointed to by shared pointers in the vector.
 * It must have an operator<< that writes information about to a stream.
 * @param vector The vector to get the objects from.
 * @param stream The stream to insert the information.
 */
template <class T>
void write_vector_to_stream(const std::vector<boost::shared_ptr<T> > & vector, std::ostream & stream)
{
    std::size_t number_of_elements = vector.size();
    stream << number_of_elements << "\n";
    for (unsigned int i = 0; i < number_of_elements; i++)
    {
        stream << *(vector[i]) << "\n";
    }
}

/** get an int from a stream. Useful for constructors.
 * Reads an int from a stream, advancing it to the end of the int.
 * @param stream The stream to get the int from.
 * @return first integer found after the stream's position.
 */
inline int int_from_stream(std::istream & stream)
{
    int i;
    stream >> i;
    return i;
}

/** get a btTransform from a stream.
 * @param stream The stream to extract data from. It is advanced to the end of
 * the btTransform.
 * @return The btTransform read.
 */
btTransform transform_from_stream(std::istream & stream);

/** Write a btTransform to a stream.
 * @param stream The stream to write data to.
 * @param transform The btTransform write to the stream.
 * @return The same stream that was passed in.
 */
std::ostream & operator<<(std::ostream & stream, btTransform transform);

/** Write a string to a stream with known whitespace
 * surrounding it.
 * You can get the string back with string_from_stream, no matter what
 * whitespace was placed around the string.
 * The string is allowed to contain any characters.
 * @param stream The stream to write to.
 * @param string the string to record.
 */
void string_to_stream(std::ostream & stream, std::string string);

/** Read a string written with string_to_stream().
 * @param stream a stream either at the beginning of a string written with 
 * string_to_stream(), or with only whitespace before one.
 * @return the string passed to write_string_vector
 */
std::string string_from_stream(std::istream & stream);


std::ostream & operator<<(std::ostream & destination, const btVector3 & vector);
std::istream & operator>>(std::istream & source, btVector3 & vector);

/// Write a btQuaternion to a stream. Useful for btQuaternion too.
std::ostream & operator<<(std::ostream & destination, const btQuadWord & word);
/// Read a btQuadWord from a stream. Useful for btQuaternion too.
std::istream & operator>>(std::istream & source, btQuadWord & word);

#endif /*LIBTRACK_STREAM_LOADER_H_*/
