/** @file libtrack/stream_loader.cpp
 *  @brief Implement some functions to help with loading information from c++
 * stream objects.
 *  @author James Legg
 */
/* Copyright © 2009 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "stream_loader.h"
#include "FormErrors.h"

btTransform transform_from_stream(std::istream & stream)
{
    btTransform result;
    btScalar x,y,z,w;
    // origin
    stream >> x >> y >> z;
    result.setOrigin(btVector3(x,y,z));
    
    // rotation
    stream >> x >> y >> z >> w;
    result.setRotation(btQuaternion(x, y, z, w));
    return result;
}

std::ostream & operator<<(std::ostream & stream, btTransform transform)
{
    btVector3 origin(transform.getOrigin());
    stream << origin.x() << ' '
           << origin.y() << ' '
           << origin.z() << '\t';
    
    btQuaternion rotation(transform.getRotation());
    stream << rotation.x() << ' '
           << rotation.y() << ' '
           << rotation.z() << ' '
           << rotation.w();
    return stream;
}

const unsigned int latest_string_version = 1;

void string_to_stream(std::ostream & stream, std::string string)
{
    stream << latest_string_version << ' '
           << string.length() << ' '
           << string;
}

std::string string_from_stream(std::istream & stream)
{
    unsigned int version;
    // Do we know how to read this string?
    stream >> version;
    if (version == 0) throw CorruptedError();
    if (version > latest_string_version) throw NewVersionError();
    // find the length of the string.
    std::size_t length;
    stream >> length;
    std::string result;
    result.reserve(length);
    // read the space seperator.
    char c;
    stream >> c;
    // get the string 1 character at a time.
    for (unsigned int i = 0; i < length; i++)
    {
        result.push_back(c);
        stream.get(c);
    }
    DEBUG_MESSAGE(result);
    return result;
}

std::ostream & operator<<(std::ostream & destination, const btVector3 & vector)
{
    destination << vector.x() << ' '
                << vector.y() << ' '
                << vector.z();
    return destination;
}

std::istream & operator>>(std::istream & source, btVector3 & vector)
{
    btScalar x, y, z;
    source >> x >> y >> z;
    vector.setX(x);
    vector.setY(y);
    vector.setZ(z);
    return source;
}

std::ostream & operator<<(std::ostream & destination, const btQuadWord & quad)
{
    destination << quad.x()
         << " " << quad.y()
         << " " << quad.z()
         << " " << quad.w();
    return destination;
}

std::istream & operator>>(std::istream & source, btQuadWord & quad)
{
    btScalar x, y, z, w;
    source >> x >> y >> z >> w;
    quad.setX(x);
    quad.setY(y);
    quad.setZ(z);
    quad.setW(w);
    return source;
}
