/** @file libtrack/path/PathVertex.h
 *  @brief Declare the Track::PathVertex class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#ifndef LIBTRACK_PATH_PATH_VERTEX_H
#define LIBTRACK_PATH_PATH_VERTEX_H

#include <ostream>
#include <istream>

#include <LinearMath/btVector3.h>
#include <LinearMath/btQuaternion.h>
#include <BulletCollision/CollisionShapes/btTriangleMesh.h>

#include "../UniqueIdentifier.h"
#include "../Segment.h"
#include "../edit_base/Dragable.h"
#include "../Theme.h"
#include "../edit_base/VertexRotationHandle.h"
#include "../edit_base/SegmentConnectionHandle.h"

#include "../AxisAlignedBoundingBox.h"
#include "../AABBDrawable.h"

namespace Track
{

class Track;

/** Information to store at the vertices of a Path.
 */
class PathVertex
    :   public UniqueIdentifier
    ,   public EditAssist::Dragable
    ,   public AABBDrawable
{
    friend std::ostream & operator<<(std::ostream & destination,
                                     const PathVertex & path_vertex);
    friend AxisAlignedBoundingBox & operator|=(AxisAlignedBoundingBox & box,
                                     const PathVertex & path_vertex);
public:
    /** Failing constuctor.
     * This is added for compatibility with boost's graph library. It should
     * never be called, since we only create graph edges where the vertices
     * already exist.
     */
    PathVertex();
    PathVertex(const Track * track);
    PathVertex(std::istream & source, const Theme & theme, const Track * track);
    virtual ~PathVertex();
    
    /// Get the position of the centre of the vertex.
    const btVector3 & get_position() const;
    
    /** Get a SegmentConnection's position on this vertex.
     * @param index The index of the SegmentConnection in the Segment to look
     * up.
     * @return The position of the SegmentConnection transformed to its
     * position on the vertex.
     */
    btVector3 get_position(std::size_t index) const;
    
    void set_position(const btVector3 & position);
    
    /** Get the tangent of the arcs where they meet the vertex.
     * It is unit length.
     */
    const btVector3 & get_gradient() const;
    
    /** Get the tangent of the arcs where they meet a SegmentConnection.
     * It is unit length.
     * @param index The index the Segment uses for the SegmentConnection.
     */
    btVector3 get_gradient(std::size_t index) const;
    
    /** Get the up vector of the track at the vertex.
     * It is perpendicular to gradient, and unit length.
     */
    const btVector3 & get_up() const;
    
    /** Get the up vector of the track at a SegmentConnection.
     * It is perpendicular to gradient, and unit length.
     * @param index The index the Segment uses for the SegmentConnection.
     */
    btVector3 get_up(std::size_t index) const;
    
    /** Set the angle used to determine the gradient and up vector.
     * The gradient is wherever the rotation takes y axis, and the up vector is
     * wherever the rotation takes the z axis.
     */
    void set_angle(btQuaternion angle);
    
    /** Get the angle used to determine the gradient and up vector of the
     * vertex itself.
     */
    const btQuaternion & get_angle() const;
    
    /** Get the angle used to determine the gradient and up vector at a
     * SegmentConnection.
     * @param index The index the Segment uses for the SegmentConnection.
     */
    btQuaternion get_angle(std::size_t index) const;
    
    std::size_t get_segment_index() const;
    
    void set_segment(std::size_t new_segment_index, const Segment * new_segment);
    
    const Segment * get_segment() const;
    
    // Implement virtual functions from Selectable.
    virtual bool is_here(btVector3 start, btVector3 stop, btScalar radius) const;
    
    /** Create a document delta moving the vertex to a new position.
     * @param new_position The location the vertex should ocupy when the
     * produced delta is applied.
     * @return a shared pointer to a Document::MoveNodeDelta.
     */
    virtual boost::shared_ptr<Document::DocumentDelta> make_delta(btVector3 new_position) const;
    
    /** Return any vertex handles near a line segment.
     * @param start The starting point of the line segment to scan.
     * @param stop The end point of the the line segment to scan
     * @param radius The radius around the line to consider.
     * @return A pointer to a control point that alters this vertex within a
     * cylinder around the line, with the request radius, or 0 if there is no
     * such control point.
     */
    const EditAssist::ControlPoint * get_control_point_here(btVector3 start, btVector3 stop, btScalar radius) const;
    
    /** Draw all ControlPoint objects adjusting this vertex.
     * @see void Track::EditAssist::ControlPoint::draw() const
     */
    void draw_control_points() const;
    
    /// Set the lengths of the rotation handles.
    void set_handle_lengths(btScalar handle_length);
    
    const std::vector<boost::shared_ptr<EditAssist::SegmentConnectionHandle> > & get_connection_handles() const;
    
    void set_track(const Track * track);
    
    /// Add the collision faces of the PathVertex into a shape.
    void add_collision_faces(btTriangleMesh & shape) const;
    /// Add the drivable surface of the PathVertex into a shape.
    void add_floor_faces(btTriangleMesh & shape) const;
    /// Add the surface the AI considers driving on to a mesh.
    void add_ai_faces(MeshFaces & mesh) const;
    
    // from AABBBounded, to make class a not an abstract AABBDrawable.
    virtual AxisAlignedBoundingBox get_bounds() const;
    // from Drawable, to make class a not an abstract AABBDrawable.
    virtual void draw() const;
    // from Drawable, overridden in AABDrawable
    /** Draw an appropriate mesh considering the distance to the camera.
     * Checks if the bounding box is in view, if not no action is taken.
     */
    virtual void conditional_draw(const OcclusionTester &occlusion_tester) const;
protected:
    unsigned int file_version;
    /// The position of the vertex
    btVector3 position;
    /// The segment to use at the vertex
    const Segment * segment;
    
    /// The index of the segment in the Theme
    std::size_t segment_index;
    /// The normalised forward direction from this vertex.
    btVector3 gradient;
    /// The normalised up direction from this vertex.
    btVector3 up;
    /// The rotation from gradient (0, 1, 0) and up (0, 0, 1).
    btQuaternion angle;
    
    /// Rotation of each of the connections.
    std::vector<btQuaternion> connection_angles;
    
    /// control point used to change the up direction in the editor.
    EditAssist::VertexRotationHandle up_handle;
    /// control point used to change the forward direction in the editor.
    EditAssist::VertexRotationHandle forward_handle;
    
    std::vector<boost::shared_ptr<EditAssist::SegmentConnectionHandle> > connection_handles;
    
    const Track * track;
    
    /// The bounds of the segment's transformed graphics mesh.
    AxisAlignedBoundingBox m_bounds;
    /// The transformation of the segment to world space.
    btTransform m_transform;
    
    /// The navigation mesh.
    MeshFaces m_nav_mesh;
    MeshFaces::Graph m_navigation_graph;
    
    /** Set the correct value for m_bounds and m_transform and update the
     * navigation graph.
     */
    void calculate_bounds();
    
    /// Update the properties of the vertices control points.
    void update_handles();
};

std::ostream & operator<<(std::ostream & destination,
                          const PathVertex & path_vertex);

/// Expand an AxisAlignedBoundingBox almost minimally to contain a PathVertex.
AxisAlignedBoundingBox & operator|=(AxisAlignedBoundingBox & box,
                                    const PathVertex & path_vertex);

}


#endif // LIBTRACK_PATH_PATH_VERTEX_H
