/** @file libtrack/path/PathVertex.cpp
 *  @brief Implement the Track::PathVertex class.
 *  @author James Legg
 */
/* Copyright © 2009, 2010 James Legg.
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
*/

#include "PathVertex.h"

#include "../stream_loader.h"
#include "../FormErrors.h"
#include "../document/MoveNodeDelta.h"

namespace Track
{

const unsigned int path_vertex_newest_version = 1;

PathVertex::PathVertex()
    :   file_version(0)
    ,   up_handle(0, EditAssist::VertexRotationHandle::DIR_UP)
    ,   forward_handle(0, EditAssist::VertexRotationHandle::DIR_FORWARD)
{
    DEBUG_MESSAGE("PathVertex should be initalised with a Track*!");
    assert(false);
    throw;
}

PathVertex::PathVertex(const Track * track)
    :   file_version(0)
    ,   position(btVector3(0, 0, 0))
    ,   segment(0)
    ,   segment_index(0)
    ,   gradient(btVector3(1.0, 0.0, 0.0))
    ,   up_handle(get_name(), EditAssist::VertexRotationHandle::DIR_UP)
    ,   forward_handle(get_name(), EditAssist::VertexRotationHandle::DIR_FORWARD)
    ,   track(track)
{
    set_angle(btQuaternion::getIdentity());
    update_handles();
}

PathVertex::PathVertex(std::istream & source, const Theme & theme, const Track * track)
    :   file_version(int_from_stream(source))
    ,   up_handle(get_name(), EditAssist::VertexRotationHandle::DIR_UP)
    ,   forward_handle(get_name(), EditAssist::VertexRotationHandle::DIR_FORWARD)
    ,   track(track)
{
    if (file_version == 0) throw CorruptedError();
    if (file_version > path_vertex_newest_version) throw NewVersionError();
    source >> position >> gradient >> angle;
    std::string segment_name(string_from_stream(source));
    segment_index = theme.get_segment_index(segment_name);
    set_segment(segment_index, &theme.get_segment(segment_index));
    set_angle(angle);
    update_handles();
    calculate_bounds();
}

PathVertex::~PathVertex()
{
    
}

const btVector3 & PathVertex::get_position() const
{
    return position;
}

btVector3 PathVertex::get_position(std::size_t index) const
{    
    return btTransform(angle)(segment->get_connection(index).get_transform().getOrigin()) + position;
}

void PathVertex::set_position(const btVector3 & position_in)
{
    position = position_in;
    update_handles();
    calculate_bounds();
}

void PathVertex::update_handles()
{
    assert(track);
    up_handle.set_centre(position);
    up_handle.update_angle(angle);
    
    forward_handle.set_centre(position);
    forward_handle.update_angle(angle);
    
    if (segment)
    {
        connection_angles.resize(segment->get_number_of_connections());
        for (std::size_t index = 0;
             index < segment->get_number_of_connections();
             index++)
        {
            connection_handles[index]->set_position(get_position(index));
            // find angles of connections.
            btTransform segment_connection_transform(segment->get_connection(index).get_transform());
            connection_angles[index] = (btTransform(angle) * segment_connection_transform).getRotation();
        }
    }
}

const btVector3 & PathVertex::get_gradient() const
{
    return gradient;
}

btVector3 PathVertex::get_gradient(std::size_t index) const
{
    return btTransform(get_angle(index))
        (btVector3(0.0, 1.0, 0.0));
}

const btVector3 & PathVertex::get_up() const
{
    return up;
}

btVector3 PathVertex::get_up(std::size_t index) const
{
    return btTransform(get_angle(index))
        (btVector3(0.0, 0.0, 1.0));
}

void PathVertex::set_angle(btQuaternion angle_in)
{
    angle = angle_in;
    const btTransform transform(angle_in);
    gradient = transform(btVector3(0, 1.0, 0.0));
    up = transform(btVector3(0, 0.0, 1.0));
    
    update_handles();
    calculate_bounds();
}

const btQuaternion & PathVertex::get_angle() const
{
    return angle;
}

btQuaternion PathVertex::get_angle(std::size_t index) const
{
    return connection_angles[index];
}

std::size_t PathVertex::get_segment_index() const
{
    return segment_index;
}

void PathVertex::set_segment(std::size_t new_segment_index, const Segment * new_segment)
{
    assert(new_segment);
    segment_index = new_segment_index;
    segment = new_segment;
    
    connection_handles.clear();
    connection_handles.reserve(segment->get_number_of_connections());
    for (std::size_t index = 0;
         index < segment->get_number_of_connections();
         index++)
    {
        connection_handles.push_back(boost::shared_ptr<EditAssist::SegmentConnectionHandle>(
            new EditAssist::SegmentConnectionHandle(
                *track, get_name(), index, &(segment->get_connection(index)))
            )
        );
        connection_handles.back()->set_position(get_position(index));
    }
    calculate_bounds();
}

const Segment * PathVertex::get_segment() const
{
    return segment;
}

bool PathVertex::is_here(btVector3 start, btVector3 stop, btScalar radius) const
{
    btVector3 line = stop - start;
    btVector3 to_start = position - start;
    btScalar u = to_start.dot(line) / line.length2();
    if (u < 0.0 || u > 1.0)
    {
        //Nearest to one of the end vertices. Not valid selection.
        // Is either behind the viewer or depth clipped, either way
        // not on screen so can't be selected.
        return false;
    }
    btScalar distance2 = (u * line).distance2(to_start);
    if (distance2 < radius * radius)
    {
        return true;
    }
    return false;
}

boost::shared_ptr<Document::DocumentDelta> PathVertex::make_delta(btVector3 new_position) const
{
    return boost::shared_ptr<Document::DocumentDelta>
    (
        new Document::MoveNodeDelta
        (
            Document::NodePositionFinder(get_name())
            , new_position
        )
    );
}

const EditAssist::ControlPoint * PathVertex::get_control_point_here(btVector3 start, btVector3 stop, btScalar radius) const
{
    if (up_handle.is_here(start, stop, radius))
    {
        return &up_handle;
    }
    else if (forward_handle.is_here(start, stop, radius))
    {
        return &forward_handle;
    }
    // check segment connection handles
    for (std::vector<boost::shared_ptr<EditAssist::SegmentConnectionHandle> >::const_iterator it = connection_handles.begin();
         it != connection_handles.end();
         it++
        )
    {
        if ((**it).is_here(start, stop, radius))
        return &(**it);
    }
    
    // There is no appropriate control point.
    return 0;
}

void PathVertex::draw_control_points() const
{
    up_handle.draw();
    forward_handle.draw();
    for (std::vector<boost::shared_ptr<EditAssist::SegmentConnectionHandle> >::const_iterator it = connection_handles.begin();
         it != connection_handles.end();
         it++
        )
    {
        (**it).draw();
    }
}

void PathVertex::set_handle_lengths(btScalar handle_length)
{
    up_handle.set_length(handle_length);
    forward_handle.set_length(handle_length);
    update_handles();
}

const std::vector<boost::shared_ptr<EditAssist::SegmentConnectionHandle> > &
    PathVertex::get_connection_handles() const
{
    return connection_handles;
}

void PathVertex::set_track(const Track * track_in)
{
    track = track_in;
}

void PathVertex::add_collision_faces(btTriangleMesh & shape) const
{
    btTransform transform(angle, position);
    segment->get_floor_mesh().get_distorted_faces(transform).add_faces(shape);
    segment->get_wall_mesh().get_distorted_faces(transform).add_faces(shape);
}

void PathVertex::add_floor_faces(btTriangleMesh & shape) const
{
    btTransform transform(angle, position);
    segment->get_floor_mesh().get_distorted_faces(transform).add_faces(shape);
}

void PathVertex::add_ai_faces(MeshFaces & mesh) const
{
    btTransform transform(angle, position);
    MeshFaces faces = segment->get_ai_mesh().get_distorted_faces(transform);
    faces.set_source(false, get_name());
    mesh |= faces;
}

AxisAlignedBoundingBox PathVertex::get_bounds() const
{
    return m_bounds;
}

void PathVertex::calculate_bounds()
{
    m_transform = btTransform(angle, position);
    if(segment)
    {
        // Finding an accurate graphics mesh bounding box improves the
        // level of detail selection:
        // get the bounding box of the transformed graphics mesh.
        m_bounds = segment->get_graphics_mesh().get_distorted_faces(m_transform).get_bounds();
        // update data for other meshes:
        m_nav_mesh = segment->get_ai_mesh().get_distorted_faces(m_transform);
        m_nav_mesh.set_source(false, get_name());
        m_navigation_graph = m_nav_mesh.get_connectivity();
    }
}

void PathVertex::conditional_draw(const OcclusionTester &occlusion_tester) const
{
    assert(segment);
    // test if it is at least partially on the screen
    if (occlusion_tester(m_bounds) != OcclusionTester::VS_OUT)
    {
        glPushMatrix();
            btScalar mat[16];
            mat[15] = 1.0;
            m_transform.getOpenGLMatrix(mat);
            glMultMatrixf(mat);
            // draw a suitable mesh for this distance.
            segment->get_graphics_mesh().get_faces().draw(
                // square of the distance between this and the camera
                occlusion_tester.camera_position.distance2(position)
                );
        glPopMatrix();
    }
}

void PathVertex::draw() const
{
    assert(segment);
    glPushMatrix();
        btScalar mat[16];
        mat[15] = 1.0;
        m_transform.getOpenGLMatrix(mat);
        glMultMatrixf(mat);
        segment->get_graphics_mesh().get_faces().draw();
    glPopMatrix();
}

std::ostream & operator<<(std::ostream & destination, const PathVertex & path_vertex)
{
    destination << path_vertex_newest_version << ' '
                << path_vertex.position << ' '
                << path_vertex.gradient << ' '
                << path_vertex.angle << ' ';
    string_to_stream(destination, path_vertex.segment->get_name());
    return destination;
}

AxisAlignedBoundingBox & operator|=(AxisAlignedBoundingBox & box,
                                    const PathVertex & path_vertex)
{
    box |= path_vertex.segment->get_graphics_mesh().get_faces().get_bounds().transform(btTransform(path_vertex.get_angle(), path_vertex.get_position()));
    box |= path_vertex.up_handle.get_position();
    box |= path_vertex.forward_handle.get_position();
    return box;
}

}
